#pragma once

#include "common.h"
#include "config.h"
#include "logger.h"

#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/deprecated/async_impl/async_impl.h>

class TYandexDiskClient : public IDocumentStorage {
public:
    class TBinaryFile : public IDocumentStorageFile {
    private:
        class TDiskCallbackWrapper;
    public:
        TBinaryFile(const TString& name, const NJson::TJsonValue& data, const TYandexDiskClient& owner)
            : Name(name)
            , Json(data)
            , Owner(owner)
        {}

        virtual bool GetData(TString& data, TMessagesCollector& errors) const override {
            return GetDataByLink(Json["file"].GetString(), data, errors);
        }

        virtual void GetDataAsync(IDocumentStorageFile::ICallback::TPtr callback) const override {
            TBinaryFile::GetDataByLink(Json["file"].GetString(), callback, Owner);
        }

        virtual const TString& GetName() const override {
            return Name;
        }

    private:
        bool GetDataByLink(const TString& link, TString& data, TMessagesCollector& errors) const;
        static void GetDataByLink(const TString& link, IDocumentStorageFile::ICallback::TPtr callback, const TYandexDiskClient& owner);

    private:
        TString Name;
        NJson::TJsonValue Json;
        const TYandexDiskClient& Owner;
    };

public:
    TYandexDiskClient(const TYandexDiskClientConfig& config)
        : Config(config)
        , Impl(Config.GetClientConfig(), "yandex_disk_api")
        , DownloadImpl(Config.GetDownloadClientConfig(), Config.GetDownloadClientConfig().GetHost() + ":" + ToString(Config.GetDownloadClientConfig().GetPort()))
        , Logger("yandex_disk_api")
    {
    }

    bool GetDownloadPath(const TString& path, TString& downloadPath, TMessagesCollector& errors) const;
    virtual bool GetFiles(const TString& path, TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const override;
    bool GetFiles(TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const;
    bool GetFilesRecursive(const TString& startPath, const TString& path, TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const;
    void CheckAndRegisterSource(const TString& name, const TString& host, ui16 port) const;

    const TYandexDiskClientConfig& GetConfig() const {
        return Config;
    }

    const TAsyncApiImpl& GetDownloadImpl() const {
        return DownloadImpl;
    }

private:
    NNeh::THttpRequest CreateDownloadRequest(const TString& path) const;
    NNeh::THttpRequest CreateGetFilesRequest() const;
    NNeh::THttpRequest CreateGetFilesRequest(const TString& path) const;

    NNeh::THttpRequest CreateCommonRequest(const TString& method = "GET", const TString& postData = TString()) const;

    bool AddItem(const TString& startPath, const NJson::TJsonValue& item, TVector<IDocumentStorageFile::TPtr>& fileList, TMessagesCollector& errors) const;
    bool SendRequest(EYandexDiskOperationType operationType, const NNeh::THttpRequest& request, NJson::TJsonValue& reply, TMessagesCollector& errors) const;

private:
    TYandexDiskClientConfig Config;
    TAsyncApiImpl Impl;
    mutable TAsyncApiImpl DownloadImpl;
    TMutex DownloadAgentGuard;
    TYandexDiskLogger Logger;
};
