#include "client.h"

#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/messages_collector.h>

#include <library/cpp/json/json_reader.h>

#include <util/string/builder.h>


namespace NDrive {

    TElementFinesClient::TElementFinesClient(const TFinesApiConfig& config)
        : Config(config)
        , Client(new NNeh::THttpClient(Config.GetUri(), Config.GetRequestConfig()))
    {
    }

    const TFinesApiConfig& TElementFinesClient::GetConfig() const {
        return Config;
    }

    template <class T>
    class TApplier {
    public:
        TApplier(const TString& fieldName = "")
            : FieldName(fieldName)
        {
        }

        NThreading::TFuture<TVector<T>> operator()(const NThreading::TFuture<NUtil::THttpReply>& r) const {
            const auto& report = r.GetValue();
            TUnistatSignalsCache::SignalAdd("element-api-codes", ToString(report.Code()), 1);
            if (!report.HasReply()) {
                ythrow yexception() << "No reply: " << report.ErrorMessage();
            }
            NJson::TJsonValue rawResult;
            if (!report.IsSuccessReply() || !NJson::ReadJsonFastTree(report.Content(), &rawResult)) {
                ythrow yexception() << "Error getting objects: " << report.GetDebugReply();
            }
            TVector<T> objects;
            if (FieldName) {
                if (!NJson::ParseField(rawResult, FieldName, objects, /* required = */ true)) {
                    ythrow yexception() << "Fail to parse reply: " << rawResult.GetStringRobust();
                }
            } else {
                if (!NJson::ParseField(rawResult, objects, /* required = */ true)) {
                    ythrow yexception() << "Fail to parse reply: " << rawResult.GetStringRobust();
                }
            }
            {
                bool hasError = false;
                if (!NJson::ParseField(rawResult["error"], hasError, /* required = */ false)) {
                    ythrow yexception() << "Fail to parse reply error: " << rawResult.GetStringRobust();
                }
                if (hasError) {
                    TString message;
                    if (!NJson::ParseField(rawResult, "errorMessage", message, /* required = */ false)) {
                        ythrow yexception() << "Fail to parse reply error message: " << rawResult.GetStringRobust();
                    }
                    ythrow yexception() << "Failed with error from element: " << message;
                }
            }
            return NThreading::MakeFuture(objects);
        }

    private:
        TString FieldName;
    };


    NThreading::TFuture<TVector<TElementFine>> TElementFinesClient::GetFines(const TFinesRequestParams& context) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetFinesUriPath());
        request.SetContentType("application/json");
        {
            NJson::TJsonValue payload;
            payload["limit"] = context.Limit;
            payload["offset"] = context.Offset;
            if (context.Since) {
                payload["DateFrom"] = context.Since.FormatLocalTime("%Y-%m-%d");
            }
            if (context.LastUpdate) {
                payload["LastRenewalDate"] = context.LastUpdate.FormatLocalTime("%Y-%m-%d");
            }
            payload["LoadPhotos"] = false;
            payload["LoadDetail"] = false;
            request.SetPostData(payload);
        }
        Config.Authorize(request);
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TElementFine>("Fines"));
    }

    NThreading::TFuture<TVector<TElementFineAttachment>> TElementFinesClient::GetViolationPhotos(const TString& rulingNumber) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetPhotosUriPath());
        request.SetContentType("application/json");
        {
            NJson::TJsonValue payload;
            payload["Number"] = rulingNumber;
            request.SetPostData(payload);
        }
        Config.Authorize(request);
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TElementFineAttachment>("OffensePhotos"));
    }

    NThreading::TFuture<TVector<TElementCarData>> TElementFinesClient::GetCarsData() const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetCarUriPath() + "/Get");
        Config.Authorize(request);
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TElementCarData>());
    }

    NThreading::TFuture<TVector<TElementCarData>> TElementFinesClient::AddCars(const TVector<TElementCarData>& cars) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetCarUriPath() + "/Add");
        request.SetPostData(NJson::ToJson(cars));
        Config.Authorize(request);
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TElementCarData>());
    }

    NThreading::TFuture<TVector<TElementCarData>> TElementFinesClient::ChangeCarsActivity(const TVector<TElementCarData>& cars) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetCarUriPath() + "/ChangeActivity");
        request.SetPostData(NJson::ToJson(cars));
        Config.Authorize(request);
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TElementCarData>());
    }
}
