#include "entities.h"

#include <rtline/library/json/parse.h>

#include <drive/library/cpp/raw_text/datetime.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/string/subst.h>


template <>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrive::TElementFineAttachment& result) {
    if (!json.IsString()) {
        return false;
    }
    try {
        TString data = json.GetString();
        SubstGlobal(data, "\r\n", "");
        result.SetData(Base64Decode(data));
    } catch (const std::exception& e) {
        return false;
    }
    return true;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrive::TElementFine& result) {
    TString timeValue;
    if (!NJson::ParseField(json, "OffenseDate", result.MutableViolationTimestamp(), /* required = */ true)
        || !NJson::ParseField(json, "OffenseTime", timeValue, /* required = */ false))
    {
        return false;
    }
    if (timeValue) {
        if (SubstGlobal(timeValue, "0001-01-01", result.GetViolationTimestamp().ToString().substr(0, 10)) != 1
            || !TInstant::TryParseIso8601(timeValue, result.MutableViolationTimestamp())
            || !NUtil::ApplyTimeZone(result.GetViolationTimestamp(), "Europe/Moscow", result.MutableViolationTimestamp()))
        {
            return false;
        }
    }
    if (!NJson::ParseField(json["Car"], "CarSTS", result.MutableCarSTS(), /* required = */ false)) {
        return false;
    }
    if (!NJson::ParseField(json["Car"], "CarNumber", result.MutableCarNumber(), /* required = */ false)) {
        return false;
    }
    if (!NJson::ParseField(json["Division"], "Name", result.MutableDepartment(), /* required = */ false)) {
        return false;
    }
    if (json["Division"].IsMap()) {
        const auto& division = json["Division"];
        for (const auto& field : GetEnumAllValues<NDrive::TElementFine::EPamentFields>()) {
            if (auto value = division[ToString(field)].GetString()) {
                result.MutablePaymentData()[ToString(field)] = value;
            }
        }
    }
    ui32 photoCount = 0;
    if (!NJson::ParseField(json, "NumberOfPhotos", photoCount, /* required = */ false)) {
        return false;
    }
    result.SetPhotoFixation(photoCount > 0);
    if (!NJson::ParseField(json, "DiscountDate", result.MutableDiscountDate(), /* required = */ false)) {
        return false;
    }
    return
           NJson::ParseField(json, "Number", result.MutableProtocolNumber(), /* required = */ true)
        && NJson::ParseField(json, "Date", result.MutableProtocolDate(), /* required = */ true)
        && NJson::ParseField(json, "Sum", result.MutableSum(), /* required = */ true)
        && NJson::ParseField(json, "PartPayment", result.MutablePartPayment(), /* required = */ false)
        && NJson::ParseField(json, "Paid", result.MutablePaid(), /* required = */ true)
        && NJson::ParseField(json, "PaidByCompany", result.MutablePaidByCompany(), /* required = */ false)
        && NJson::ParseField(json, "Article", result.MutableArticle(), /* required = */ false)
        && NJson::ParseField(json, "ArticleCode", result.MutableArticleCode(), /* required = */ false)
        && NJson::ParseField(json, "PossibleDiscount", result.MutablePossibleDiscount(), /* required = */ true)
        && NJson::ParseField(json, "OffenseAddress", result.MutableViolationPlace(), /* required = */ false)
        && NJson::ParseField(json, "Status", result.MutableStatus(), /* required = */ false);
    ;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrive::TElementCarData& result) {
    if (!NJson::ParseField(json, "Department", NJson::Stringify(result.MutableDepartment()), /* required = */ false)) {
        WARNING_LOG << "Fail to parse car department " << json.GetStringRobust() << Endl;
    }
    return
           NJson::ParseField(json, "Number", result.MutableNumber(), /* required = */ true)
        && NJson::ParseField(json, "Activity", result.MutableActivity(), /* required = */ false)
        && NJson::ParseField(json, "Model", result.MutableModel(), /* required = */ false)
        && NJson::ParseField(json, "VIN", result.MutableVin(), /* required = */ true)
        && NJson::ParseField(json, "STSNumber", result.MutableStsNumber(), /* required = */ true)
        && NJson::ParseField(json, "STSSeries", result.MutableStsSeries(), /* required = */ false);
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrive::TElementCarData& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "Number", object.GetNumber());
    NJson::InsertField(result, "STSNumber", object.GetStsNumber());
    NJson::InsertField(result, "STSSeries", object.GetStsSeries());
    NJson::InsertNonNull(result, "Model", object.GetModel());
    NJson::InsertNonNull(result, "VIN", object.GetVin());
    NJson::InsertNonNull(result, "Activity", object.OptionalActivity());
    if (object.GetDepartment() != NDrive::EDepartment::Unknown) {
        NJson::InsertField(result, "Department", NJson::Stringify(object.GetDepartment()));
    }
    return result;
}
