#include "exp_calculator.h"

#include <util/string/strip.h>
#include <util/string/subst.h>
#include <util/generic/map.h>

THolder<TExpressionCalculator::IElement> TExpressionCalculator::IElement::Construct(TStringBuf original) {
   TStringBuf leftStr = original.Before('+');
   if (leftStr.size() != original.size()) {
       return THolder(new TOperation(EOperation::Add, Construct(leftStr), Construct(original.After('+'))));
   }
   leftStr = original.Before('-');
   if (leftStr.size() != original.size()) {
       return THolder(new TOperation(EOperation::Sub, Construct(leftStr), Construct(original.After('-'))));
   }
   leftStr = original.Before('*');
   if (leftStr.size() != original.size()) {
       return THolder(new TOperation(EOperation::Mul, Construct(leftStr), Construct(original.After('*'))));
   }
   leftStr = original.Before('/');
   if (leftStr.size() != original.size()) {
       return THolder(new TOperation(EOperation::Div, Construct(leftStr), Construct(original.After('/'))));
   }
   double value = 0;
   if (TryFromString(StripString(original), value)) {
       return THolder(new TConst(value));
   }
   return nullptr;
}

bool TExpressionCalculator::TOperation::Calc(double& result) const {
    double l = 0, r = 0;
    if (Left && Left->Calc(l) && Right && Right->Calc(r)) {
        switch (Operation) {
        case EOperation::Add:
            result = l + r;
            break;
        case EOperation::Sub:
            result = l - r;
            break;
        case EOperation::Mul:
            result = l * r;
            break;
        case EOperation::Div:
            if (fabs(r) < 0.0000001) {
                return false;
            }
            result = l / r;
            break;
        default:
            return false;
        }
        return true;
    }
    return false;
}

TExpected<double, TString> TExpressionCalculator::Calc(const TVector<double>& values) const {
    TString newExp = RawExpression;
    for (size_t i = 0; i < values.size(); ++i) {
        SubstGlobal(newExp, "(" + ::ToString(i + 1) + ")", ::ToString(values[i]));
    }
    return Calc(newExp);
}

TExpected<double, TString> TExpressionCalculator::Calc(const TMap<TString, double>& values) const {
    TString newExp = RawExpression;
    for (const auto& [key, value] : values) {
        SubstGlobal(newExp, key, ::ToString(value));
    }
    return Calc(newExp);
}

TExpected<double, TString> TExpressionCalculator::Calc(const TString& newExp) const {
    auto expression = IElement::Construct(TStringBuf(newExp));
    if (!expression) {
        return MakeUnexpected<TString>("cannot construst expression from: " + newExp);
    }
    double result = 0;
    if (!expression->Calc(result)) {
        return MakeUnexpected<TString>("cannot calc expression: " + newExp);
    }
    return result;
}

