#include <rtline/util/queue.h>

#include "exp_calculator.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/output.h>

Y_UNIT_TEST_SUITE(TExpressionCalculatorSuite) {
    template <class TValues>
    bool Check(const TExpressionCalculator& exp, const TValues& values, double expectedResult) {
        auto result = exp.Calc(values);
        if (!result) {
            ERROR_LOG << result.GetError() << Endl;
            return false;
        }
        return std::fabs(*result - expectedResult) < 0.0001;
    }

    Y_UNIT_TEST(Const) {
        {
            TExpressionCalculator exp("(1)");
            UNIT_ASSERT(Check(exp, TVector<double>({1}), 1));
            UNIT_ASSERT(Check(exp, TVector<double>({0.5}), 0.5));
            UNIT_ASSERT(!Check(exp, TVector<double>({}), 0));
        }
        {
            TExpressionCalculator exp("beta");
            UNIT_ASSERT(Check(exp, TMap<TString, double>({ {"beta", 1} }), 1));
            UNIT_ASSERT(Check(exp, TMap<TString, double>({ {"beta", 0.5} }), 0.5));
            UNIT_ASSERT(!Check(exp, TMap<TString, double>({ {"alpha", 1} }), 1));
            UNIT_ASSERT(!Check(exp, TMap<TString, double>({}), 0));
        }
        return;
    }

    Y_UNIT_TEST(Operation) {
        {
            TExpressionCalculator exp("alpha + beta * gamma");
            UNIT_ASSERT(Check(exp, TMap<TString, double>({ {"alpha", 1}, {"beta", 2}, {"gamma", 2} }), 5));
        }
        {
            TExpressionCalculator exp("0.5 * alpha + beta * 0.5 * gamma");
            UNIT_ASSERT(Check(exp, TMap<TString, double>({ {"alpha", 1}, {"beta", 2}, {"gamma", 2} }), 2.5));
        }
        {
            TExpressionCalculator exp("alpha + beta * gamma");
            UNIT_ASSERT(!Check(exp, TMap<TString, double>({ {"alpha", 1}, {"beta", 2} }), 5));
        }
        {
            TExpressionCalculator exp("alpha + beta % gamma");
            UNIT_ASSERT(!Check(exp, TMap<TString, double>({ {"alpha", 1}, {"beta", 2}, {"gamma", 2} }), 5));
        }
        return;
    }
}
