#include "client.h"

#include <drive/library/cpp/external/proto/device.pb.h>

#include <library/cpp/string_utils/base64/base64.h>

NThreading::TFuture<NDrive::TExternalDeviceSnapshots> NDrive::TExternalDeviceClient::GetOperator(const TString& name) const {
    return GetSnapshots("Operator", name);
}

NThreading::TFuture<NDrive::TExternalDeviceSnapshots> NDrive::TExternalDeviceClient::GetRegion(const TString& name) const {
    return GetSnapshots("Region", name);
}

NThreading::TFuture<NDrive::TExternalDeviceSnapshots> NDrive::TExternalDeviceClient::GetSnapshots(const TString& key, const TString& value) const {
    NRTLine::TQuery query;
    query.SetReqClass("EXTDEV");
    query.AddExtraParam("key_name", key);
    query.SetNumDoc(Max<ui32>());
    query.SetText(value);
    NThreading::TFuture<NRTLine::TSearchReply> reply = Client.SendAsyncQueryF(query);
    NThreading::TFuture<NDrive::TExternalDeviceSnapshots> result = reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        const NRTLine::TSearchReply& reply = r.GetValue();
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());

        NDrive::TExternalDeviceSnapshots result;
        for (auto&& grouping : reply.GetReport().GetGrouping()) {
            for (auto&& group : grouping.GetGroup()) {
                for (auto&& document : group.GetDocument()) {
                    const TString& url = document.GetUrl();
                    if (!url.StartsWith("snapshot:external")) {
                        continue;
                    }
                    try {
                        NDrive::TExternalDeviceSnapshot snapshot;
                        for (auto&& attribute : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                            const TString& key = attribute.GetKey();
                            const TString& value = attribute.GetValue();
                            if (key == "Data") {
                                TString decoded = Base64Decode(value);
                                NDrive::NProto::TExternalDeviceSnapshot proto;
                                Y_ENSURE(proto.ParseFromString(decoded));
                                Y_ENSURE(snapshot.Deserialize(proto));
                                continue;
                            }
                            if (snapshot.Operator.empty() && key == "Operator") {
                                snapshot.Operator = value;
                                continue;
                            }
                            if (snapshot.Region.empty() && key == "Region") {
                                snapshot.Region = value;
                                continue;
                            }
                            if (snapshot.Timestamp == TInstant::Zero() && key == "Timestamp") {
                                snapshot.Timestamp = TInstant::Seconds(FromString<ui64>(value));
                                continue;
                            }
                        }
                        result.push_back(std::move(snapshot));
                    } catch (const std::exception& e) {
                        ERROR_LOG << "cannot read " << url << ": " << FormatExc(e) << Endl;
                    }
                }
            }
        }
        return result;
    });
    return result;
}
