#include "manager.h"

#include <drive/library/cpp/threading/future.h>

#include <rtline/util/algorithm/container.h>

NDrive::TExternalDeviceManager::TExternalDeviceManager(const NRTLine::TNehSearchClient& client, const TOptions& options)
    : IAutoActualization("external_device_manager", TDuration::Seconds(5))
    , Client(client)
    , Options(options)
{
    Y_ENSURE_BT(Start());
}

NDrive::TExternalDeviceManager::~TExternalDeviceManager() {
    if (IsActive() && !Stop()) {
        ERROR_LOG << "cannot stop ExternalDeviceManager" << Endl;
    }
}

NDrive::TExternalDeviceManager::TDevices NDrive::TExternalDeviceManager::GetDevices(const TGeoRect& rectangle, TStringBuf group, TInstant actuality) const {
    const TSet<TString>* operators = Options.OperatorGroups.FindPtr(group);
    TDevices result;
    TSnapshots snapshots = GetSnapshots();
    for (auto&& snapshot : snapshots) {
        if (!snapshot) {
            continue;
        }
        if (snapshot->GetTimestamp() < actuality) {
            continue;
        }
        auto devices = snapshot->GetDevices(rectangle);
        for (auto&& device : devices) {
            if (!device) {
                continue;
            }
            const TString& op = snapshot->GetOperator();
            if (operators && !operators->contains(op)) {
                continue;
            }
            result.emplace_back(*device, op, snapshot->GetTimestamp());
        }
    }
    return result;
}

NDrive::TExternalDeviceManager::TDevices NDrive::TExternalDeviceManager::GetDevices(const TGeoRect& rectangle, TStringBuf group, TDuration freshness) const {
    return GetDevices(rectangle, group, Now() - freshness);
}

NDrive::TExternalDeviceManager::TSnapshots NDrive::TExternalDeviceManager::GetSnapshots() const {
    TReadGuard guard(Lock);
    return MakeVector(NContainer::Values(Snapshots));
}

bool NDrive::TExternalDeviceManager::GetStartFailIsProblem() const {
    return false;
}

bool NDrive::TExternalDeviceManager::Refresh() {
    TVector<NThreading::TFuture<TExternalDeviceSnapshots>> futures;
    for (auto&& region : Options.Regions) {
        futures.push_back(Client.GetRegion(region));
    }
    bool result = true;
    for (auto&& i : futures) {
        if (!i.Wait(GetPeriod())) {
            ERROR_LOG << "Wait error" << Endl;
            result = false;
            continue;
        }
        if (i.HasException()) {
            ERROR_LOG << "An exception occurred in GetRegion: " << NThreading::GetExceptionMessage(i) << Endl;
            result = false;
            continue;
        }
        const auto& snapshots = i.GetValue();
        for (auto&& snapshot : snapshots) {
            {
                TReadGuard guard(Lock);
                auto p = Snapshots.find(snapshot);
                if (p != Snapshots.end() && p->second->GetTimestamp() >= snapshot.Timestamp) {
                    INFO_LOG << "skip " << snapshot.Operator << '-' << snapshot.Region << ' ' << snapshot.Timestamp << Endl;
                    continue;
                }
            }
            auto deviceMap = MakeAtomicShared<TExternalDeviceMap>(snapshot);
            {
                TWriteGuard guard(Lock);
                Snapshots[snapshot] = deviceMap;
            }
            NOTICE_LOG << "updated " << snapshot.Operator << '-' << snapshot.Region << ' ' << snapshot.Timestamp << Endl;
        }
    }
    return result;
}
