#include "client.h"

#include <util/string/builder.h>
#include <util/stream/file.h>

namespace {
    class TTvmLogger : public NTvmAuth::ILogger {
        void Log(int lvl, const TString& msg) override {
            TEMPLATE_LOG(static_cast<ELogPriority>(lvl)) << msg << Endl;
        }
    };

    NRTLine::TNehSearchClient CreateSearchClient(
        const NDrive::TGeocoder::TOptions& options,
        TMaybe<NDrive::TTvmAuth> tvmAuth,
        TAtomicSharedPtr<TAsyncDelivery> ad
    ) {
        NRTLine::TNehSearchClient::TEndpoint endpoint(options.Host, options.Port);
        endpoint.SetPath(options.Path);

        NSimpleMeta::TConfig config;
        config.SetMaxAttempts(1);
        config.SetGlobalTimeout(options.Timeout);

        return { endpoint, config, tvmAuth, "", ad };
    }
}

void NDrive::TGeocoder::TOptions::ToString(IOutputStream& os) const {
    os << "Host: " << Host << Endl;
    os << "Port: " << Port << Endl;
    os << "Path: " << Path << Endl;
    os << "Origin: " << Origin << Endl;
    os << "RevMode: " << RevMode << Endl;
    os << "Timeout: " << ::ToString(Timeout) << Endl;
    os << "TvmDestination: " << TvmDestination << Endl;
    os << "TvmSource: " << TvmSource << Endl;
}

void NDrive::TGeocoder::TOptions::Init(const TYandexConfig::Section* section) {
    Host = section->GetDirectives().Value("Host", Host);
    Port = section->GetDirectives().Value("Port", Port);
    Path = section->GetDirectives().Value("Path", Path);
    Origin = section->GetDirectives().Value("Origin", Origin);
    RevMode = section->GetDirectives().Value("RevMode", RevMode);
    Timeout = section->GetDirectives().Value("Timeout", Timeout);
    TvmDestination = section->GetDirectives().Value("TvmDestination", TvmDestination);
    TvmSource = section->GetDirectives().Value("TvmSource", TvmSource);
    TvmToken = section->GetDirectives().Value<TString>("TvmToken", TvmToken);
    TString tokenPath;
    tokenPath = section->GetDirectives().Value<TString>("TokenPath", tokenPath);
    if (!TvmToken && tokenPath) {
        TvmToken = Strip(TFileInput(tokenPath).ReadAll());
    }
}

NDrive::TGeocoder::TGeocoder(const TOptions& options, TMaybe<NDrive::TTvmAuth> tvmAuth /*= Nothing()*/, TAtomicSharedPtr<TAsyncDelivery> ad /*= nullptr*/)
    : Options(options)
    , Client(CreateSearchClient(options, tvmAuth, ad))
{
}

NDrive::TGeocoder::~TGeocoder() {
}

NThreading::TFuture<NDrive::TGeocoder::TResponse> NDrive::TGeocoder::Decode(const TGeoCoord& coordinate, ELanguage language /*= LANG_RUS*/) const {
    NRTLine::TQuery query;
    query.AddExtraParam("ll", TStringBuilder() << coordinate.X << ',' << coordinate.Y); // lon,lat
    query.AddExtraParam("lang", IsoNameByLanguage(language));

    query.AddExtraParam("mode", "reverse");
    query.AddExtraParam("geocoder_revmode", Options.RevMode);
    query.AddExtraParam("origin", Options.Origin);
    query.AddExtraParam("type", "geo");

    auto reply = Client.SendAsyncQueryF(query, Options.Timeout);
    auto response = reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        const NRTLine::TSearchReply& reply = r.GetValue();
        Y_ENSURE(reply.IsSucceeded(), "request " << reply.GetReqId() << " unsuccessful: " << reply.GetCode());
        const NMetaProtocol::TReport& report = reply.GetReport();
        Y_ENSURE(report.GroupingSize() > 0, "request " << reply.GetReqId() << ": no groupings");
        const NMetaProtocol::TGrouping& grouping = report.GetGrouping(0);
        Y_ENSURE(grouping.GroupSize() > 0, "request " << reply.GetReqId() << ": no groups");
        const NMetaProtocol::TGroup& group = grouping.GetGroup(0);
        Y_ENSURE(group.DocumentSize() > 0, "request " << reply.GetReqId() << ": no documents");
        const NMetaProtocol::TDocument& document = group.GetDocument(0);

        NDrive::TGeocoder::TResponse response;
        response.Title = document.GetArchiveInfo().GetTitle();
        for (auto&& propertie : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
            const TString& key = propertie.GetKey();
            const TString& value = propertie.GetValue();
            if (key == "description") {
                response.Description = value;
                continue;
            }
            if (key == "geoid") {
                TryFromString(value, response.GeoId);
                continue;
            }
            if (key == "kind") {
                response.Kind = value;
                continue;
            }
        }
        return response;
    });
    return response;
}

DECLARE_FIELDS_JSON_SERIALIZER(NDrive::TGeocoder::TResponse);
