#include "formatter.h"

#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>
#include <util/string/split.h>
#include <util/string/vector.h>
#include <util/charset/wide.h>

namespace NDrive {
    static const TMap<ELanguage, TMap<TUtf16String, TUtf16String>> REPLACEMENTS = {
        {LANG_RUS,
         {// Copied from with small changes:
          // https://a.yandex-team.ru/arc/trunk/arcadia/sprav/algorithms/address_formatter/replacements.config.yaml
          {u"бульвар", u"бул."},
          {u"город", u"г."},
          {u"деревня", u"д."},
          {u"набережная", u"наб."},
          {u"переулок", u"пер."},
          {u"поселок городского типа", u"п. г. т."},
          {u"посёлок городского типа", u"п. г. т."},
          {u"проспект", u"пр-т"},
          {u"проезд", u"пр-д"},
          {u"район", u"р."},
          {u"улица", u"ул."},
          {u"шоссе", u"ш."},
          // Additional special cases.
          {u"летия", u"л."}}}};

    static const TSet<char16_t> RUS_VOWELS = {
        u'а', u'е', u'ё', u'и', u'о', u'у', u'ы', u'э', u'ю', u'я'};

    TUtf16String ShortenAddressWord(const TWtringBuf word, ELanguage language) {
        if (language != LANG_RUS || word.size() <= 5) {
            return TUtf16String(word);
        }
        size_t length = word.size();
        for (size_t i = 4; i < word.size(); i++) {
            if (RUS_VOWELS.find(word[i]) == RUS_VOWELS.end()) {
                continue;
            }
            if (RUS_VOWELS.find(word[i - 1]) == RUS_VOWELS.end()) {
                length = i;
                break;
            }
        }
        if (length + 1 >= word.size()) {
            return TUtf16String(word);
        }
        return TUtf16String(word.substr(0, length)) + u".";
    }

    TUtf16String SmartShortenAddress(const TWtringBuf address, size_t length, /*= SIZE_MAX*/ ELanguage language /*= LANG_RUS*/) {
        size_t currentLength = address.size();
        if (currentLength <= length) {
            return TUtf16String(address);
        }
        TVector<TUtf16String> parts;
        TContainerConsumer<TVector<TUtf16String>> partsConsumer(&parts);
        TKeepDelimiters<TContainerConsumer<TVector<TUtf16String>>> consumer(&partsConsumer);
        TSetDelimiter<const char16_t> delim(u" ,-/");
        SplitString(address.data(), address.data() + address.size(), delim, consumer);
        auto it = REPLACEMENTS.find(language);
        if (it != REPLACEMENTS.end()) {
            const auto& replacements = it->second;
            for (auto&& part : parts) {
                if (currentLength <= length) {
                    break;
                }
                auto it = replacements.find(part);
                if (it == replacements.end()) {
                    continue;
                }
                currentLength -= part.length();
                part = it->second;
                currentLength += part.length();
            }
        }
        for (auto&& part : parts) {
            if (currentLength <= length) {
                break;
            }
            currentLength -= part.length();
            part = ShortenAddressWord(part, language);
            currentLength += part.length();
        }
        return JoinStrings(parts, u"");
    }

    TString SmartShortenAddress(const TStringBuf address, size_t length, /*= SIZE_MAX*/ ELanguage language /*= LANG_RUS*/) {
        return WideToUTF8(SmartShortenAddress(UTF8ToWide(address), length, language));
    }

    TUtf16String ShortenAddress(const TWtringBuf address, size_t length, /*= SIZE_MAX*/ ELanguage language /*= LANG_RUS*/) {
        if (address.size() <= length) {
            return TUtf16String(address);
        }
        TVector<TUtf16String> parts;
        TContainerConsumer<TVector<TUtf16String>> partsConsumer(&parts);
        TKeepDelimiters<TContainerConsumer<TVector<TUtf16String>>> consumer(&partsConsumer);
        TSetDelimiter<const char16_t> delim(u" ,-/");
        SplitString(address.data(), address.data() + address.size(), delim, consumer);
        auto it = REPLACEMENTS.find(language);
        if (it != REPLACEMENTS.end()) {
            const auto& replacements = it->second;
            for (auto&& part : parts) {
                auto it = replacements.find(part);
                if (it == replacements.end()) {
                    continue;
                }
                part = it->second;
            }
        }
        auto shortened = JoinStrings(parts, u"");
        if (length < 3 || shortened.size() <= length) {
            return shortened;
        }
        return TUtf16String(shortened.substr(0, length - 3)) + u"...";
    }

    TString ShortenAddress(const TStringBuf address, size_t length, /*= SIZE_MAX*/ ELanguage language /*= LANG_RUS*/) {
        return WideToUTF8(ShortenAddress(UTF8ToWide(address), length, language));
    }

}
