#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/api/search_client/neh_client.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/json/parse.h>

#include <util/generic/ymath.h>
#include <util/string/builder.h>

NDrive::TGeoFeaturesClient::TGeoFeaturesClient(
    const NRTLine::TNehSearchClient& client,
    const NDrive::TRtmrSurgeClient& rtmrClient,
    TAtomicSharedPtr<TScootersSurge> scootersSurge,
    TOptions options
)
    : Client(client)
    , RtmrClient(rtmrClient)
    , ScootersSurge(std::move(scootersSurge))
    , Options(std::move(options))
{
}

NDrive::TGeoFeaturesClient::~TGeoFeaturesClient() {
}

TString NDrive::TGeoFeaturesClient::GetUrl(double latitude, double longitude) {
    return TStringBuilder() << "geo:"
        << static_cast<int>(std::floor(latitude * 100)) << '_'
        << static_cast<int>(std::floor(longitude * 100)) << ':'
        << "features";
}

TString NDrive::TGeoFeaturesClient::GetUrl(ui64 geobaseId) {
    return TStringBuilder() << "geoid:" << geobaseId << ":features";
}

TString NDrive::TGeoFeaturesClient::GetGeoId(double latitude, double longitude) {
    return TStringBuilder()
        << static_cast<int>(std::floor(latitude * 100)) << '_'
        << static_cast<int>(std::floor(longitude * 100));
}

TString NDrive::TGeoFeaturesClient::GetGeoId(ui64 geobaseId) {
    return ToString(geobaseId);
}

template <class... TArgs>
NThreading::TFuture<NDrive::TOptionalGeoFeatures> NDrive::TGeoFeaturesClient::GetImpl(TArgs&&... args) const {
    if (ScootersSurge) {
        return GetScooters(*ScootersSurge, std::forward<TArgs>(args)...);
    }
    auto url = GetUrl(std::forward<TArgs>(args)...);
    auto query = NRTLine::TQuery();
    query.SetText(url);
    query.SetReqClass(Options.ReqIdClass);
    auto reply = Client.SendAsyncQueryF(query, Options.Timeout);
    auto result = reply.Apply([url](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        TOptionalGeoFeatures result;
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        reply.ScanDocs([&result, &url](const NMetaProtocol::TDocument& document) {
            if (document.GetArchiveInfo().GetUrl() != url && document.GetUrl() != url) {
                return;
            }
            result.ConstructInPlace();
            for (auto&& i : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const auto& key = i.GetKey();
                const auto& value = i.GetValue();
                if (value == "null") {
                    continue;
                }
                if (key == "SURGE_LAST_DAY") {
                    result->SurgeLastDay = FromString(value);
                    continue;
                }
                if (key == "SURGE_LAST_2_WEEKS") {
                    result->SurgeLast2Weeks = FromString(value);
                    continue;
                }
                if (key == "SURGE_DAY_WEEK_AGO") {
                    result->SurgeDayWeekAgo = FromString(value);
                    continue;
                }
                if (key == "SESSIONS_LAST_DAY") {
                    result->SessionsLastDay = FromString(value);
                    continue;
                }
                if (key == "SESSIONS_LAST_2_WEEKS") {
                    result->SessionsLast2Weeks = FromString(value);
                    continue;
                }
                if (key == "SESSIONS_DAY_WEEK_AGO") {
                    result->SessionsDayWeekAgo = FromString(value);
                    continue;
                }
                if (key == "CARS_LAST_DAY") {
                    result->CarsLastDay = FromString(value);
                    continue;
                }
                if (key == "CARS_LAST_2_WEEKS") {
                    result->CarsLast2Weeks = FromString(value);
                    continue;
                }
                if (key == "CARS_DAY_WEEK_AGO") {
                    result->CarsDayWeekAgo = FromString(value);
                    continue;
                }
                if (key == "ORDERS_LAST_2_WEEKS") {
                    result->OrdersLast2Weeks = FromString(value);
                    continue;
                }
                if (key == "SURGE_LAST_DAY_GEOBASE") {
                    result->SurgeLastDayGeobase = FromString(value);
                    continue;
                }
                if (key == "SURGE_LAST_2_WEEKS_GEOBASE") {
                    result->SurgeLast2WeeksGeobase = FromString(value);
                    continue;
                }
                if (key == "SURGE_DAY_WEEK_AGO_GEOBASE") {
                    result->SurgeDayWeekAgoGeobase = FromString(value);
                    continue;
                }
                if (key == "SESSIONS_LAST_DAY_GEOBASE") {
                    result->SessionsLastDayGeobase = FromString(value);
                    continue;
                }
                if (key == "SESSIONS_LAST_2_WEEKS_GEOBASE") {
                    result->SessionsLast2WeeksGeobase = FromString(value);
                    continue;
                }
                if (key == "SESSIONS_DAY_WEEK_AGO_GEOBASE") {
                    result->SessionsDayWeekAgoGeobase = FromString(value);
                    continue;
                }
                if (key == "CARS_LAST_DAY_GEOBASE") {
                    result->CarsLastDayGeobase = FromString(value);
                    continue;
                }
                if (key == "CARS_LAST_2_WEEKS_GEOBASE") {
                    result->CarsLast2WeeksGeobase = FromString(value);
                    continue;
                }
                if (key == "CARS_DAY_WEEK_AGO_GEOBASE") {
                    result->CarsDayWeekAgoGeobase = FromString(value);
                    continue;
                }
                if (key == "ORDERS_LAST_2_WEEKS_GEOBASE") {
                    result->OrdersLast2WeeksGeobase = FromString(value);
                }
                if (key == "SURGE_LAST_WEEK_GEOBASE") {
                    result->SurgeLastWeekGeobase = FromString(value);
                }
                if (key == "SESSIONS_LAST_WEEK_GEOBASE") {
                    result->SessionsLastWeekGeobase = FromString(value);
                }

            }
        });
        return result;
    });
    auto response = RtmrClient.GetSurge(url);
    auto realtime = response.Apply([](const NThreading::TFuture<TMaybe<NDrive::TRtmrSurge>>& r) {
        TOptionalGeoFeatures result;
        const auto& surge = r.GetValue();
        if (!surge) {
            return result;
        }
        result.ConstructInPlace();
        result->RtSurgePrediction1 = surge->Prediction;
        result->RtSurgePrediction2 = surge->FloatFeatures.at(surge->FloatFeatures.size() - 3);
        result->RtSurgePrediction3 = surge->FloatFeatures.at(surge->FloatFeatures.size() - 2);
        result->RtSurgePrediction4 = surge->FloatFeatures.at(surge->FloatFeatures.size() - 1);
        result->RtSurgePrediction5 = surge->FloatFeatures.at(4);
        return result;
    });
    auto responseExtended = RtmrClient.GetExtendedSurge(url);
    auto realtimeExtended = responseExtended.Apply([](const NThreading::TFuture<TMaybe<NDrive::TRtmrExtendedSurge>>& r) {
        TOptionalGeoFeatures result;
        auto surge = r.GetValue();
        if (!surge) {
            return result;
        }
        result.ConstructInPlace();
        result->RtSessions60MinExtended = surge->Predictions60min["sessions_prediction"];
        result->RtSessions120MinExtended = surge->Predictions120min["sessions_prediction"];
        result->RtSessions180MinExtended = surge->Predictions180min["sessions_prediction"];
        result->RtSurge60MinExtended = Sigmoid(surge->Predictions60min["ratio_prediction"]);
        result->RtSurge120MinExtended = Sigmoid(surge->Predictions120min["ratio_prediction"]);
        result->RtSurge180MinExtended = Sigmoid(surge->Predictions180min["ratio_prediction"]);
        result->RtCars180MinExtended = surge->Predictions180min["cars_prediction"];
        result->RtCars0005MinWindow = surge->FloatFeatures.at(7);
        result->RtCars0510MinWindow = surge->FloatFeatures.at(15);
        result->RtCars1015MinWindow = surge->FloatFeatures.at(23);
        result->RtCars1520MinWindow = surge->FloatFeatures.at(31);
        result->RtCars2025MinWindow = surge->FloatFeatures.at(39);
        result->RtCars2530MinWindow = surge->FloatFeatures.at(47);
        result->RtCars3035MinWindow = surge->FloatFeatures.at(55);
        result->RtCars3540MinWindow = surge->FloatFeatures.at(63);
        result->RtCars4045MinWindow = surge->FloatFeatures.at(71);
        result->RtCars4550MinWindow = surge->FloatFeatures.at(79);
        result->RtCars5055MinWindow = surge->FloatFeatures.at(87);
        result->RtCars5560MinWindow = surge->FloatFeatures.at(95);
        return result;
    });

    NThreading::TFutures<NDrive::TOptionalGeoFeatures> results = {
        result,
        realtime,
        realtimeExtended,
    };
    auto waiter = NThreading::WaitAll(results);
    return waiter.Apply([
        result = std::move(result),
        realtime = std::move(realtime),
        realtimeExtended = std::move(realtimeExtended)
    ](const NThreading::TFuture<void>& /*w*/) {
        TOptionalGeoFeatures merged;
        if (result.HasValue()) {
            merged = result.GetValue();
        } else {
            merged.ConstructInPlace();
            ERROR_LOG << "an exception occurred:" << NThreading::GetExceptionMessage(result) << Endl;
        }
        if (realtime.HasValue()) {
            auto optionalRealtime = realtime.GetValue();
            if (optionalRealtime) {
                merged->RtSurgePrediction1 = optionalRealtime->RtSurgePrediction1;
                merged->RtSurgePrediction2 = optionalRealtime->RtSurgePrediction2;
                merged->RtSurgePrediction3 = optionalRealtime->RtSurgePrediction3;
                merged->RtSurgePrediction4 = optionalRealtime->RtSurgePrediction4;
                merged->RtSurgePrediction5 = optionalRealtime->RtSurgePrediction5;
            }
        } else {
            ERROR_LOG << "an exception occurred:" << NThreading::GetExceptionMessage(realtime) << Endl;
        }
        if (realtimeExtended.HasValue()) {
            auto optionalRealtimeExtended = realtimeExtended.GetValue();
            if (optionalRealtimeExtended) {
                merged->RtSessions60MinExtended = optionalRealtimeExtended->RtSessions60MinExtended;
                merged->RtSessions120MinExtended = optionalRealtimeExtended->RtSessions120MinExtended;
                merged->RtSessions180MinExtended = optionalRealtimeExtended->RtSessions180MinExtended;
                merged->RtSurge60MinExtended = optionalRealtimeExtended->RtSurge60MinExtended;
                merged->RtSurge120MinExtended = optionalRealtimeExtended->RtSurge120MinExtended;
                merged->RtSurge180MinExtended = optionalRealtimeExtended->RtSurge180MinExtended;
                merged->RtCars180MinExtended = optionalRealtimeExtended->RtCars180MinExtended;
                merged->RtCars0005MinWindow = optionalRealtimeExtended->RtCars0005MinWindow;
                merged->RtCars0510MinWindow = optionalRealtimeExtended->RtCars0510MinWindow;
                merged->RtCars1015MinWindow = optionalRealtimeExtended->RtCars1015MinWindow;
                merged->RtCars1520MinWindow = optionalRealtimeExtended->RtCars1520MinWindow;
                merged->RtCars2025MinWindow = optionalRealtimeExtended->RtCars2025MinWindow;
                merged->RtCars2530MinWindow = optionalRealtimeExtended->RtCars2530MinWindow;
                merged->RtCars3035MinWindow = optionalRealtimeExtended->RtCars3035MinWindow;
                merged->RtCars3540MinWindow = optionalRealtimeExtended->RtCars3540MinWindow;
                merged->RtCars4045MinWindow = optionalRealtimeExtended->RtCars4045MinWindow;
                merged->RtCars4550MinWindow = optionalRealtimeExtended->RtCars4550MinWindow;
                merged->RtCars5055MinWindow = optionalRealtimeExtended->RtCars5055MinWindow;
                merged->RtCars5560MinWindow = optionalRealtimeExtended->RtCars5560MinWindow;
            }
        } else {
            ERROR_LOG << "an exception occurred:" << NThreading::GetExceptionMessage(realtimeExtended) << Endl;
        }
        return merged;
    });
}

NThreading::TFuture<NDrive::TOptionalGeoFeatures> NDrive::TGeoFeaturesClient::Get(const TGeoCoord& coordinate) const {
    return Get(coordinate.Y, coordinate.X);
}

NThreading::TFuture<NDrive::TOptionalGeoFeatures> NDrive::TGeoFeaturesClient::Get(double latitude, double longitude) const {
    return GetImpl(latitude, longitude);
}

NThreading::TFuture<NDrive::TOptionalGeoFeatures> NDrive::TGeoFeaturesClient::Get(ui64 geobaseId) const {
    return GetImpl(geobaseId);
}

NThreading::TFuture<NDrive::TOptionalGeoFeatures> NDrive::TGeoFeaturesClient::GetScooters(const TScootersSurge& scootersSurgeClient, double latitude, double longitude) const {
    return scootersSurgeClient.GetScootersSurge(latitude, longitude).Apply([](const NThreading::TFuture<NDrive::TScootersSurge::TResult>& scootersSurgeFut) -> NDrive::TOptionalGeoFeatures {
        TGeoFeatures result{0, 0, 0, 0, 0};
        try {
            auto scootersSurge = scootersSurgeFut.GetValue();
            result.RtSurgePrediction1 = scootersSurge.Value;
            TVector<double*> featuresRefs = {
                &result.RtSurgePrediction2,
                &result.RtSurgePrediction3,
                &result.RtSurgePrediction4,
                &result.RtSurgePrediction5,
            };
            for (size_t i = 0; i < scootersSurge.Features.size() && i < featuresRefs.size(); i++) {
                *(featuresRefs[i]) = scootersSurge.Features[i];
            }
        } catch (const yexception& err) {
            ERROR_LOG << "Error fetching scooters surge: " << err.what() << Endl;
        }
        return result;
    });
}

NThreading::TFuture<NDrive::TOptionalGeoFeatures> NDrive::TGeoFeaturesClient::GetScooters(const TScootersSurge&, ui64 geobaseId) const {
    WARNING_LOG << "geobaseId (= " << geobaseId << ") search in scooters surge is not implemented yet" << Endl;
    return NThreading::MakeFuture<NDrive::TOptionalGeoFeatures>(Nothing());
}
