#include "geohash.h"

#include <library/cpp/geohash/geohash.h>
#include <rtline/library/geometry/polyline.h>
#include <util/generic/queue.h>
#include <util/generic/set.h>

namespace {

    const int MaxGeohashSize = 1'000'000;

}

namespace NDrive {

    TString GetCoordGeohash(const TGeoCoord& coord, ui8 precision) {
        return NGeoHash::EncodeToString(coord.Y, coord.X, precision);
    }

    TSet<TString> GetPolygonGeohash(const TPolyLine<TGeoCoord>& polygon, ui8 precision) {
        TSet<TString> seen;
        TQueue<TString> q;
        for (auto&& coord : polygon.GetCoords()) {
            TString geoHash = GetCoordGeohash(coord, precision);
            seen.insert(geoHash);
            q.push(geoHash);
        }
        while (q.size()) {
            TString geoHash = q.front();
            q.pop();
            for (auto&& neighbour : NGeoHash::GetNeighbours(geoHash)) {
                if (neighbour && !seen.contains(*neighbour)) {
                    NGeoHash::TBoundingBoxLL bbox = NGeoHash::DecodeToBoundingBox(*neighbour);
                    TVector<TGeoCoord> bboxArea{
                        TGeoCoord(bbox.GetMinX(), bbox.GetMaxY()),
                        TGeoCoord(bbox.GetMaxX(), bbox.GetMaxY()),
                        TGeoCoord(bbox.GetMaxX(), bbox.GetMinY()),
                        TGeoCoord(bbox.GetMinX(), bbox.GetMinY()),
                        TGeoCoord(bbox.GetMinX(), bbox.GetMaxY()),
                    };
                    if (polygon.IsPointInternal(bboxArea.front()) || polygon.AreasIntersection(TPolyLine<TGeoCoord>(bboxArea))) {
                        seen.insert(*neighbour);
                        q.push(*neighbour);
                        Y_ENSURE(seen.size() < MaxGeohashSize);
                    }
                }
            }
        }
        return seen;
    }

}
