#include "abstract.h"

#include <rtline/util/types/accessor.h>

#include <contrib/libs/opencv/include/opencv2/opencv.hpp>
#include <contrib/libs/opencv/include/opencv2/imgcodecs/imgcodecs_c.h>
#include <contrib/libs/opencv/include/opencv2/imgproc/imgproc_c.h>
#include <contrib/libs/opencv/modules/imgcodecs/include/opencv2/imgcodecs.hpp>

#include <library/cpp/mime/detect/detectmime.h>

#include <util/generic/vector.h>
#include <util/stream/file.h>
#include <util/stream/fwd.h>
#include <util/string/builder.h>
#include <util/generic/guid.h>


namespace NImageTransformation {

    TMaybe<cv::Mat> BlobToOpenCV(const TBlob& blob, TMessagesCollector& errors) {
        MimeTypes format = MIME_UNKNOWN;
        {
            TMimeDetector detector;
            if (!detector.Detect(blob.Data(), blob.Size())) {
                WARNING_LOG << "BlobToOpenCV: incomplete detection." << Endl;
            }
            format = detector.Mime();
        }
        switch (format) {
            case MIME_IMAGE_GIF:
            case MIME_IMAGE_BMP:
            case MIME_IMAGE_PJPG:
            case MIME_IMAGE_PNM:
            case MIME_IMAGE_PNG:
            case MIME_IMAGE_WEBP:
            case MIME_IMAGE_JPG:
                try {
                    auto image = cv::imdecode(cv::_InputArray(reinterpret_cast<const ui8*>(blob.Data()), blob.Size()), cv::IMREAD_COLOR);
                    if (image.empty() || image.rows <= 0 || image.cols <= 0) {
                        errors.AddMessage(__LOCATION__, "Fail to load image");
                        return {};
                    }
                    return image;
                } catch (const cv::Exception& e) {
                    errors.AddMessage(__LOCATION__, TStringBuilder() << "Fail to load image with exception: " << e.what());
                    return {};
                }
            default:
                errors.AddMessage(__LOCATION__, "Unsupported image format: " + ToString(format));
                return {};
        }
        return {};
    }

    TMaybe<TBlob> OpenCVToBlob(const cv::Mat& image, TMessagesCollector& errors, const TString& format /* = "jpg" */) {
        if (image.empty()) {
            errors.AddMessage(__LOCATION__, "No image data");
            return {};
        }
        TVector<uchar> result;
        try {
            TString ext = "." + format;
            cv::imencode(ext.c_str(), image, result);
        } catch (const cv::Exception& e) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Fail to get jpeg from image with exception: " << e.what());
            return {};
        }
        TBuffer buf(reinterpret_cast<const char*>(result.data()), result.size());
        return TBlob::FromBuffer(buf);
    }

    cv::Mat BlobToOpenCV(const TBlob& blobData, const TString& format) {
        TString auxPath = "/dev/shm/" + CreateGuidAsString() + "." + format;
        {
            TMemoryInput inp(blobData.Data(), blobData.Size());
            TFileOutput fileOutput(auxPath);
            fileOutput << inp.ReadAll();
        }
        auto result = cv::imread(auxPath.Data());
        TFsPath(auxPath).ForceDelete();
        return result;
    }

    TBlob OpenCVToBlob(const cv::Mat& image, const TString& format) {
        TString auxPath = "/dev/shm/" + CreateGuidAsString() + "." + format;
        cv::imwrite(auxPath.Data(), image);
        auto result = TBlob::FromFile(auxPath);
        TFsPath(auxPath).ForceDelete();
        return result;
    }

    bool IAbstractTransformation::Apply(cv::Mat& source, TMessagesCollector& errors) const {
        try {
            return DoApply(source);
        } catch (const yexception& e) {
            ERROR_LOG << "opencv yexception: " << e.what() << Endl;
            errors.AddMessage("OpenCVYException", e.what());
            return false;
        } catch (const cv::Exception &e) {
            errors.AddMessage("OpenCV", e.what());
            return false;
        }
    }

    bool IAbstractTransformation::Transform(const cv::Mat& source, cv::Mat& result, TMessagesCollector& errors) const {
        result = source;
        return Apply(result, errors);;
    }

    bool IAbstractTransformation::Transform(const TBlob& source, cv::Mat& result, TMessagesCollector& errors, const TString& /* format */) const {
        auto res = BlobToOpenCV(source, errors);
        if (!res) {
            return false;
        }
        result = std::move(*res);
        return Apply(result, errors);
    }

    bool IAbstractTransformation::Transform(const cv::Mat& source, TBlob& result, TMessagesCollector& errors, const TString& format) const {
        cv::Mat resultRaw = source;
        if (Apply(resultRaw, errors)) {
            auto res = OpenCVToBlob(resultRaw, errors, format);
            if (!res) {
                return false;
            }
            result = std::move(*res);
            return true;
        }
        return false;
    }

    bool IAbstractTransformation::Transform(const TBlob& source, TBlob& result, TMessagesCollector& errors, const TString& format) const {
        cv::Mat resultRaw;
        if (!Transform(source, resultRaw, errors, format)) {
            return false;
        }
        auto res = OpenCVToBlob(resultRaw, errors, format);
        if (!res) {
            return false;
        }
        result = std::move(*res);
        return true;
    }
};
