#include "transformation.h"

#include <rtline/util/types/messages_collector.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/logger/global/global.h>

#include <util/stream/file.h>
#include <util/stream/fwd.h>
#include <util/generic/guid.h>

#include <contrib/libs/opencv/include/opencv2/imgcodecs/imgcodecs_c.h>


namespace NImageTransformation {

    bool TTransformationConfig::NeedCompression() const {
        return Compression != 100;
    }

    NJson::TJsonValue TTransformationConfig::SerializeToJson() const {
        NJson::TJsonValue json;
        if (NeedWatermark) {
            json.InsertValue("watermark", WatermarkConfig.SerializeToJson());
        }
        json.InsertValue("compression", Compression);
        NJson::InsertNonNull(json, "resize", MaxSide);
        return json;
    }

    bool TTransformationConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        NeedWatermark = json.Has("watermark");
        if (NeedWatermark && !WatermarkConfig.DeserializeFromJson(json["watermark"])) {
            return false;
        }
        if (json.Has("compression")) {
            unsigned long long compession = 100;
            if (json["compression"].GetUInteger(&compession) && compession > 0 && compession <= 100) {
                Compression = compession;
            } else {
                ERROR_LOG << "Incorrect compression value " << json["compression"].GetStringRobust() << Endl;
                return false;
            }
        }
        return NJson::ParseField(json["resize"], MaxSide, /* readahead = */ false);
    }

    NDrive::TScheme TTransformationConfig::GetScheme() const {
        NDrive::TScheme scheme;
        scheme.Add<TFSStructure>("watermark", "Настройки водяных знаков").SetStructure(WatermarkConfig.GetScheme());
        scheme.Add<TFSNumeric>("compression", "Коэффициент сжатия").SetMax(100).SetMin(0).SetDefault(Compression);
        scheme.Add<TFSNumeric>("resize", "Максимальная длина сторон").SetMin(0);
        return scheme;
    }

    bool TTransformation::Transform(const TString& source, TString& result, TMessagesCollector& errors, const TString& format /* = "jpg" */) const {
        TVector<THolder<IAbstractTransformation>> transformations;
        if (Config.GetMaxSide()) {
            transformations.push_back(MakeHolder<TResize>(Config.GetMaxSide()));
        }
        if (Config.NeedCompression()) {
            transformations.push_back(MakeHolder<TCompress>(Config.GetCompression()));
        }
        if (WatermarkText && Config.GetNeedWatermark()) {
            transformations.push_back(MakeHolder<TWatermark>(Config.GetWatermarkConfig(), WatermarkText));
        }
        if (transformations.empty()) {
            return true;
        }
        TBlob src = TBlob::FromString(source);
        for (const auto& item : transformations) {
            if (!item->Transform(src, src, errors, format)) {
                return false;
            }
        }
        result = TString((const char*)src.Data(), src.Size());
        return true;
    }

}
