#include "watermark.h"

#include <library/cpp/json/writer/json_value.h>

#include <contrib/libs/opencv/modules/imgproc/include/opencv2/imgproc.hpp>

namespace NImageTransformation {

    TWatermarkConfig::TWatermarkConfig()
        : VerticalSpace{ 0, 0, 3.0 }
        , HorizontalSpace{ 0, 0, 0.2 }
    {
    }

    ui32 TWatermarkConfig::TSpaceParams::GetSpace(const ui32 size) const {
        ui32 space = size * (1 + Coefficient);
        space = ::Max<ui32>(space, Min + size);
        if (Max) {
            space = ::Min<ui32>(space, Max + size);
        }
        return space;
    }

    NJson::TJsonValue TWatermarkConfig::SerializeToJson() const {
        NJson::TJsonValue json;
        NJson::InsertField(json, "font", NJson::Stringify(Font));
        json.InsertValue("font_scale", FontScale);
        json.InsertValue("thickness", Thickness);
        json.InsertValue("vertical_space_min", VerticalSpace.Min);
        json.InsertValue("vertical_space_max", VerticalSpace.Max);
        json.InsertValue("vertical_space_coefficient", VerticalSpace.Coefficient);
        json.InsertValue("horizontal_space_min", HorizontalSpace.Min);
        json.InsertValue("horizontal_space_max", HorizontalSpace.Max);
        json.InsertValue("horizontal_space_coefficient", HorizontalSpace.Coefficient);
        json.InsertValue("rotation", Rotation);
        json.InsertValue("alpha", Alpha);
        return json;
    }

    bool TWatermarkConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json["font"], NJson::Stringify(Font))
            && NJson::ParseField(json["font_scale"], FontScale)
            && NJson::ParseField(json["thickness"], Thickness)
            && NJson::ParseField(json["vertical_space_min"], VerticalSpace.Min)
            && NJson::ParseField(json["vertical_space_max"], VerticalSpace.Max)
            && NJson::ParseField(json["vertical_space_coefficient"], VerticalSpace.Coefficient)
            && NJson::ParseField(json["horizontal_space_min"], HorizontalSpace.Min)
            && NJson::ParseField(json["horizontal_space_max"], HorizontalSpace.Max)
            && NJson::ParseField(json["horizontal_space_coefficient"], HorizontalSpace.Coefficient)
            && NJson::ParseField(json["rotation"], Rotation)
            && NJson::ParseField(json["alpha"], Alpha);
    }

    NDrive::TScheme TWatermarkConfig::GetScheme() const {
        NDrive::TScheme scheme;
        scheme.Add<TFSVariants>("font", "Шрифт").InitVariants<EFont>().SetDefault(::ToString(EFont::Duplex));
        scheme.Add<TFSNumeric>("font_scale", "Размер шрифта").SetMin(0).SetDefault(FontScale);
        scheme.Add<TFSNumeric>("thickness", "Толщина шрифта").SetMin(0).SetDefault(Thickness);
        scheme.Add<TFSNumeric>("vertical_space_min", "Минимальный вертикальный отступ").SetMin(0).SetDefault(0);
        scheme.Add<TFSNumeric>("vertical_space_max", "Максимальный вертикальный отступ").SetMin(0).SetDefault(0);
        scheme.Add<TFSNumeric>("vertical_space_coefficient", "Коэффициент для вертикального отступа").SetMax(100).SetMin(0).SetDefault(VerticalSpace.Coefficient);
        scheme.Add<TFSNumeric>("horizontal_space_min", "Минимальный горизонтальный отступ").SetMin(0).SetDefault(100);
        scheme.Add<TFSNumeric>("horizontal_space_max", "Максимальный горизонтальный отступ отступ").SetMin(0).SetDefault(0);
        scheme.Add<TFSNumeric>("horizontal_space_coefficient", "Коэффициент для горизонтального отступа").SetMin(0).SetDefault(HorizontalSpace.Coefficient);
        scheme.Add<TFSNumeric>("rotation", "Угл поворота").SetDefault(Rotation);
        scheme.Add<TFSNumeric>("alpha", "Альфа канал").SetMax(1).SetMin(0).SetDefault(Alpha);
        return scheme;
    }


    TWatermark::TWatermark(const TWatermarkConfig& config, const TString& text)
        : Config(config)
        , Text(text)
    {
    }

    bool TWatermark::DoApply(cv::Mat& source) const {
        const auto height = source.rows * 2;
        const auto width = source.cols * 2;
        cv::Mat mask;
        {
            cv::Mat tmp(height, width, CV_8UC4, cv::Scalar(0));
            cv::cvtColor(tmp, mask, CV_BGRA2RGB);
        }
        int baseLine = 0;
        auto fontSize = cv::getTextSize(Text.c_str(), (ui32)Config.GetFont(), Config.GetFontScale(), Config.GetThickness(), &baseLine);
        const ui32 vertStep = Config.GetVerticalSpace().GetSpace(fontSize.height);
        const ui32 horStep = Config.GetHorizontalSpace().GetSpace(fontSize.width);
        for (ui32 vpos(vertStep), rowCount(0); vpos < (ui32)height; vpos += vertStep, ++rowCount) {
            const ui32 shift = horStep * (rowCount % 2) / 2;
            for (ui32 hpos(0); hpos < (ui32)width; hpos += horStep) {
                putText(mask, Text.c_str(), cv::Point(hpos - shift, vpos), (ui32)Config.GetFont(), Config.GetFontScale(), cv::Scalar::all(255), Config.GetThickness());
            }
        }
        {
            cv::Point2f center(mask.cols / 2.0, mask.rows / 2.0);
            cv::Mat rotation = cv::getRotationMatrix2D(center, Config.GetRotation(), 1.0);
            cv::warpAffine(mask, mask, rotation, mask.size());
            cv::Rect roi(mask.cols / 4, mask.rows / 4, mask.cols / 2, mask.rows / 2);
            mask = mask(roi);
        }
        addWeighted(mask, Config.GetAlpha(), source, 1 - Config.GetAlpha(), 0.0, source);
        return true;
    }
};
