#include "its_watcher.h"
#include "message.h"

#include <market/library/external_task_executer/event_fd.h>
#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/mediator/messenger.h>
#include <kernel/daemon/bomb.h>

#include <util/network/poller.h>
#include <util/string/join.h>
#include <util/system/file.h>
#include <util/stream/file.h>
#include <util/stream/format.h>
#include <library/cpp/json/json_reader.h>
#include <util/folder/path.h>

#include <sys/inotify.h>

namespace {
    constexpr ui32 INOTIFY_DIR_MASK = IN_CREATE | IN_DELETE_SELF | IN_DELETE | IN_MOVED_TO | IN_MOVED_FROM;
    constexpr ui32 INOTIFY_FILE_MASK = IN_MODIFY;
    TString EMPTY_MD5_HASH = "d41d8cd98f00b204e9800998ecf8427e"; // MD5("")

    struct TFileInfo {
        TFsPath Path;
        TString Hash;
    };

    using TChanges = TVector<TString>;

    class TInotifyEventHelper {
    public:
        explicit TInotifyEventHelper(const TFsPath& path)
            : BasePath(path)
            , InotifyDesc(inotify_init())
        {
            Y_ENSURE(InotifyDesc != INVALID_FHANDLE, "ITS watcher: cannot create ITS-file change notifier");
            Poller.WaitRead(InotifyDesc, &InotifyDesc);
            Poller.WaitRead(StopEventDesc.GetHandle(), nullptr);

            Y_ENSURE(path);
            Y_ENSURE(path.Exists(), "ITS watcher: watch path " << path.GetPath().Data() << " does not exist");
            DirWatchDesc = inotify_add_watch(InotifyDesc, path.GetPath().Data(), INOTIFY_DIR_MASK);
            Y_ENSURE(DirWatchDesc >= 0, "ITS watcher: cannot add directory watch");
        }

    public:
        TFsPath GetFullPath(const TString& name) {
            return BasePath / TFsPath(name);
        }

        TMaybe<TChanges> WaitForChanges() {
            if (FirstRun) {
                TChanges updatedFiles;
                TVector<TString> names;
                BasePath.ListNames(names);
                for (auto& name : names) {
                    if (!GetFullPath(name).IsFile()) {
                        continue;
                    }
                    if (int fd = AddWatch(name); fd >= 0 && FilesInfo[fd].Hash != EMPTY_MD5_HASH) {
                        updatedFiles.push_back(name);
                    }
                }
                FirstRun = false;
                return updatedFiles;
            }

            constexpr size_t EVENTS_NUM = 8;

            TTempBuf eventsBuf(EVENTS_NUM * sizeof(void*));
            void** events = reinterpret_cast<void**>(eventsBuf.Data());
            size_t eventsCount = Poller.WaitI(events, EVENTS_NUM);

            TChanges result;
            for (size_t i = 0; i < eventsCount; ++i) {
                if (events[i] == reinterpret_cast<void*>(&InotifyDesc)) {
                    // inotify event
                    auto changes = CheckInotifyEvent();
                    if (!changes) {
                        return Nothing();
                    }
                    result.insert(result.end(), changes->begin(), changes->end());
                } else if (events[i] == nullptr) {
                    // stop event
                    return Nothing();
                }
            }
            return result;
        }

        TMaybe<TChanges> CheckInotifyEvent() {
            INFO_LOG << "ITS watcher: checking inotify events" << Endl;

            constexpr size_t EVENTS_BUF_SIZE = 1024 * (sizeof(inotify_event) + NAME_MAX + 1);
            TTempBuf inotifyEvents(EVENTS_BUF_SIZE);

            ssize_t readSize = InotifyDesc.Read(inotifyEvents.Data(), inotifyEvents.Size());
            Y_ENSURE(readSize >= 0, "ITS watcher: bad read from inotify descriptor");

            TChanges updatedFiles;

            for (ssize_t p = 0; p < readSize;) {
                auto event = reinterpret_cast<const inotify_event*>(inotifyEvents.Data() + p);

                if (event->wd == DirWatchDesc && !(event->mask & static_cast<ui32>(IN_ISDIR))) {
                    TString name = event->name;
                    if (event->mask & (static_cast<ui32>(IN_CREATE) | static_cast<ui32>(IN_MOVED_TO))) {
                        INFO_LOG << "ITS watcher: new file: " << name << Endl;
                        if (int fd = AddWatch(name); fd >= 0 && FilesInfo[fd].Hash != EMPTY_MD5_HASH) {
                            updatedFiles.push_back(name);
                        }
                    } else if (event->mask & (static_cast<ui32>(IN_DELETE) | static_cast<ui32>(IN_MOVED_FROM))) {
                        INFO_LOG << "ITS watcher: removed file: " << name << Endl;
                        if (FilesInfo[NameToDesc[name]].Hash != EMPTY_MD5_HASH) {
                            updatedFiles.push_back(name);
                        }
                        RemoveWatch(name);
                    } else if (event->mask & static_cast<ui32>(IN_DELETE_SELF)) {
                        ALERT_LOG << "ITS watcher: directory with watched file has been removed" << Endl;
                        return Nothing();
                    } else {
                        WARNING_LOG << "ITS watcher: unwatched inotify event type for directory: " << Hex(event->mask) << Endl;
                    }
                } else {
                    auto name = FilesInfo[event->wd].Path.GetName();
                    if (event->mask & static_cast<ui32>(IN_MODIFY)) {
                        INFO_LOG << "File has been modified: " << name << Endl;
                        if (UpdateInfo(event->wd)) {
                            updatedFiles.push_back(name);
                        }
                    } else {
                        WARNING_LOG << "ITS watcher: unwatched inotify event type for file " << name << ": " << Hex(event->mask) << Endl;
                    }
                }
                p += sizeof(inotify_event) + event->len;
            }

            return updatedFiles;
        }

        int AddWatch(const TString& name) {
            auto path = GetFullPath(name);
            int fileWatchDesc = inotify_add_watch(InotifyDesc, path.GetPath().Data(), INOTIFY_FILE_MASK);
            if (fileWatchDesc < 0) {
                return fileWatchDesc;
            }
            FilesInfo[fileWatchDesc] = {path, MD5::File(path)};
            NameToDesc[name] = fileWatchDesc;
            return fileWatchDesc;
        }

        void RemoveWatch(const TString& name) {
            int desc = NameToDesc[name];
            FilesInfo.erase(desc);
            NameToDesc.erase(name);
        }

        bool UpdateInfo(int desc) {
            auto& info = FilesInfo[desc];
            auto hash = MD5::File(info.Path);
            if (hash != info.Hash) {
                INFO_LOG
                    << "ITS watcher: ITS file has been modified: "
                    << info.Path.GetName() << Endl;
                info.Hash = hash;
                return true;
            }

            return false;
        }
        void Stop() {
            StopEventDesc.Write(1);
        }

    private:
        TFsPath BasePath;
        TFileHandle InotifyDesc;
        bool FirstRun = true;
        TMap<int, TFileInfo> FilesInfo;
        TMap<TString, int> NameToDesc;
        NTaskExecuter::TEventFd StopEventDesc;
        TSocketPoller Poller;
        int DirWatchDesc;
    };

    class TItsWatcher: public IThreadFactory::IThreadAble, public NDrive::IItsWatcher {
    public:
        explicit TItsWatcher(const TFsPath& path)
            : ItsDirPath(path)
            , InotifyEvent(MakeHolder<TInotifyEventHelper>(path))
        {
            INFO_LOG << "ITS watcher: watching directory: " << ItsDirPath << Endl;
        }

        ~TItsWatcher() override {
            InotifyEvent->Stop();
            if (ReadThread) {
                ReadThread->Join();
                ReadThread.Reset();
            }
        }

    public:
        void Run() override {
            if (!ReadThread) {
                ReadThread = SystemThreadFactory()->Run(this);
            }
        }

    protected:
        void DoExecute() override {
            INFO_LOG << "ITS watcher: start" << Endl;
            while (true) {
                try {
                    auto changed = InotifyEvent->WaitForChanges();
                    if (!changed.Defined()) {
                        INFO_LOG << "ITS watcher: got a message for its watcher shutdown" << Endl;
                        break;
                    }
                    if (changed->empty()) {
                        continue;
                    }
                    NDrive::TItsChangedMessage message;
                    for (auto&& name : *changed) {
                        auto path = ItsDirPath / TFsPath(name);
                        auto value = path.Exists() ? TFileInput(path).ReadAll() : "";
                        message.MutableChangedValues()[name] = value;
                        INFO_LOG << "ITS watcher: updated value: " << name << " = " << value << Endl;
                    }
                    INFO_LOG << "ITS watcher: sending updated value" << Endl;
                    SendGlobalMessage(message);
                } catch (const std::exception& e) {
                    ALERT_LOG << "ITS watcher: exception: " << FormatExc(e) << Endl;
                }
            }
            INFO_LOG << "ITS watcher: finished" << Endl;
        }

    protected:
        TFsPath ItsDirPath;
        THolder<IThreadFactory::IThread> ReadThread{};
        THolder<TInotifyEventHelper> InotifyEvent;
    };
}

namespace NDrive {
    THolder<IItsWatcher> MakeItsWatcher(const TFsPath& path) {
        return MakeHolder<TItsWatcher>(path);
    }
}
