#include "client.h"

#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/messages_collector.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/xml/document/xml-document.h>

#include <util/stream/str.h>
#include <util/string/builder.h>
#include <util/string/subst.h>


namespace {
    const TString AuthRequestTemplate = "<s:Envelope xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\"><s:Body xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\"><Auth xmlns=\"http://tempuri.org/\"><request><Login>__JEEP_LOGIN__</Login><Password>__JEEP_PASSWORD__</Password></request></Auth></s:Body></s:Envelope>";
    const TString LeadRequestTemplate = "<s:Envelope xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\"><s:Body xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\"><LoadRussianLeads xmlns=\"http://tempuri.org/\"><request><Token>__AUTH_TOKEN__</Token>__LEADDATA__</request></LoadRussianLeads></s:Body></s:Envelope>";

    template <class TEntity>
    class TApplier {
    public:
        NThreading::TFuture<typename TEntity::THandleResult> operator()(const NThreading::TFuture<NUtil::THttpReply>& r) const {
            const auto& report = r.GetValue();
            TUnistatSignalsCache::SignalAdd("jeep-" + TEntity::GetName() + "-api-codes", ToString(report.Code()), 1);
            if (!report.HasReply()) {
                ythrow yexception() << "No reply: " << report.ErrorMessage();
            }
            if (!report.IsSuccessReply()) {
                ythrow yexception() << "Error getting objects(" << TEntity::GetName() << "): " << report.GetDebugReply();
            }
            return TEntity::HandleXml(report.Content());
        }

    private:
        TString FieldName;
    };

    class TAuthEntity {
    public:
        using THandleResult = TString;

        static NThreading::TFuture<THandleResult> HandleXml(const TString& xmlStr) {
            NXml::TDocument xml(xmlStr, NXml::TDocument::String);
            NXml::TNamespacesForXPath nss = {{"x", "http://tempuri.org/"}};
            if (auto node = xml.Root().Node("//x:ResultCodeId", /* quiet = */ true, nss); node.IsNull() || node.Value<TString>() != "200") {
                ythrow yexception() << "Error getting auth token reply: " << xmlStr;
            }
            if (auto node = xml.Root().Node("//x:Token", /* quiet = */ true, nss); !node.IsNull() && node.Value<TString>()) {
                return NThreading::MakeFuture(node.Value<TString>());
            }
            ythrow yexception() << "Error parse token reply: " << xmlStr;
        }

        static TString GetName() {
            return "auth";
        }
    };

    class TLeadEntity {
    public:
        using THandleResult = void;

        static  NThreading::TFuture<THandleResult> HandleXml(const TString& xmlStr) {
            NXml::TDocument xml(xmlStr, NXml::TDocument::String);
            auto node = xml.Root().Node("//x:ResultCodeId", /* quiet = */ true, {{"x", "http://tempuri.org/"}});
            if (node.IsNull() || node.Value<TString>() != "200") {
                ythrow yexception() << "Error getting lead reply: " << xmlStr;
            }
            return NThreading::MakeFuture();
        }

        static TString GetName() {
            return "lead";
        }
    };

}


namespace NDrive {

    TJeepClient::TJeepClient(const TJeepApiConfig& config)
        : Config(config)
        , Client(MakeAtomicShared<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig()))
    {
    }

    const TJeepApiConfig& TJeepClient::GetConfig() const {
        return Config;
    }

    NThreading::TFuture<TString> TJeepClient::GetAuthToken() const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath());
        request.SetContentType("application/soap+xml");
        {
            TString payload = AuthRequestTemplate;
            SubstGlobal(payload, "__JEEP_LOGIN__", Config.GetLogin());
            SubstGlobal(payload, "__JEEP_PASSWORD__", Config.GetToken());
            request.SetPostData(payload);
        }
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TAuthEntity>());
    }

    NThreading::TFuture<void> TJeepClient::SendLead(const TJeepRequestData& context) const {
        TString payload = LeadRequestTemplate;
        {
            TStringStream s;
            context.ToString(s);
            SubstGlobal(payload, "__LEADDATA__", s.Str());
        }
        const auto deadline = Now() + Config.GetRequestTimeout();
        return GetAuthToken().Apply([client = Client, deadline, uri = Config.GetUriPath(), payload = std::move(payload)](const NThreading::TFuture<TString>& token) mutable {
            NNeh::THttpRequest request;
            request.SetUri(uri);
            request.SetContentType("application/soap+xml");
            SubstGlobal(payload, "__AUTH_TOKEN__", token.GetValue());
            request.SetPostData(payload);
            return client->SendAsync(request, deadline).Apply(TApplier<TLeadEntity>());
        });
    }

}
