#include "entities.h"

#include <util/generic/serialized_enum.h>

namespace {

    template <typename TKey, typename TValue>
    void AddField(IOutputStream& os, const TKey& key, const TValue& value) {
        os << "<" << key << ">" << value << "</" << key << ">";
    }

    template <class T>
    void AddStructField(IOutputStream& os, const TString& key, const T& object) {
        os << "<" << key << ">";
        object.ToString(os);
        os << "</" << key << ">";
    }

    void AddNoNullField(IOutputStream& os, const TString& key, const TString& value) {
        if (value) {
            os << "<" << key << ">" << value << "</" << key << ">";
        }
    }

}


namespace NDrive {
        void TJeepRequestData::TPersonalData::ToString(IOutputStream& os) const {
            AddField(os, "ExternalId", UserId);
            AddField(os, "Name", UserName);
            AddField(os, "Surname", Surname);
            AddNoNullField(os, "Phone", Phone);
            AddNoNullField(os, "Email", Email);
            AddField(os, "Country", "RUS");
            switch (City) {
            case ECity::Msk:
                AddField(os, "RegionId", "77");
                AddField(os, "City", "Г. Москва");
                break;
            case ECity::Spb:
                AddField(os, "RegionId", "78");
                AddField(os, "City", "Г. Санкт-Петербург");
                break;
            case ECity::Kzn:
                AddField(os, "RegionId", "16");
                AddField(os, "City", "Г. Казань");
                break;
            }
        }

        void TJeepRequestData::TPrivacyData::ToString(IOutputStream& os) const {
            AddField(os, "DisclaimerId", DisclaimerId);
            for (const auto& item : GetEnumAllValues<TPrivacyData::EAgreements>()) {
                if (item == EAgreements::ProcessingPermission) {
                    AddField(os, EAgreements::ProcessingPermission, "true");
                } else {
                    AddField(os, item, (item & Agreements) == item ? "true" : "false");
                }
            }
        }

        void TJeepRequestData::TLeadData::ToString(IOutputStream& os) const {
            AddField(os, "LeadTypeId", LeadTypeId);
            AddField(os, "InterestBrandCode", InterestBrandCode);
            AddField(os, "InterestModelCode", InterestModelCode);
            AddField(os, "DealerId", DealerId);
            AddField(os, "LeadStatusId", LeadStatusId);
        }

        void TJeepRequestData::TRequestData::ToString(IOutputStream& os) const {
            AddField(os, "ExternalRequestId", ExternalRequestId);
            AddField(os, "OriginId", OriginId);
            AddField(os, "RequestDate", RequestDate.ToIsoStringLocalUpToSeconds());
        }

        TJeepRequestData::TRequestExtraField::TRequestExtraField(EExtraFieldName type, const TString& data)
            : ExtraFieldName(type)
            , Data(data)
        {
        }

        void TJeepRequestData::TRequestExtraField::ToString(IOutputStream& os) const {
            AddField(os, "Id", Id);
            AddField(os, "ExtraFieldName", ExtraFieldName);
            AddField(os, "ExtraField", Data);
        }

        void TJeepRequestData::ToString(IOutputStream& os) const {
            AddStructField(os, "PersonalData", PersonalData);
            AddStructField(os, "PrivacyData", PrivacyData);
            AddStructField(os, "LeadData", LeadData);
            AddStructField(os, "RequestData", RequestData);
            if (!RequestExtraFields.empty()) {
                os << "<RequestExtraFields>";
                for (const auto& field : RequestExtraFields) {
                    AddStructField(os, "RequestExtraField", field);
                }
                os << "</RequestExtraFields>";
            }
        }
}
