#include "client.h"

#include <library/cpp/json/json_reader.h>

#include <library/cpp/string_utils/url/url.h>

NDrive::TLBSClient::TResponse ParseResponse(const NUtil::THttpReply& reply);

NJson::TJsonValue NDrive::TLBSClient::TCell::ToJson() const {
    NJson::TJsonValue result;
    result["countrycode"] = MCC;
    result["operatorid"] = MNC;
    result["cellid"] = CellId;
    result["lac"] = LAC;
    result["signal_strength"] = SignalLevel;
    result["age"] = Age.MilliSeconds();
    return result;
}

NJson::TJsonValue NDrive::TLBSClient::TQuery::ToJson(TStringBuf apikey) const {
    NJson::TJsonValue result;
    result["common"]["version"] = "1.0";
    result["common"]["api_key"] = apikey;
    for (auto&& cell : Cells) {
        result["gsm_cells"].AppendValue(cell.ToJson());
    }
    if (IpV4) {
        result["ip"]["address_v4"] = IpV4.GetRef();
    }
    return result;
}

bool NDrive::TLBSClient::TPosition::TryFromJson(const NJson::TJsonValue& value) {
    return
        value["latitude"].GetDouble(&Latitude) &&
        value["longitude"].GetDouble(&Longitude) &&
        value["altitude"].GetDouble(&Altitude) &&
        value["precision"].GetDouble(&Precision) &&
        value["altitude_precision"].GetDouble(&AltitudePrecision) &&
        value["type"].GetString(&Type);
}

bool NDrive::TLBSClient::TResponse::TryFromJson(const NJson::TJsonValue& value) {
    if (value.Has("position")) {
        Position.ConstructInPlace();
        return Position->TryFromJson(value["position"]);
    }
    if (value.Has("error")) {
        Error = value["error"].GetStringRobust();
        return true;
    }
    return false;
}

NDrive::TLBSClient::TLBSClient(const TString& token, const TString& url /*= "http://api-int.lbs.yandex.net/geolocation"*/, TAsyncDelivery::TPtr asyncDelivery/* = nullptr*/)
    : Base(GetPathAndQuery(url))
    , Token(token)
    , Timeout(TDuration::Seconds(10))
    , AsyncDelivery(asyncDelivery ? asyncDelivery : MakeAtomicShared<TAsyncDelivery>())
    , ExternalAsyncDelivery(asyncDelivery)
    , Requester(MakeHolder<NNeh::THttpClient>(AsyncDelivery))
{
    if (!ExternalAsyncDelivery) {
        AsyncDelivery->Start(4, 4);
    }

    NSimpleMeta::TConfig config;
    config.SetMaxAttempts(1);
    config.SetGlobalTimeout(TDuration::Seconds(10));
    Requester->RegisterSource("lbs", TString(GetOnlyHost(url)), 443, config, true);
}

NDrive::TLBSClient::~TLBSClient() {
    if (!ExternalAsyncDelivery) {
        AsyncDelivery->Stop();
    }
}

NThreading::TFuture<NDrive::TLBSClient::TResponse> NDrive::TLBSClient::Locate(const TQuery& query) const {
    const NNeh::THttpRequest& request = CreateRequest(query);
    const TBlob& post = request.GetPostData();

    DEBUG_LOG << "Requesting " << Base << " " << TStringBuf(post.AsCharPtr(), post.Size()) << Endl;
    NThreading::TFuture<NUtil::THttpReply> reply = Requester->SendAsync(request, Now() + Timeout);
    NThreading::TFuture<TResponse> result = reply.Apply([](const NThreading::TFuture<NUtil::THttpReply>& re) {
        const auto& response = re.GetValue();
        DEBUG_LOG << "Received " << response.Code() << ": " << response.Content() << Endl;
        return ParseResponse(response);
    });
    return result;
}

NNeh::THttpRequest NDrive::TLBSClient::CreateRequest(const TQuery& query) const {
    NNeh::THttpRequest request;
    request.AddHeader("Content-Type", "application/x-www-form-urlencoded");
    request.SetRequestType("POST");
    request.SetUri(Base);

    TString post = "json=" + query.ToJson(Token).GetStringRobust();
    request.SetPostData(TBlob::FromString(post));
    return request;
}

NDrive::TLBSClient::TResponse ParseResponse(const NUtil::THttpReply& reply) {
    NDrive::TLBSClient::TResponse result;
    if (!reply.IsSuccessReply()) {
        result.Error = "cannot fetch: " + ToString<ui32>(reply.Code()) + " " + reply.Content();
        return result;
    }

    NJson::TJsonValue response;
    if (!NJson::ReadJsonFastTree(reply.Content(), &response)) {
        result.Error = "cannot parse json: " + reply.Content();
        return result;
    }
    if (!result.TryFromJson(response)) {
        result.Error = "cannot parse response: " + response.GetStringRobust();
        return result;
    }
    return result;
}
