#include "linker.h"

#include <maps/doc/proto/analyzer/gpssignal.pb.h>
#include <maps/doc/proto/yandex/maps/proto/matcher/matched_path.pb.h>

#include <rtline/library/unistat/signals.h>
#include <rtline/util/types/timestamp.h>

namespace {
    TUnistatSignal<> MapsLinkerException({ "maps_linker-exception" }, false);
    TUnistatSignal<> MapsLinkerRequest({ "maps_linker-request" }, false);
    TUnistatSignal<> MapsLinkerResponseServerError({ "maps_linker-response-server_error" }, false);
    TUnistatSignal<> MapsLinkerResponseUserError({ "maps_linker-response-user_error" }, false);
    TUnistatSignal<> MapsLinkerResponseTimes({ "maps_linker-response-times" }, NRTLineHistogramSignals::IntervalsRTLineReply);
}

NDrive::TMapsLinker::TMapsLinker(const TString& endpoint, TAtomicSharedPtr<NGraph::ILinker> relinker)
    : Endpoint(endpoint)
    , Client(MakeHolder<NNeh::THttpClient>(endpoint))
    , Relinker(std::move(relinker))
    , RequestTimeout(TDuration::Seconds(1))
{
}

NThreading::TFuture<NGraph::TRouter::TMatch> NDrive::TMapsLinker::Match(
    const NGraph::TRouter::TTimedGeoCoordinates& coordinates,
    const NGraph::TMatchingOptions& options
) const {
    yandex::maps::proto::analyzer::GpsSignalCollectionData collection;
    for (auto&& coordinate : coordinates) {
        auto data = collection.add_signals();
        data->set_lat(coordinate.Y);
        data->set_lon(coordinate.X);
        data->set_average_speed(coordinate.Speed);
        data->set_time(coordinate.Timestamp.Seconds());
    }
    auto request = NNeh::THttpRequest();
    request.SetUri("/collect_batch_of_gpssignal_sync/");
    request.SetPostData(collection.SerializeAsString());

    auto start = Now();
    auto response = Yensured(Client)->SendAsync(request);
    MapsLinkerRequest.Signal(1);
    auto matched = response.Apply([coordinates, start](const NThreading::TFuture<NNeh::THttpReply>& r) {
        if (!r.HasValue()) {
            MapsLinkerException.Signal(1);
        }

        const auto finish = Now();
        const auto duration = finish - start;
        const auto& response = r.GetValue();
        const auto code = response.Code();
        if (IsServerError(code)) {
            MapsLinkerResponseServerError.Signal(1);
        }
        if (IsUserError(code)) {
            MapsLinkerResponseUserError.Signal(1);
        }
        response.EnsureSuccessfulReply();
        MapsLinkerResponseTimes.Signal(duration.MilliSeconds());

        yandex::maps::proto::matcher::MatchedPath matchedPath;
        Y_ENSURE(matchedPath.ParseFromString(response.Content()));

        TMap<TTimestamp, size_t> indices;
        for (size_t i = 0; i < coordinates.size(); ++i) {
            indices[coordinates[i].Timestamp] = i;
        }

        NGraph::TRouter::TMatch match;

        for (auto&& point : matchedPath.points()) {
            TGeoCoord coordinate;
            coordinate.X = point.point().lon();
            coordinate.Y = point.point().lat();
            auto edgeId = point.position_on_graph().persistent_edge_id();
            if (match.Elements.empty() || match.Elements.back().EdgeExternalId != edgeId) {
                match.Elements.emplace_back();
            }
            auto& edge = match.Elements.back();
            edge.EdgeExternalId = edgeId;

            auto timestamp = TInstant::Seconds(point.timestamp());
            auto indice = indices.FindPtr(timestamp);
            if (indice) {
                edge.Indices.push_back(*indice);
                edge.RawIndices.push_back(*indice);
                edge.Projection.push_back(coordinate);
                Y_ENSURE(*indice < coordinates.size(), *indice << " < " << coordinates.size());
                edge.Points.push_back(coordinates[*indice]);
            }
        }
        return match;
    });
    if (Relinker) {
        return Rematch(std::move(matched), options);
    } else {
        return matched;
    }
}

NThreading::TFuture<NGraph::TRouter::TMatch> NDrive::TMapsLinker::Rematch(NThreading::TFuture<NGraph::TRouter::TMatch>&& match, const NGraph::TMatchingOptions& options) const {
    return match.Apply([options, relinker = Relinker](const NThreading::TFuture<NGraph::TRouter::TMatch>& m) {
        NGraph::TRouter::TTimedGeoCoordinates coordinates;
        for (auto&& element : m.GetValue().Elements) {
            for (auto&& coordinate : element.Projection) {
                coordinates.emplace_back(coordinate, TInstant::Zero());
            }
        }
        return Yensured(relinker)->Match(coordinates, options);
    });
}
