#include "router.h"

#include <maps/doc/proto/converters/geolib/include/yandex/maps/geolib3/proto.h>
#include <maps/doc/proto/yandex/maps/proto/common2/response.pb.h>
#include <maps/doc/proto/yandex/maps/proto/driving/matrix.pb.h>
#include <maps/doc/proto/yandex/maps/proto/driving/summary.pb.h>
#include <maps/doc/proto/yandex/maps/proto/bicycle/summary.pb.h>
#include <maps/doc/proto/yandex/maps/proto/masstransit_matrix/matrix.pb.h>
#include <maps/doc/proto/yandex/maps/proto/masstransit/route.pb.h>
#include <maps/doc/proto/yandex/maps/proto/masstransit/section.pb.h>
#include <maps/doc/proto/yandex/maps/proto/masstransit/summary.pb.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/ptr.h>
#include <maps/libs/geolib/include/convex_hull.h>

#include <library/cpp/cgiparam/cgiparam.h>

namespace NDrive {

NThreading::TFuture<NGraph::TRouter::TOptionalRoute> TVehicleRouter::GetSummary(
    const TGeoCoord& from,
    const TGeoCoord& to,
    bool avoidTolls, /*= false*/
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    auto summary = RequestAsync("/v2/summary", "summary", {from, to}, avoidTolls, 1, reqId);
    return summary.Apply([avoidTolls](const NThreading::TFuture<TString>& r) -> NGraph::TRouter::TOptionalRoute {
        const TString& content = r.GetValue();
        yandex::maps::proto::driving::summary::Summaries result;
        Y_ENSURE(result.ParseFromString(content));
        const auto& summaries = result.summaries();
        if (summaries.empty()) {
            return Nothing();
        }
        const auto& weight = summaries[0].weight();
        NGraph::TRouter::TRoute route;
        route.HasPath = true;
        route.Time = weight.time_with_traffic().value();
        route.Length = weight.distance().value();
        route.Origin = NGraph::TRouter::EOrigin::MapsRouter;
        if (avoidTolls) {
            route.Flags |= NGraph::TRouter::EFlag::AvoidTolls;
        }
        return MakeMaybe(std::move(route));
    });
}

TVehicleRouter::TVehicleRouter(
    const TString& hostUrl,
    TMaybe<TTvmAuth> tvmAuth,
    const TDuration& timeout
)
    : TBase(hostUrl, tvmAuth, timeout)
{
}

NThreading::TFuture<NGraph::TRouter::TOptionalRoute> TScooterRouter::GetSummary(
    const TGeoCoord& from,
    const TGeoCoord& to,
    bool avoidTolls, /*= false*/
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    auto summary = RequestAsync("/v2/summary", "summary", {from, to}, avoidTolls, 1, reqId, "scooter");
    return summary.Apply([](const NThreading::TFuture<TString>& r) -> NGraph::TRouter::TOptionalRoute {
        const TString& content = r.GetValue();
        yandex::maps::proto::bicycle::summary::Summaries result;
        Y_ENSURE(result.ParseFromString(content));
        const auto& summaries = result.summaries();
        if (summaries.empty()) {
            return Nothing();
        }
        const auto& weight = summaries[0].weight();
        NGraph::TRouter::TRoute route;
        route.HasPath = true;
        route.Time = weight.time().value();
        route.Length = weight.distance().value();
        route.Origin = NGraph::TRouter::EOrigin::MapsRouter;
        return MakeMaybe(std::move(route));
    });
}

TScooterRouter::TScooterRouter(
    const TString& hostUrl,
    TMaybe<TTvmAuth> tvmAuth,
    const TDuration& timeout
)
    : TBase(hostUrl, tvmAuth, timeout)
{
}


TVector<TGeoCoord> ParsePedestrianIsochrone(const TString& content) {
    yandex::maps::proto::common2::response::Response result;
    Y_ENSURE(result.ParseFromString(content));
    Y_ENSURE(result.has_reply());
    const auto& reply = result.reply();
    if (!reply.geo_object_size()) {
        return {};
    }
    const auto& object = reply.geo_object(0);
    Y_ENSURE(object.geometry_size() == 1);
    const auto& geometry = object.geometry(0);
    Y_ENSURE(geometry.has_multipolygon());
    auto multipolygon = maps::geolib3::proto::decode(geometry.multipolygon());
    Y_ENSURE(multipolygon.polygonsNumber() > 0);
    Y_ENSURE(multipolygon.area() > 0);
    auto polygon = multipolygon.polygonAt(0);
    auto ring = polygon.exteriorRing();
    Y_ENSURE(ring.pointsNumber() > 2);
    auto polyline = ring.toPolyline();
    auto convexPolygon = maps::geolib3::convexHull(polyline.points());
    auto convexRing = convexPolygon.exteriorRing();
    Y_ENSURE(convexRing.pointsNumber() > 2);
    TVector<TGeoCoord> isochrone;
    isochrone.reserve(convexRing.pointsNumber() + 1);
    for (size_t i = 0; i < convexRing.pointsNumber(); i++) {
        const auto& point = convexRing.pointAt(i);
        isochrone.emplace_back(point.x(), point.y());
    }
    const auto& point = convexRing.pointAt(0);
    isochrone.emplace_back(point.x(), point.y());
    return isochrone;
}

TRouterBase::TRouterBase(
    const TString& hostUrl,
    TMaybe<TTvmAuth> tvmAuth,
    const TDuration& timeout
)
    : Client(new NNeh::THttpClient(hostUrl))
    , RequestTimeout(timeout)
    , TvmAuth(tvmAuth)
{
}

NThreading::TFuture<TString> TRouterBase::RequestAsync(
    const TString& handler,
    const TString& signals,
    TConstArrayRef<TGeoCoord> points,
    bool avoidTolls, /*= false*/
    ui16 results, /*= 1*/
    TMaybe<TString> reqId, /*= Nothing()*/
    TMaybe<TString> vehicleType /*= Nothing()*/
) const {
    Y_ENSURE_BT(points.size() >= 2);
    Y_ENSURE_BT(results >= 1);
    TCgiParameters params;
    params.InsertUnescaped("rll", BuildRll(points));
    params.InsertUnescaped("mode", "best");
    params.InsertUnescaped("results", ToString(results));
    if (avoidTolls) {
        params.InsertUnescaped("avoid", "tolls");
    }
    if (reqId) {
        params.InsertUnescaped("reqid", *reqId);
    }
    if (vehicleType) {
        params.InsertUnescaped("vehicle_type", *vehicleType);
    }
    NNeh::THttpRequest request;
    request.SetUri(handler);
    request.SetCgiData(params.Print());
    request.AddHeader("Accept", "application/x-protobuf");
    if (TvmAuth) {
        TvmAuth->UpdateRequest(request);
    }
    return SendRequest(request, signals);
}

NThreading::TFuture<TString> TRouterBase::RequestAsync(
    const TString& handler,
    const TString& signals,
    TConstArrayRef<TGeoCoord> from,
    TConstArrayRef<TGeoCoord> to,
    bool avoidTolls, /*= false*/
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    Y_ENSURE_BT(!from.empty());
    Y_ENSURE_BT(!to.empty());
    Y_ENSURE_BT(from.size() <= 100);
    Y_ENSURE_BT(to.size() <= 100);
    Y_ENSURE_BT(from.size() * to.size() <= 100);
    TCgiParameters params;
    params.InsertUnescaped("srcll", BuildRll(from));
    params.InsertUnescaped("dstll", BuildRll(to));
    if (avoidTolls) {
        params.InsertUnescaped("avoid", "tolls");
    }
    if (reqId) {
        params.InsertUnescaped("reqid", *reqId);
    }
    NNeh::THttpRequest request;
    request.SetUri(handler);
    request.SetCgiData(params.Print());
    request.AddHeader("Accept", "application/x-protobuf");
    if (TvmAuth) {
        TvmAuth->UpdateRequest(request);
    }
    return SendRequest(request, signals);
}

NThreading::TFuture<TString> TRouterBase::RequestAsync(
    const TString& handler,
    const TString& signals,
    TConstArrayRef<TGeoCoord> points,
    const TDuration& time,
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    Y_ENSURE_BT(points.size() == 1);
    TCgiParameters params;
    params.InsertUnescaped("rll", BuildRll(points));
    params.InsertUnescaped("timeleft", ToString(time.Seconds()));
    if (reqId) {
        params.InsertUnescaped("reqid", *reqId);
    }
    NNeh::THttpRequest request;
    request.SetUri(handler);
    request.SetCgiData(params.Print());
    request.AddHeader("Accept", "application/x-protobuf");
    if (TvmAuth) {
        TvmAuth->UpdateRequest(request);
    }
    return SendRequest(request, signals);
}

NThreading::TFuture<TString> TRouterBase::SendRequest(
    const NNeh::THttpRequest& request,
    const TString& signals
) const {
    TInstant start = Now();
    NThreading::TFuture<NUtil::THttpReply> reply = Yensured(Client)->SendAsync(request, start + RequestTimeout);
    TUnistatSignalsCache::SignalAdd("maps_router-" + signals, "request", 1);
    return reply.Apply([start, signals](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const TDuration duration = Now() - start;
        const NUtil::THttpReply& reply = r.GetValue();
        if (reply.IsUserError()) {
            TUnistatSignalsCache::SignalAdd("maps_router-" + signals, "response-user_error", 1);
        }
        if (reply.IsServerError()) {
            TUnistatSignalsCache::SignalAdd("maps_router-" + signals, "response-server_error", 1);
        }
        reply.EnsureSuccessfulReply();
        TUnistatSignalsCache::SignalAdd("maps_router-" + signals, "response-success", 1);
        TUnistatSignalsCache::SignalHistogram(
            "maps_router-" + signals, "response-times", duration.MilliSeconds(),
            NRTLineHistogramSignals::IntervalsRTLineReply
        );
        return reply.Content();
    });
}

TString TRouterBase::BuildRll(TConstArrayRef<TGeoCoord> points) {
    TString rll;
    for (const auto& point : points) {
        if (!rll.empty()) {
            rll.push_back('~');
        }
        rll += ToString(point.X);
        rll.push_back(',');
        rll += ToString(point.Y);
    }
    return rll;
}

NThreading::TFuture<NGraph::TRouter::TOptionalRoute> TPedestrianRouter::GetSummary(
    const TGeoCoord& from,
    const TGeoCoord& to,
    bool avoidTolls /*= false*/,
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    Y_UNUSED(avoidTolls);
    auto summary = RequestAsync("/pedestrian/v2/summary", "pedestrian_summary", {from, to}, false, 1, reqId);
    return summary.Apply([](const NThreading::TFuture<TString>& r) -> NGraph::TRouter::TOptionalRoute {
        const TString& content = r.GetValue();
        yandex::maps::proto::masstransit::summary::Summaries result;
        Y_ENSURE(result.ParseFromString(content));
        const auto& summaries = result.summaries();
        if (summaries.empty()) {
            return Nothing();
        }
        const auto& weight = summaries[0].weight();
        NGraph::TRouter::TRoute route;
        route.HasPath = true;
        route.Time = weight.time().value();
        route.Length = weight.walking_distance().value();
        route.Origin = NGraph::TRouter::EOrigin::MapsRouter;
        return MakeMaybe(std::move(route));
    });
}

TPedestrianRouter::TPedestrianRouter(
    const TString& hostUrl,
    TMaybe<TTvmAuth> tvmAuth,
    const TDuration& timeout
)
    : TBase(hostUrl, tvmAuth, timeout)
{
}

NThreading::TFuture<NGraph::TRouter::TOptionalRoute> TPedestrianRouter::GetRoute(
    const TGeoCoord& from,
    const TGeoCoord& to,
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    auto routes = RequestAsync("/pedestrian/v2/route", "pedestrian_route", {from, to}, false, 1, reqId);
    return routes.Apply([](const NThreading::TFuture<TString>& r) -> NGraph::TRouter::TOptionalRoute {
        const TString& content = r.GetValue();
        yandex::maps::proto::common2::response::Response result;
        Y_ENSURE(result.ParseFromString(content));
        Y_ENSURE(result.has_reply());
        const auto& reply = result.reply();
        if (!reply.geo_object_size()) {
            return Nothing();
        }
        const auto& object = reply.geo_object(0);
        Y_ENSURE(object.metadata_size() > 0);
        const auto& metadata = object.metadata(0);
        Y_ENSURE(metadata.HasExtension(yandex::maps::proto::masstransit::route::ROUTE_METADATA));
        const auto& extension = metadata.GetExtension(yandex::maps::proto::masstransit::route::ROUTE_METADATA);
        NGraph::TRouter::TRoute route;
        route.HasPath = true;
        const auto& weight = extension.weight();
        route.Time = weight.time().value();
        route.Length = weight.has_total_distance() ? weight.total_distance().value() : weight.walking_distance().value();
        route.Origin = NGraph::TRouter::EOrigin::MapsRouter;
        maps::geolib3::Polyline2 polyline;
        for (auto&& segment : object.geo_object()) {
            for (auto&& geometry : segment.geometry()) {
                if (!geometry.has_polyline()) {
                    continue;
                }
                polyline.extend(
                    maps::geolib3::proto::decode(geometry.polyline()),
                    maps::geolib3::MergeEqualPoints
                );
            }
        }
        NGraph::TRouter::TRouteElement element;
        for (auto&& point : polyline.points()) {
            element.Points.emplace_back(point.x(), point.y());
        }
        route.Elements.push_back(std::move(element));
        return MakeMaybe(std::move(route));
    });
}

NThreading::TFuture<NGraph::TRouter::TRoutingMatrix> TPedestrianRouter::GetMatrix(
    TConstArrayRef<TGeoCoord> from,
    TConstArrayRef<TGeoCoord> to,
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    auto matrix = RequestAsync("/pedestrian/v2/matrix", "pedestrian_matrix", from, to, false, reqId);
    auto fromSize = from.size();
    auto toSize = to.size();
    return matrix.Apply([fromSize, toSize](const NThreading::TFuture<TString>& r) {
        const TString& content = r.GetValue();
        yandex::maps::proto::masstransit_matrix::Matrix result;
        Y_ENSURE(result.ParseFromString(content));
        Y_ENSURE_BT(result.row_size() == (int)fromSize);
        NGraph::TBaseRouter::TMatrixElements elements;
        for (int i = 0; i < result.row_size(); i++) {
            const auto& row = result.row(i);
            Y_ENSURE_BT(row.element_size() == (int)toSize);
            for (int j = 0; j < row.element_size(); j++) {
                const auto& col = row.element(j);
                if (!col.has_summary()) {
                    continue;
                }
                NGraph::TBaseRouter::TMatrixElement element;
                element.FromIdx = i;
                element.ToIdx = j;
                const auto& weight = col.summary().weight();
                element.Time = weight.time().value();
                element.Length = weight.walking_distance().value();
                elements.push_back(std::move(element));
            }
        }
        return NGraph::TBaseRouter::TRoutingMatrix(
            std::move(elements),
            NGraph::TBaseRouter::TMetaInfo()
        );
    });
}

NThreading::TFuture<TVector<TGeoCoord>> TPedestrianRouter::GetIsochrone(
    const TGeoCoord& from,
    const TDuration& time,
    TMaybe<TString> reqId /*= Nothing()*/
) const {
    auto isochrone = RequestAsync("/pedestrian/v2/isochrone", "pedestrian_isochrone", {from}, time, reqId);
    return isochrone.Apply([](const NThreading::TFuture<TString>& r) -> TVector<TGeoCoord> {
        const TString& content = r.GetValue();
        return ParsePedestrianIsochrone(content);
    });
}
}
