#pragma once

#include <drive/library/cpp/auth/tvm.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/util/network/neh.h>

#include <util/generic/yexception.h>
#include <util/generic/ptr.h>

namespace NDrive {

class IRouter {
public:
    virtual NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetSummary(
        const TGeoCoord& from,
        const TGeoCoord& to,
        bool avoidTolls = false,
        TMaybe<TString> reqId = Nothing()
    ) const = 0;

    virtual NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetRoute(
        const TGeoCoord& from,
        const TGeoCoord& to,
        TMaybe<TString> reqId = Nothing()
    ) const = 0;

    virtual NThreading::TFuture<NGraph::TRouter::TRoutingMatrix> GetMatrix(
        TConstArrayRef<TGeoCoord> from,
        TConstArrayRef<TGeoCoord> to,
        TMaybe<TString> reqId = Nothing()
    ) const = 0;

    virtual NThreading::TFuture<TVector<TGeoCoord>> GetIsochrone(
        const TGeoCoord& from,
        const TDuration& time,
        TMaybe<TString> reqId = Nothing()
    ) const = 0;

    virtual ~IRouter() = default;
};

class TRouterBase {
public:
    TRouterBase(
        const TString& hostUrl,
        TMaybe<TTvmAuth> tvmAuth = Nothing(),
        const TDuration& timeout = TDuration::Seconds(20)
    );

protected:
    THolder<NNeh::THttpClient> Client;
    TDuration RequestTimeout;
    TMaybe<TTvmAuth> TvmAuth;

    static TString BuildRll(TConstArrayRef<TGeoCoord> points);

    // RequestAsync used for requesting routes and summaries.
    NThreading::TFuture<TString> RequestAsync(
        const TString& handler,
        const TString& signals,
        TConstArrayRef<TGeoCoord> points,
        bool avoidTolls = false,
        ui16 results = 1,
        TMaybe<TString> reqId = Nothing(),
        TMaybe<TString> vehicleType = Nothing()
    ) const;

    // RequestAsync used for requesting matrix.
    //
    // Requires from.size() * to.size() <= 100.
    NThreading::TFuture<TString> RequestAsync(
        const TString& handler,
        const TString& signals,
        TConstArrayRef<TGeoCoord> from,
        TConstArrayRef<TGeoCoord> to,
        bool avoidTolls = false,
        TMaybe<TString> reqId = Nothing()
    ) const;

    // RequestAsync used for requesting isochrones.
    NThreading::TFuture<TString> RequestAsync(
        const TString& handler,
        const TString& signals,
        TConstArrayRef<TGeoCoord> points,
        const TDuration& time,
        TMaybe<TString> reqId = Nothing()
    ) const;

    // SendRequest sends request.
    NThreading::TFuture<TString> SendRequest(
        const NNeh::THttpRequest& request,
        const TString& signals
    ) const;
};

class TPedestrianRouter 
    : public IRouter
    , public TRouterBase
{
private:
    using TBase = TRouterBase;

public:
    TPedestrianRouter(
        const TString& hostUrl,
        TMaybe<TTvmAuth> tvmAuth = Nothing(),
        const TDuration& timeout = TDuration::Seconds(20)
    );

    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetSummary(
        const TGeoCoord& from,
        const TGeoCoord& to,
        bool avoidTolls = false,
        TMaybe<TString> reqId = Nothing()
    ) const override;

    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetRoute(
        const TGeoCoord& from,
        const TGeoCoord& to,
        TMaybe<TString> reqId = Nothing()
    ) const override;

    NThreading::TFuture<NGraph::TRouter::TRoutingMatrix> GetMatrix(
        TConstArrayRef<TGeoCoord> from,
        TConstArrayRef<TGeoCoord> to,
        TMaybe<TString> reqId = Nothing()
    ) const override;

    NThreading::TFuture<TVector<TGeoCoord>> GetIsochrone(
        const TGeoCoord& from,
        const TDuration& time,
        TMaybe<TString> reqId = Nothing()
    ) const override;
};

class TVehicleRouter 
    : public IRouter
    , public TRouterBase
{
private:
    using TBase = TRouterBase;

public:
    TVehicleRouter(
        const TString& hostUrl,
        TMaybe<TTvmAuth> tvmAuth = Nothing(),
        const TDuration& timeout = TDuration::Seconds(20)
    );

    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetSummary(
        const TGeoCoord& from,
        const TGeoCoord& to,
        bool avoidTolls = false,
        TMaybe<TString> reqId = Nothing()
    ) const override;

    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetRoute(
        const TGeoCoord& from,
        const TGeoCoord& to,
        TMaybe<TString> reqId = Nothing()
    ) const override {
        Y_UNUSED(reqId);
        Y_UNUSED(from);
        Y_UNUSED(to);
        ythrow yexception() << "TVehicleRouter::GetRoute not implemented!";
    }

    NThreading::TFuture<NGraph::TRouter::TRoutingMatrix> GetMatrix(
        TConstArrayRef<TGeoCoord> from,
        TConstArrayRef<TGeoCoord> to,
        TMaybe<TString> reqId = Nothing()
    ) const override {
        Y_UNUSED(reqId);
        Y_UNUSED(from);
        Y_UNUSED(to);
        ythrow yexception() << "TVehicleRouter::GetMatrix not implemented!";
    };

    NThreading::TFuture<TVector<TGeoCoord>> GetIsochrone(
        const TGeoCoord& from,
        const TDuration& time,
        TMaybe<TString> reqId = Nothing()
    ) const override {
        Y_UNUSED(reqId);
        Y_UNUSED(from);
        Y_UNUSED(time);
        ythrow yexception() << "TVehicleRouter::GetIsochrone not implemented!";
    }
};

class TScooterRouter 
    : public IRouter
    , public TRouterBase
{
private:
    using TBase = TRouterBase;

public:
    TScooterRouter(
        const TString& hostUrl,
        TMaybe<TTvmAuth> tvmAuth = Nothing(),
        const TDuration& timeout = TDuration::Seconds(20)
    );

    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetSummary(
        const TGeoCoord& from,
        const TGeoCoord& to,
        bool avoidTolls = false,
        TMaybe<TString> reqId = Nothing()
    ) const override;

    NThreading::TFuture<NGraph::TRouter::TOptionalRoute> GetRoute(
        const TGeoCoord& from,
        const TGeoCoord& to,
        TMaybe<TString> reqId = Nothing()
    ) const override {
        Y_UNUSED(reqId);
        Y_UNUSED(from);
        Y_UNUSED(to);
        ythrow yexception() << "TScooterRouter::GetRoute not implemented!";
    }

    NThreading::TFuture<NGraph::TRouter::TRoutingMatrix> GetMatrix(
        TConstArrayRef<TGeoCoord> from,
        TConstArrayRef<TGeoCoord> to,
        TMaybe<TString> reqId = Nothing()
    ) const override {
        Y_UNUSED(reqId);
        Y_UNUSED(from);
        Y_UNUSED(to);
        ythrow yexception() << "TScooterRouter::GetMatrix not implemented!";
    };

    NThreading::TFuture<TVector<TGeoCoord>> GetIsochrone(
        const TGeoCoord& from,
        const TDuration& time,
        TMaybe<TString> reqId = Nothing()
    ) const override {
        Y_UNUSED(reqId);
        Y_UNUSED(from);
        Y_UNUSED(time);
        ythrow yexception() << "TScooterRouter::GetIsochrone not implemented!";
    }
};

TVector<TGeoCoord> ParsePedestrianIsochrone(const TString& content);

}
