#include "client.h"

#include <drive/library/cpp/aes/crypto.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/wait_guard/wait_guard.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/threading/future/subscription/wait_all_or_exception.h>
#include <library/cpp/mime/detect/detectmime.h>
#include <library/cpp/xml/sax/sax.h>

#include <util/datetime/base.h>
#include <util/string/join.h>


namespace {
    class TS3KeyListParser : public NXml::ISaxHandler {
    public:
        void OnStartElement(const char* name, const char** /*attrs*/) override {
            IsKey = TStringBuf(name) == "Key";
            IsTruncate = TStringBuf(name) == "IsTruncated";
            IsModificationTime = TStringBuf(name) == "LastModified";
        }

        void OnEndElement(const char* name) override {
            bool isElementEnd = TStringBuf(name) == "Contents";
            if (isElementEnd) {
                Elements.emplace_back(std::move(Key), std::move(ModificationTime));
                Key.clear();
                ModificationTime = TInstant::Zero();
            }
            IsKey = false;
            IsTruncate = false;
            IsModificationTime = false;
        }

        void OnText(const char* text, size_t len) override {
            if (IsKey) {
                Key.append(text, len);
            } else if (IsTruncate) {
                Truncated = FromString<bool>(TStringBuf(text, len));
            } else if (IsModificationTime) {
                if (!TInstant::TryParseIso8601(TStringBuf(text, len), ModificationTime)) {
                    ModificationTime = TInstant::Zero();
                }
            }
        }

        const TVector<TS3Client::TBucketElement>& GetKeys() const {
            return Elements;
        }

        bool IsPart() const {
            return Truncated;
        }

    private:
        TVector<TS3Client::TBucketElement> Elements;
        bool IsKey = false;
        bool IsTruncate = false;
        bool IsModificationTime = false;
        bool Truncated = false;
        TString Key;
        TInstant ModificationTime = TInstant::Zero();
    };

    class TS3BucketListParser : public NXml::ISaxHandler {
    public:
        void OnStartElement(const char* name, const char** /*attrs*/) override {
            IsName = TStringBuf(name) == "Name";
        }

        void OnEndElement(const char* /*name*/) override {
            if (IsName) {
                Names.emplace_back(std::move(Name));
                Name.clear();
            }
            IsName = false;
        }

        void OnText(const char* text, size_t len) override {
            if (IsName) {
                Name.append(text, len);
            }
        }

        const TVector<TString>& GetNames() const {
            return Names;
        }

    private:
        TVector<TString> Names;
        bool IsName = false;
        TString Name;
    };

    class TS3ClientCodeCallback : public  NNeh::THttpAsyncReport::ICallback {
    public:
        TS3ClientCodeCallback(ui32& code)
            : Code(code) {}

        void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override {
            CHECK_WITH_LOG(reports.size() == 1);
            const auto& result = reports.front();
            DEBUG_LOG << "Code: " << result.GetHttpCode() << Endl;
            Code = result.GetHttpCode();
        };

    private:
        ui32& Code;
    };

    class TS3ClientXmlCallback : public TS3ClientCodeCallback {
    public:
        TS3ClientXmlCallback(NXml::ISaxHandler* result, ui32& code)
            : TS3ClientCodeCallback(code)
            , ParsedResult(result) {}

        void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override {
            TS3ClientCodeCallback::OnResponse(reports);
            CHECK_WITH_LOG(reports.size() == 1);
            const auto& result = reports.front();
            if (auto report = result.GetReport()) {
                TMemoryInput input(report->data(), report->size());
                try {
                    NXml::ParseXml(&input, ParsedResult);
                } catch (const yexception& e) {
                    ERROR_LOG << e.what() << Endl;
                }
            }
        };

    private:
        NXml::ISaxHandler* ParsedResult;
    };

    class TCallbackWrapper : public NNeh::THttpAsyncReport::THandlerCallback {
        using TBase = NNeh::THttpAsyncReport::THandlerCallback;

    private:
        TAtomicSharedPtr<TS3Client::IBaseCallback> HighLevelCallback;
        const TString Path;

    public:
        TCallbackWrapper(TAtomicSharedPtr<TS3Client::IBaseCallback> highLevelCallback, const TString& path, IThreadPool& handler)
            : TBase(handler)
            , HighLevelCallback(highLevelCallback)
            , Path(path) {}

        void Process(void*) final {
            CHECK_WITH_LOG(Reports.size() == 1);
            auto result = Reports.front();
            auto code = result.GetHttpCode();
            if (code / 100 == 2) {
                HighLevelCallback->OnSuccess(Path, result);
            } else {
                ERROR_LOG << "request into MDS bucket was failed, error: " << code << " " << result.GetReportSafe() << Endl;
                HighLevelCallback->OnFailure(Path, result);
            }
        }
    };

    class ISyncCallback : public TS3Client::IBaseCallback {
    public:
        ISyncCallback(TS3Client::ERequestType type, TMessagesCollector& error, TWaitGuard& guard)
            : TS3Client::IBaseCallback(type)
            , Error(error)
            , Guard(guard)
        {
            Guard.RegisterObject();
        }

        virtual ~ISyncCallback() {
            Guard.UnRegisterObject();
        }

    protected:
        void DoOnFailure(const TString& path, const THttpReplyData<TString>& reply) override {
            TGuard<TMutex> g(Mutex);
            Error.AddMessage("mds_file", path + " " + ToString(reply.GetHttpCode()) + " " + reply.GetReportSafe());
        }

    private:
        TMessagesCollector& Error;
        TWaitGuard& Guard;
        TMutex Mutex;
    };
}

NJson::TJsonValue TS3Client::TBucketElement::GetReport() const {
    NJson::TJsonValue report;
    report.InsertValue("key", Key);
    report.InsertValue("modification_time", ModificationTime.Seconds());
    if (Owner) {
        report.InsertValue("link", Owner->GetTmpFilePath(Key));
    }
    return report;
}

const TS3ClientConfig& NS3::TBucket::GetConfig() const {
    return Config;
}

bool NS3::TBucket::GetPartialList(const TString& prefix, const TString& marker, TVector<TS3Client::TBucketElement>& resultList, ui32& code) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    TS3KeyListParser parsedResult;
    auto g = Impl->SendPtr(CreateGetKeysRequest(prefix, marker), deadline, new TS3ClientXmlCallback(&parsedResult, code));
    g.Wait();
    resultList = parsedResult.GetKeys();
    for (auto& key : resultList) {
        key.Owner = this;
    }
    return parsedResult.IsPart();
}

ui32 NS3::TBucket::GetKeys(const TString& prefix, TVector<TS3Client::TBucketElement>& resultKeys) const {
    resultKeys.clear();
    TString marker;
    TVector<TBucketElement> keyList;
    ui32 code;
    TUnistatSignalsCache::SignalAdd("drive-frontend-mds-requests", ::ToString(ERequestType::GetKeys), 1);
    while (GetPartialList(prefix, marker, keyList, code)) {
        if (code != 200) {
            INFO_LOG << "Partial list request failed " << code << Endl;
            TUnistatSignalsCache::SignalAdd("drive-frontend-mds-reply-codes", ::ToString(ERequestType::GetKeys) + "-" + ::ToString(code), 1);
            TUnistatSignalsCache::SignalAdd("drive-frontend-mds-errors", ::ToString(ERequestType::GetKeys), 1);
            return code;
        }
        resultKeys.insert(resultKeys.end(), keyList.begin(), keyList.end());
        marker = keyList.back().Key;
    }
    TUnistatSignalsCache::SignalAdd("drive-frontend-mds-reply-codes", ::ToString(ERequestType::GetKeys) + "-" + ::ToString(code), 1);
    resultKeys.insert(resultKeys.end(), keyList.begin(), keyList.end());
    return code;
}

NThreading::TFuture<NS3::TBucketElementList> NS3::TBucket::GetKeys(const TString& prefix, const TString& marker) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->SendAsync(CreateGetKeysRequest(prefix, marker), deadline).Apply([] (const auto& result) -> NThreading::TFuture<TBucketElementList> {
        const auto& report = result.GetValue();
        if (!report.IsSuccessReply()) {
            return NThreading::TExceptionFuture() << "NS3::TBucket::GetKeys: request failed: " << report.Code() << ", content: " << report.Content() << ", error: " << report.ErrorMessage();
        }
        const TString content = result.GetValue().Content();
        if (content) {
            TMemoryInput input(content.data(), content.size());
            TS3KeyListParser parsedResult;
            NXml::ParseXml(&input, &parsedResult);
            return NThreading::MakeFuture(TBucketElementList{ parsedResult.GetKeys(), parsedResult.IsPart() });
        }
        return NThreading::MakeFuture(TBucketElementList());
    });
}

void NS3::TBucket::GetFile(const TString& key, TAtomicSharedPtr<TS3Client::IBaseCallback> callback) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    auto g = Impl->Send(CreateGetFileRequest(key), deadline, new TCallbackWrapper(callback, key, RepliesHandler));
}

TString NS3::DetectContentType(TStringBuf content) {
    TMimeDetector detector;
    detector.Detect(content.data(), content.size());
    return detector.Mime() != MIME_UNKNOWN ? strByMime(detector.Mime()) : "";
}

NThreading::TFuture<NS3::TFile> NS3::TBucket::GetFile(const TString& key, const TString& defContentType, const bool skipTypeDetect) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->SendAsync(CreateGetFileRequest(key), deadline).Apply([key, defContentType, skipTypeDetect] (const NThreading::TFuture<NUtil::THttpReply>& reply) {
        if (!reply.HasValue()) {
            ythrow yexception() << "No reply for file(" << key << "):" << NThreading::GetExceptionMessage(reply);
        }
        auto report = reply.GetValue();
        if (!report.IsSuccessReply()) {
            ythrow yexception() << "Error getting file(" << key << "): " << report.GetDebugReply();
        }
        if (skipTypeDetect) {
            return NThreading::MakeFuture(TFile(key, report.Content(), defContentType));
        }
        TString contentType = DetectContentType(report.Content());
        if (contentType.empty()) {
            contentType = defContentType;
        }
        if (contentType.empty()) {
            for (const auto& header : report.GetHeaders()) {
                if (header.Name() == "Content-Type") {
                    contentType = header.Value();
                    break;
                }
            }
        }
        return NThreading::MakeFuture(TFile(key, report.Content(), contentType));
    });
}

void NS3::TBucket::PutKey(const TString& key, const TString& data, TAtomicSharedPtr<TS3Client::IBaseCallback> callback, const TString& contentType) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    auto g = Impl->Send(CreatePutKeyRequest(key, data, contentType), deadline, new TCallbackWrapper(callback, key, RepliesHandler));
}

void NS3::TBucket::DeleteKey(const TString& key, TAtomicSharedPtr<TS3Client::IBaseCallback> callback) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    auto g = Impl->Send(CreateDeleteKeyRequest(key), deadline, new TCallbackWrapper(callback, key, RepliesHandler));
}

NThreading::TFuture<void> NS3::TBucket::CheckReply(const NThreading::TFuture<NUtil::THttpReply>& reply) {
    return reply.Apply([] (const auto& result) -> NThreading::TFuture<void> {
        if (result.HasException() || !result.HasValue()) {
            return NThreading::TExceptionFuture() << "NS3::TBucket::CheckReply: exception: " << NThreading::GetExceptionMessage(result);
        }
        const auto& report = result.GetValue();
        if (!report.IsSuccessReply()) {
            return NThreading::TExceptionFuture() << "NS3::TBucket::CheckReply: request failed: " << report.Code() << ", content: " << report.Content() << ", error: " << report.ErrorMessage();
        }
        return NThreading::MakeFuture();
    });
}

NThreading::TFuture<NUtil::THttpReply> NS3::TBucket::PutKey(const TString& key, const TString& data, const TString& contentType) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->SendAsync(CreatePutKeyRequest(key, data, contentType), deadline);
}

NThreading::TFuture<NUtil::THttpReply> TS3Client::TBucket::DeleteKey(const TString& key) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->SendAsync(CreateDeleteKeyRequest(key), deadline);
}

TMaybe<bool> NS3::TBucket::HasFile(const TString& key) const {
    TVector<TS3Client::TBucketElement> images;
    if (GetKeys(key, images) != HTTP_OK) {
        return Nothing();
    }
    if (images.size() == 1 && images.front().Key == key) {
        return true;
    }
    return false;
}

NThreading::TFuture<bool> NS3::TBucket::HasKey(const TString& key) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    return Impl->SendAsync(CreateGetKeysRequest(key, ""), deadline).Apply([key](const auto& result) -> NThreading::TFuture<bool> {
        const auto& report = result.GetValue();
        if (!report.IsSuccessReply()) {
            return NThreading::TExceptionFuture() << "NS3::TBucket::HasKey: request failed: " << report.Code() << ", content: " << report.Content() << ", error: " << report.ErrorMessage();
        }
        TS3KeyListParser parsedResult;
        TMemoryInput input(report.Content().data(), report.Content().size());
        NXml::ParseXml(&input, &parsedResult);
        return NThreading::MakeFuture(AnyOf(parsedResult.GetKeys().begin(), parsedResult.GetKeys().end(), [&key](const auto& item) { return item.Key == key; }));
    });
}

class TCodeSyncCallback : public ISyncCallback {
public:
    TCodeSyncCallback(TS3Client::ERequestType type, ui32& code, TMessagesCollector& error, TWaitGuard& guard)
        : ISyncCallback(type, error, guard)
        , Code(code)
    {}

protected:
    void DoOnSuccess(const TString& /*path*/, const THttpReplyData<TString>& reply) override {
        Code = reply.GetHttpCode();
    }

    void DoOnFailure(const TString& path, const THttpReplyData<TString>& reply) override {
        ISyncCallback::DoOnFailure(path, reply);
        Code = reply.GetHttpCode();
    }

private:
    ui32& Code;
};

class TGetFileSyncCallback : public TCodeSyncCallback {
    using Base = TCodeSyncCallback;
public:
    TGetFileSyncCallback(TString& file, ui32&code, TMessagesCollector& error, TWaitGuard& guard)
        : Base(TS3Client::ERequestType::GetFile, code, error, guard)
        , File(file)
    {}

private:
    void DoOnSuccess(const TString& path, const THttpReplyData<TString>& reply) override {
        Base::DoOnSuccess(path, reply);
        File = reply.GetReportSafe();
    }

private:
    TString& File;
};

ui32 NS3::TBucket::GetFile(const TString& key, TString& file, TMessagesCollector& errors) const {
    ui32 code = HTTP_INTERNAL_SERVER_ERROR;
    {
        TWaitGuard waitGuard;
        GetFile(key, new TGetFileSyncCallback(file, code, errors, waitGuard));
    }
    return code;
}

ui32 NS3::TBucket::PutKey(const TString& key, const TString& data, TMessagesCollector& errors, const TString& contentType) const {
    ui32 code = HTTP_INTERNAL_SERVER_ERROR;
    {
        TWaitGuard waitGuard;
        PutKey(key, data, new TCodeSyncCallback(ERequestType::PutKey, code, errors, waitGuard), contentType);
    }
    return code;
}

ui32 NS3::TBucket::DeleteKey(const TString& key, TMessagesCollector& errors) const {
    ui32 code = HTTP_INTERNAL_SERVER_ERROR;
    {
        TWaitGuard waitGuard;
        DeleteKey(key, new TCodeSyncCallback(ERequestType::DeleteKey, code, errors, waitGuard));
    }
    return code;
}

NNeh::THttpRequest NS3::TBucket::CreateBaseRequest(const TString& key, const TString& method, const TString& data, const TString& contentType) const {
    auto now = TInstant::Now();
    tm tM;
    TString date = Strftime("%a, %d %b %Y %H:%M:%S GMT", now.GmTime(&tM));

    TString authStr =
        TStringBuilder()
        << method << Endl
        << Endl // content-md5
        << contentType << Endl // content-type
        << date << Endl // date
        << "/" << BucketName << "/" << key; // auth path

    return Config.CreateCommonRequest(authStr, date, method, data);
}

NNeh::THttpRequest NS3::TBucket::CreateGetKeysRequest(const TString& prefix, const TString& marker) const {
    auto req = CreateBaseRequest("", "GET", "");

    TCgiParameters params;
    params.emplace("prefix", prefix);
    if (!!marker) {
        params.emplace("marker", marker);
    }
    req.SetCgiData(params.Print());

    return req;
}

NNeh::THttpRequest NS3::TBucket::CreateGetFileRequest(const TString& key) const {
    return CreateBaseRequest(key, "GET", "").SetUri(key);
}

NNeh::THttpRequest NS3::TBucket::CreatePutKeyRequest(const TString& key, const TString& data, const TString& contentType) const {
    auto req = CreateBaseRequest(key, "PUT", data, contentType).SetUri(key);
    if (contentType) {
        req.AddHeader("Content-Type", contentType);
    }
    return req;
}

TString NS3::TBucket::GetTmpFilePath(const TString& path) const {
    return Config.GetTmpFilePath(BucketName, path);
}

NNeh::THttpRequest NS3::TBucket::CreateDeleteKeyRequest(const TString& key) const {
    return CreateBaseRequest(key, "DELETE", "").SetUri(key);
}

TS3Client::TBucket::TEncryptConfPtr FindEncryptConf(const std::multimap<TString, TS3Client::TBucket::TEncryptConfPtr>& map, const TString& bucketName, const TString& encryptionName) {
    const auto equalRange = map.equal_range(bucketName);
    for (auto it = equalRange.first; it != equalRange.second; ++it) {
        if (it->second->GetName() == encryptionName) {
            return it->second;
        }
    }
    return nullptr;
}

bool TS3Client::TBucket::HasEncryption(const TString& encryptionName) const {
    return !!FindEncryptConf(Config.GetEncryptedBuckets(), GetBucketName(), encryptionName);
}

TString TS3Client::TBucket::GetDefEncryptionName() const {
    const auto equalRange = Config.GetEncryptedBuckets().equal_range(GetBucketName());
    for (auto it = equalRange.first; it != equalRange.second; ++it) {
        if (it->second->IsDefault()) {
            return it->second->GetName();
        }
    }
    return "";
}

TCryptoGCMProcessor::EMode GetCryptoMode(const TS3ClientConfig::EBucketEncryptionMode mode) {
    switch (mode) {
        case TS3ClientConfig::EBucketEncryptionMode::AES_GCM_128:
            return TCryptoGCMProcessor::EMode::GCM_128;
        case TS3ClientConfig::EBucketEncryptionMode::AES_GCM_256:
            return TCryptoGCMProcessor::EMode::GCM_256;
    }
}

NThreading::TFuture<TVector<NUtil::THttpReply>> TS3Client::TBucket::PutEncrypted(const TString& key, const TString& data, const TString& encryptionName) const {
    auto bucketConfig = FindEncryptConf(Config.GetEncryptedBuckets(), GetBucketName(), encryptionName);
    if (!bucketConfig) {
        return NThreading::TExceptionFuture() << "No encryption data for bucket(" << GetBucketName() << ") with encryption(" << encryptionName << ")";
    }
    TCryptoGCMProcessor processor(GetCryptoMode(bucketConfig->GetEncryptionMode()), bucketConfig->GetEncryptionKey());
    auto encrypted = processor.Encrypt(data, key);
    if (!encrypted) {
        return NThreading::TExceptionFuture() << "Fail to encrypt file(" << key << ")";
    }
    auto tagFuture = PutKey(key + "_tag", encrypted->Tag);
    auto filefuture = PutKey(key, encrypted->Data);
    return NThreading::NWait::WaitAllOrException(tagFuture, filefuture).Apply([tagFuture = std::move(tagFuture), fileFuture = std::move(filefuture)](const NThreading::TFuture<void>&) mutable {
        TVector<NUtil::THttpReply> result;
        result.emplace_back(tagFuture.ExtractValue());
        result.emplace_back(fileFuture.ExtractValue());
        return NThreading::MakeFuture(result);
    });
}

NThreading::TFuture<NS3::TFile> TS3Client::TBucket::GetDecrypted(const TString& key, const TString& encryptionName) const {
    auto bucketConfig = FindEncryptConf(Config.GetEncryptedBuckets(), GetBucketName(), encryptionName);
    if (!bucketConfig) {
        return NThreading::TExceptionFuture() << "No encryption data for bucket(" << GetBucketName() << ") with encryption(" << encryptionName << ")";
    }
    auto tagFuture = GetFile(key + "_tag", /* defContentType = */ "", /* skipTypeDetect = */ true);
    auto filefuture = GetFile(key, /* defContentType = */ "", /* skipTypeDetect = */ true);
    TCryptoGCMProcessor processor(GetCryptoMode(bucketConfig->GetEncryptionMode()), bucketConfig->GetEncryptionKey());
    return NThreading::NWait::WaitAllOrException(tagFuture, filefuture).Apply([tagFuture = std::move(tagFuture), fileFuture = std::move(filefuture), cipher = std::move(processor), key](const NThreading::TFuture<void>&) {
        auto result = cipher.Decrypt(fileFuture.GetValue().GetContent(), key, tagFuture.GetValue().GetContent());
        if (!result) {
            ythrow yexception() << "Fail to decode file(" << key << ")";
        }
        return NThreading::MakeFuture(TFile(key, *result, DetectContentType(*result)));
    });
}


class TInitTask : public NBus::NPrivate::IScheduleItem {
public:
    TInitTask(TS3Client& client, const TInstant instantSchedule)
        : NBus::NPrivate::IScheduleItem(instantSchedule)
        , Client(client)
    {}

    void Do() override {
        if (!Client.Init(true)) {
            ERROR_LOG << "Can't initialize mds client" << Endl;
        }
    }

private:
    TS3Client& Client;
};

TS3Client::TS3Client(const TS3ClientConfig& config)
    : Config(config)
    , RepliesHandler(IThreadPool::TParams()
        .SetThreadName("s3_client")
    )
    , Impl(Config, "default")
{
    RepliesHandler.Start(Config.GetHandlerThreads());
}

TS3Client::~TS3Client() {
    ActiveFlag = false;
    RepliesHandler.Stop();
    Scheduler.Stop();
}

bool TS3Client::Init(const bool reinitialize) {
    if (IsInitialized) {
        return true;
    }
    TVector<TString> buckets;
    ui32 code = GetBucketList(buckets);
    if (code != HTTP_OK) {
        ERROR_LOG << "GetBucketList request finished with code: " << code << Endl;
        if (reinitialize) {
            Scheduler.Schedule(new TInitTask(*this, Now() + Config.GetInitPeriod()));
        }
        return false;
    }

    INFO_LOG << "MDS buckets: " << JoinSeq(", ", buckets) << Endl;
    {
        TWriteGuard wg(Lock);
        if (IsInitialized) {
            return true;
        }
        if (Buckets.empty()) {
            for (auto&& bucketName : buckets) {
                Buckets.emplace(bucketName, MakeHolder<TBucket>(Config.GetBucketConfig(bucketName), bucketName, RepliesHandler));
            }
        }
    }

    IsInitialized = true;
    return true;
}

const TS3Client::TBucket* TS3Client::GetBucket(const TString& bucketName) const {
    TReadGuard rg(Lock);
    auto it = Buckets.find(bucketName);
    if (it != Buckets.end()) {
        return it->second.Get();
    }
    return nullptr;
}

bool TS3Client::UploadBlob(const TString& bucketName, const TString& key, TBlob data, TMessagesCollector& errors) const {
    auto bucket = GetBucket(bucketName);
    if (!bucket) {
        errors.AddMessage("mds_client", "bucket doesn't exist");
        return false;
    }
    const ui32 code = bucket->PutKey(key, TString(data.AsCharPtr(), data.Size()), errors);
    if (code != HTTP_OK) {
        errors.AddMessage("mds_client", TStringBuilder() << "http code " << code);
        return false;
    }
    return true;
}

NNeh::THttpRequest TS3Client::CreateBucketListRequest() const {
    auto now = TInstant::Now();
    tm tM;
    TString date = Strftime("%a, %d %b %Y %H:%M:%S GMT", now.GmTime(&tM));

    TString authStr =
        TStringBuilder()
        << "GET" << Endl
        << Endl // content-md5
        << Endl // content-type
        << date << Endl // date
        << "/"; // auth path

    return Config.CreateCommonRequest(authStr, date);
}

ui32 TS3Client::GetBucketList(TVector<TString>& buckets) const {
    const TInstant deadline = Now() + Config.GetRequestTimeout();
    TS3BucketListParser result;
    ui32 code = 500;
    TUnistatSignalsCache::SignalAdd("drive-frontend-mds-requests", ::ToString(ERequestType::GetBucketsList), 1);
    auto g = Impl->SendPtr(CreateBucketListRequest(), deadline, new TS3ClientXmlCallback(&result, code));
    g.Wait();
    if (code == 200) {
        buckets = result.GetNames();
    } else {
        TUnistatSignalsCache::SignalAdd("drive-frontend-mds-errors", ::ToString(ERequestType::GetBucketsList), 1);
    }
    TUnistatSignalsCache::SignalAdd("drive-frontend-mds-reply-codes", ::ToString(ERequestType::GetBucketsList) + ::ToString(code), 1);
    return code;
}

TString TS3Client::GetTmpFilePath(const TString& bucketName, const TString& path) const {
    return Config.GetTmpFilePath(bucketName, path);
}

bool TS3Client::ParseMdsLink(const TString& resourceLink, TMessagesCollector& errors, TString& bucket, TString& filePath) {
    auto pos = resourceLink.find("://");
    if (pos == TString::npos) {
        errors.AddMessage("MediaStorage", "bad resource link: missing ://");
        return false;
    }
    pos += 3;
    TString path = resourceLink.substr(pos, resourceLink.size() - pos);
    if (path.StartsWith("s3")) {
        auto posBucket = path.find('/');
        if (posBucket == TString::npos) {
            errors.AddMessage("MediaStorage", "bad resource link: no bucket found");
            return false;
        }
        auto posFilePath = path.find('/', posBucket + 1);
        if (posFilePath == TString::npos) {
            errors.AddMessage("MediaStorage", "bad resource link: no path found");
            return false;
        }
        bucket = path.substr(posBucket + 1, posFilePath - posBucket - 1);
        filePath = path.substr(posFilePath + 1, path.size() - posFilePath - 1);
    } else {
        auto posBucketEnd = path.find(".s3");
        if (posBucketEnd == TString::npos) {
            errors.AddMessage("MediaStorage", "bad resource link: no bucket found");
            return false;
        }
        bucket = path.substr(0, posBucketEnd);
        auto posFilePath = path.find('/');
        if (posFilePath == TString::npos) {
            errors.AddMessage("MediaStorage", "bad resource link: no path found");
            return false;
        }
        filePath = path.substr(posFilePath + 1, path.size() - posFilePath - 1);
    }
    return true;
}
