#pragma once

#include "config.h"

#include <library/cpp/messagebus/scheduler/scheduler.h>

#include <rtline/library/deprecated/async_impl/async_impl.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/stream/file.h>
#include <util/stream/fwd.h>
#include <util/stream/mem.h>
#include <util/string/builder.h>


namespace NS3 {
    enum class ERequestType {
        GetKeys /* "get_keys" */,
        PutKey /* "put_key" */,
        DeleteKey /* "delete_key" */,
        GetFile /* "get_file" */,
        GetBucketsList /* "get_buckets" */,
    };

    class TBucket;

    struct TBucketElement {
        TString Key;
        TInstant ModificationTime;
        const TBucket* Owner;
        TBucketElement(const TString& key, TInstant time, const TBucket* owner = nullptr)
            : Key(key)
            , ModificationTime(time)
            , Owner(owner)
        {}
        NJson::TJsonValue GetReport() const;
    };

    struct TBucketElementList {
        TVector<TBucketElement> List;
        bool HasMore = false;
    };

    class IBaseCallback {
    public:
        IBaseCallback(ERequestType type)
            : Type(type)
        {
            TUnistatSignalsCache::SignalAdd("drive-frontend-mds-requests", ::ToString(Type), 1);
        }

        virtual void OnSuccess(const TString& path, const THttpReplyData<TString>& reply) {
            TUnistatSignalsCache::SignalAdd("drive-frontend-mds-reply-codes", ::ToString(Type) + "-" + ::ToString(reply.GetHttpCode()), 1);
            DoOnSuccess(path, reply);
        }
        virtual void OnFailure(const TString& path, const THttpReplyData<TString>& reply) {
            TUnistatSignalsCache::SignalAdd("drive-frontend-mds-reply-codes", ::ToString(Type) + "-" + ::ToString(reply.GetHttpCode()), 1);
            TUnistatSignalsCache::SignalAdd("drive-frontend-mds-errors", ::ToString(Type), 1);
            DoOnFailure(path, reply);
        }
        virtual ~IBaseCallback() = default;

    private:
        virtual void DoOnSuccess(const TString& path, const THttpReplyData<TString>& reply) = 0;
        virtual void DoOnFailure(const TString& path, const THttpReplyData<TString>& reply) = 0;

    private:
        ERequestType Type;
    };

    class TFile {
    public:
        TFile(const TString& key, const TString& content, const TString& contentType)
            : Key(key)
            , Content(content)
            , ContentType(contentType)
        {
        }

        TFile(const TString& key, TString&& content, const TString& contentType)
            : Key(key)
            , Content(std::move(content))
            , ContentType(contentType)
        {
        }

    private:
        R_FIELD(TString, Key);
        R_FIELD(TString, Content);
        R_FIELD(TString, ContentType);
    };

    class TBucket {
    public:
        using TEncryptConfPtr = TS3ClientConfig::TEncryptedBucket::TPtr;

    public:
        TBucket(const TS3ClientConfig& config, const TString& bucketName, IThreadPool& repliesHandler)
            : Config(config)
            , Impl(Config, bucketName + "_mds_api")
            , BucketName(bucketName)
            , RepliesHandler(repliesHandler)
        {}

        ui32 GetKeys(const TString& prefix, TVector<TBucketElement>& keys) const;
        NThreading::TFuture<TBucketElementList> GetKeys(const TString& prefix, const TString& marker = "") const;
        ui32 GetFile(const TString& key, TString& file, TMessagesCollector& errors) const;
        void GetFile(const TString& key, TAtomicSharedPtr<IBaseCallback> callback) const;
        NThreading::TFuture<TFile> GetFile(const TString& key, const TString& defContentType = "", const bool skipTypeDetect = false) const;
        ui32 PutKey(const TString& key, const TString& data, TMessagesCollector& errors, const TString& contentType = "") const;
        void PutKey(const TString& key, const TString& data, TAtomicSharedPtr<IBaseCallback> callback, const TString& contentType = "") const;
        NThreading::TFuture<NUtil::THttpReply> PutKey(const TString& key, const TString& data, const TString& contentType = "") const;
        ui32 DeleteKey(const TString& key, TMessagesCollector& errors) const;
        void DeleteKey(const TString& key, TAtomicSharedPtr<IBaseCallback> callback) const;
        NThreading::TFuture<NUtil::THttpReply> DeleteKey(const TString& key) const;
        TMaybe<bool> HasFile(const TString& key) const;
        NThreading::TFuture<bool> HasKey(const TString& key) const;
        static NThreading::TFuture<void> CheckReply(const NThreading::TFuture<NUtil::THttpReply>& reply);

        TString GetTmpFilePath(const TString& path) const;

        const TString& GetBucketName() const {
            return BucketName;
        }

        bool HasEncryption(const TString& encryptionName) const;
        TString GetDefEncryptionName() const;
        NThreading::TFuture<TVector<NUtil::THttpReply>> PutEncrypted(const TString& key, const TString& data, const TString& encryptionName) const;
        NThreading::TFuture<TFile> GetDecrypted(const TString& key, const TString& encryptionName) const;

        const TS3ClientConfig& GetConfig() const;

    private:
        bool GetPartialList(const TString& prefix, const TString& marker, TVector<TBucketElement>& resultList, ui32& code) const;

        NNeh::THttpRequest CreateBaseRequest(const TString& key, const TString& method, const TString& data, const TString& contentType = "") const;
        NNeh::THttpRequest CreateGetKeysRequest(const TString& prefix, const TString& marker) const;
        NNeh::THttpRequest CreateGetFileRequest(const TString& key) const;
        NNeh::THttpRequest CreatePutKeyRequest(const TString& key, const TString& data, const TString& contentType = "") const;
        NNeh::THttpRequest CreateDeleteKeyRequest(const TString& key) const;

    private:
        TS3ClientConfig Config;
        TAsyncApiImpl Impl;
        TString BucketName;
        IThreadPool& RepliesHandler;
    };

    TString DetectContentType(TStringBuf content);
}

class TS3Client {
public:
    using TBucket = NS3::TBucket;
    using TBucketElement = NS3::TBucketElement;
    using ERequestType = NS3::ERequestType;
    using IBaseCallback = NS3::IBaseCallback;

public:
    TS3Client(const TS3ClientConfig& config);
    ~TS3Client();

    bool Init(const bool reinitialize = false);

    const TS3ClientConfig& GetConfig() const {
        return Config;
    }

    bool UploadBlob(const TString& bucket, const TString& key, TBlob data, TMessagesCollector& errors) const;

    ui32 GetBucketList(TVector<TString>& buckets) const;
    const TBucket* GetBucket(const TString& name) const;

    TString GetTmpFilePath(const TString& bucketName, const TString& path) const;

    static bool ParseMdsLink(const TString& resourceLink, TMessagesCollector& errors, TString& bucket, TString& filePath);

private:
    NNeh::THttpRequest CreateBucketListRequest() const;

private:
    TS3ClientConfig Config;
    TMap<TString, THolder<TBucket>> Buckets;
    mutable TThreadPool RepliesHandler;
    TAsyncApiImpl Impl;
    TRWMutex Lock;
    bool IsInitialized = false;
    bool ActiveFlag = true;

    NBus::NPrivate::TScheduler Scheduler;
};
