#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>


Y_UNIT_TEST_SUITE(MdsSuite) {
    TS3ClientConfig GetClientConfig(const bool needEncryption = false) {
        TString key = GetEnv("AccessKeyId");
        UNIT_ASSERT(key);
        TString secret = GetEnv("AccessSecretKey");
        UNIT_ASSERT(secret);
        auto config = TStringBuilder()
            << "Host: " << GetEnv("ApiHost", "s3.mdst.yandex.net") << Endl
            << "AccessKeyId: " << key << Endl
            << "AccessSecretKey: " << secret << Endl
            << "RequestTimeout : 60s" << Endl;
        const TString port = GetEnv("Port");
        if (port == "443") {
            config
                << "Port: 443" << Endl
                << "Https: true" << Endl;
        } else if (port) {
            config << "Port: " << port << Endl;
        } else {
            config << "Port: 80" << Endl;
        }
        if (needEncryption) {
            auto bucket = GetEnv("EncryptedBucket", "carsharing");
            config << "<EncryptedBuckets>" << Endl
                << "    <test_encryption>" << Endl
                << "        Bucket: " << bucket << Endl
                << "        EncryptionKey: 00000000000000000000000000000000" << Endl
                << "        EncryptionMode: " << TS3ClientConfig::EBucketEncryptionMode::AES_GCM_128 << Endl
                << "    </test_encryption>" << Endl
                << "</EncryptedBuckets>" << Endl;
        }
        INFO_LOG << config << Endl;
        return TS3ClientConfig::ParseFromString(config);
    }

    Y_UNIT_TEST(SimpleOperations) {
        TS3Client mdsClient(GetClientConfig());
        UNIT_ASSERT(mdsClient.Init());
        auto bucket = mdsClient.GetBucket(GetEnv("Bucket", "carsharing"));
        UNIT_ASSERT(bucket);
        ui32 sizeBeforeStart = 0;
        {
            TVector<TS3Client::TBucketElement> keys;
            UNIT_ASSERT_VALUES_EQUAL(bucket->GetKeys("", keys) / 100, 2);
            sizeBeforeStart = keys.size();
        }
        const TString docName = "sample_" + ToString(TInstant::Now().MicroSeconds()) + ".txt";
        const TString doc = "Test string for " + docName;
        TMessagesCollector errors;
        UNIT_ASSERT_VALUES_EQUAL_C(bucket->PutKey(docName, doc, errors) / 100, 2, errors.GetStringReport());
        {
            TVector<TS3Client::TBucketElement> keys;
            const auto startTime = Now();
            do {
                UNIT_ASSERT_VALUES_EQUAL(bucket->GetKeys("", keys) / 100, 2);
                if (keys.size() == sizeBeforeStart + 1) {
                    break;
                }
                Sleep(TDuration::Seconds(5));
            } while (Now() - startTime < TDuration::Minutes(2));
            UNIT_ASSERT_VALUES_EQUAL(keys.size(), sizeBeforeStart + 1);
        }
        {
            auto future = bucket->HasKey(docName);
            UNIT_ASSERT(future.Wait(mdsClient.GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            UNIT_ASSERT(future.GetValue());
        }
        {
            TString file;
            UNIT_ASSERT_VALUES_EQUAL_C(bucket->GetFile(docName, file, errors) / 100, 2, errors.GetStringReport());
            UNIT_ASSERT_VALUES_EQUAL(file, doc);
        }
        UNIT_ASSERT_VALUES_EQUAL_C(bucket->DeleteKey(docName, errors) / 100, 2, errors.GetStringReport());
        {
            TVector<TS3Client::TBucketElement> keys;
            const auto startTime = Now();
            do {
                UNIT_ASSERT_VALUES_EQUAL(bucket->GetKeys("", keys) / 100, 2);
                if (keys.size() == sizeBeforeStart) {
                    break;
                }
                Sleep(TDuration::Seconds(5));
            } while (Now() - startTime < TDuration::Minutes(2));
            UNIT_ASSERT_VALUES_EQUAL(keys.size(), sizeBeforeStart);
        }
    }

    Y_UNIT_TEST(EncryptedOperations) {
        TS3Client mdsClient(GetClientConfig(/* needEncryption = */ true));
        UNIT_ASSERT(mdsClient.Init());
        auto bucket = mdsClient.GetBucket(GetEnv("EncryptedBucket", "carsharing"));
        UNIT_ASSERT(bucket);
        const TString encryptionName = "test_encryption";
        UNIT_ASSERT(bucket->HasEncryption(encryptionName));
        const TString docName = "sample_" + ToString(TInstant::Now().MicroSeconds()) + ".txt";
        const TString doc = "Test string for " + docName;
        {
            auto future = bucket->PutEncrypted(docName, doc, encryptionName);
            UNIT_ASSERT(future.Wait(mdsClient.GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto report = future.GetValue();
            UNIT_ASSERT_VALUES_EQUAL(report.size(), 2);
            UNIT_ASSERT_C(report.front().IsSuccessReply(), report.front().GetDebugReply());
            UNIT_ASSERT_C(report.back().IsSuccessReply(), report.back().GetDebugReply());
        }
        {
            auto future = bucket->GetFile(docName);
            UNIT_ASSERT(future.Wait(mdsClient.GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto docData = future.GetValue();
            UNIT_ASSERT_VALUES_UNEQUAL(docData.GetContent(), doc);
        }
        {
            auto future = bucket->GetDecrypted(docName, encryptionName);
            UNIT_ASSERT(future.Wait(mdsClient.GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto docData = future.GetValue();
            UNIT_ASSERT_VALUES_EQUAL(docData.GetContent(), doc);
        }
        TMessagesCollector errors;
        UNIT_ASSERT_VALUES_EQUAL_C(bucket->DeleteKey(docName, errors) / 100, 2, errors.GetStringReport());
    }
}
