#include "client.h"

#include <rtline/library/json/field.h>
#include <rtline/library/json/merge.h>

#include <library/cpp/json/json_reader.h>

#include <util/string/builder.h>

NDrive::TMediaBillingClient::TMediaBillingClient(const TOptions& options, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvm)
    : Options(options)
    , HttpClient(MakeAtomicShared<NNeh::THttpClient>(options.Endpoint))
    , Tvm(tvm)
{
}

NThreading::TFuture<NDrive::TMediaBillingClient::TProducts> NDrive::TMediaBillingClient::GetProducts(TStringBuf clientIp, TStringBuf uid, TStringBuf target) const {
    return MakeRequest(
        "GET",
        Options.GetProductsHandle,
        TStringBuilder() << "ip=" << clientIp << "&__uid=" << uid << "&target=" << target
    ).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        return NJson::FromJson<TProducts>(r.GetValue()["result"]["products"]);
    });
}

NThreading::TFuture<NDrive::TMediaBillingClient::TOrderInfo> NDrive::TMediaBillingClient::GetOrderInfo(TOrderId orderId, TStringBuf uid) const {
    return MakeRequest(
        "GET",
        Options.GetOrderInfoHandle,
        TStringBuilder() << "orderId=" << orderId << "&__uid=" << uid
    ).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        return NJson::FromJson<TOrderInfo>(r.GetValue()["result"]);
    });
}

NThreading::TFuture<NDrive::TMediaBillingClient::TOrderInfo> NDrive::TMediaBillingClient::SubmitOrder(
    TStringBuf productId,
    TStringBuf clientIp,
    TStringBuf uid,
    TStringBuf paymentMethodId,
    TStringBuf source,
    TStringBuf origin
) const {
    auto cgi = TStringBuilder() << "productId=" << productId << "&ip=" << clientIp << "&__uid=" << uid;
    if (paymentMethodId) {
        cgi << "&paymentMethodId=" << paymentMethodId;
    }
    if (source) {
        cgi << "&source=" << source;
    }
    if (origin) {
        cgi << "&origin=" << origin;
    }
    return MakeRequest(
        "POST",
        Options.SubmitOrderHandle,
        cgi
    ).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        return NJson::FromJson<TOrderInfo>(r.GetValue()["result"]);
    });
}

NTvmAuth::TTvmId NDrive::TMediaBillingClient::GetClientId() const {
    if (Options.DestinationClientId) {
        return Options.DestinationClientId;
    }
    if (Options.Endpoint == MediaBillingProductionEndpoint) {
        return MediaBillingProductionClientId;
    }
    if (Options.Endpoint == MediaBillingTestingEndpoint) {
        return MediaBillingTestingClientId;
    }
    return 0;
}

NThreading::TFuture<NJson::TJsonValue> NDrive::TMediaBillingClient::MakeRequest(const TString& method, const TString& handle, const TString& cgi) const {
    NNeh::THttpRequest query;
    query.SetUri(handle);
    query.SetCgiData(cgi);
    query.SetRequestType(method);
    if (Tvm) {
        auto ticket = Tvm->GetServiceTicketFor(GetClientId());
        query.AddHeader("X-Ya-Service-Ticket", ticket);
    }
    auto deadline = Now() + Options.DefaultTimeout;
    auto reply = HttpClient->SendAsync(query, deadline);
    auto result = reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        reply.EnsureSuccessfulReply();
        return NJson::ReadJsonFastTree(reply.Content());
    });
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TMediaBillingClient::TProduct& object) {
    NJson::TJsonValue result = object.Raw;
    NJson::MergeJson(NJson::FieldsToJson(object.GetFields()), result);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TMediaBillingClient::TProduct& result) {
    return
        NJson::TryFieldsFromJson(value, result.GetFields()) &&
        NJson::TryFromJson(value, result.Raw);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TMediaBillingClient::TProductDescription& object) {
    NJson::TJsonValue result = object.Raw;
    NJson::MergeJson(NJson::FieldsToJson(object.GetFields()), result);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TMediaBillingClient::TProductDescription& result) {
    return
        NJson::TryFieldsFromJson(value, result.GetFields()) &&
        NJson::TryFromJson(value, result.Raw);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TMediaBillingClient::TOrderInfo& object) {
    NJson::TJsonValue result = object.Raw;
    NJson::MergeJson(NJson::FieldsToJson(object.GetFields()), result);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TMediaBillingClient::TOrderInfo& result) {
    return
        NJson::TryFieldsFromJson(value, result.GetFields()) &&
        NJson::TryFromJson(value, result.Raw);
}
