#pragma once

#include <util/generic/map.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/string/split.h>

namespace NDrive {

    // THttpRouter represents simple implementation for HTTP routing.
    template<class THandler>
    class THttpRouter {
    private:
        struct TRoute {
            TString Method;
            TString Path;
            THandler Handler;
        };

        TVector<TRoute> Routes;

        bool MatchRoute(const TRoute& route, const TString& path, TMap<TString, TString>& vars) const {
            auto&& routeParts = StringSplitter(route.Path).Split('/');
            auto&& pathParts = StringSplitter(path).Split('/');
            auto it = routeParts.begin();
            auto jt = pathParts.begin();
            while (it != routeParts.end() && jt != pathParts.end()) {
                if (!it->empty() && (*it)[0] == ':') {
                    vars[TString(it->substr(1))] = *jt;
                } else if (*it != *jt) {
                    return false;
                }
                ++it;
                ++jt;
            }
            return it == routeParts.end() && jt == pathParts.end();
        }

    public:
        struct TMatch {
            TMap<TString, TString> Vars;
            THandler Handler;
        };

        THttpRouter() = default;

        // Add adds route with path pattern for handler.
        //
        // Examples of valid paths:
        //   /api/object
        //   /api/object/:id
        //   /api/object/:id/edit
        //   /api/object/:id/:subid
        //
        // Examples of invalid paths:
        //   /api/object-:id
        //   /api/object/:id:action
        void Add(const TString& path, const TString& method, THandler handler) {
            TRoute route;
            route.Method = method;
            route.Path = path;
            route.Handler = handler;
            Routes.push_back(route);
        }

        // Match tries to match specified paths.
        //
        // If there is no valid route, match will return false, otherwise
        // `match` parameter will be updated with specified vars and handler.
        [[nodiscard]] bool Match(TMatch& match, const TString& path, const TString& method) const {
            for (auto&& route : Routes) {
                if (route.Method != method) {
                    continue;
                }
                TMap<TString, TString> vars;
                if (MatchRoute(route, path, vars)) {
                    match.Vars = vars;
                    match.Handler = route.Handler;
                    return true;
                }
            }
            return false;
        }
    };

}
