#include "client.h"

#include <rtline/util/json_processing.h>

bool TOctopusClient::Call(const TString& phoneNumber, const TString& helloPrompt, const TString& goodbyePrompt, TString& sessionId) const {
    NJson::TJsonValue callScript = Config.GetCallScript();
    callScript["steps"]["start"]["parameters"]["called_number"] = phoneNumber;
    callScript["steps"]["ask_phrase"]["parameters"]["prompt"] = helloPrompt;
    callScript["steps"]["say_thanks"]["parameters"]["prompt"] = goodbyePrompt;

    NJson::TJsonValue requestPayload;
    requestPayload["script"] = std::move(callScript);

    NNeh::THttpRequest request;
    request
        .SetUri("/" + Config.GetCallURI())
        .SetRequestType("POST")
        .SetPostData(requestPayload.GetStringRobust())
        .AddHeader("Auth-Key", Config.GetToken())
        .AddHeader("Content-Type", "application/json");

    auto tgResult = Agent->SendMessageSync(request, Now() + Config.GetRequestTimeout());
    if (tgResult.Code() != HTTP_OK) {
        ERROR_LOG << "Octopus call failure: " << request.GetDebugRequest() << ": " << tgResult.Code() << " / " << tgResult.Content() << " / " << tgResult.ErrorMessage() << Endl;
        return false;
    }

    NJson::TJsonValue jsonResponse;
    if (!NJson::ReadJsonFastTree(tgResult.Content(), &jsonResponse)) {
        ERROR_LOG << "Octopus call failure: response not json " << request.GetDebugRequest() << ": " << tgResult.Code() << " / " << tgResult.Content() << Endl;
    }

    JREAD_STRING(jsonResponse, "session_id", sessionId);
    return true;
}

NRobotPhoneCall::ECallStatus TOctopusClient::GetCallStatus(const TString& sessionId) const {
    NNeh::THttpRequest request;
    request
        .SetUri("/" + Config.GetCallStatusURI())
        .SetCgiData("session_id=" + sessionId)
        .SetRequestType("GET")
        .AddHeader("Auth-Key", Config.GetToken());

    auto tgResult = Agent->SendMessageSync(request, Now() + Config.GetRequestTimeout());
    if (tgResult.Code() != HTTP_OK) {
        ERROR_LOG << "Octopus get status failure: " << request.GetDebugRequest() << ": " << tgResult.Code() << " / " << tgResult.Content() << " / " << tgResult.ErrorMessage() << Endl;
        return NRobotPhoneCall::ECallStatus::Error;
    }

    NJson::TJsonValue jsonResponse;
    if (!NJson::ReadJsonFastTree(tgResult.Content(), &jsonResponse) || !jsonResponse.IsArray()) {
        ERROR_LOG << "Octopus get status failure: response not json " << request.GetDebugRequest() << ": " << tgResult.Code() << " / " << tgResult.Content() << Endl;
        return NRobotPhoneCall::ECallStatus::Error;
    }

    TMap<TString, NJson::TJsonValue> sectionsByType;
    for (auto&& section : jsonResponse.GetArray()) {
        if (!section.Has("action_type") || !section["action_type"].IsString()) {
            continue;
        }
        TString actionType = section["action_type"].GetString();
        sectionsByType[actionType] = section;
    }

    if (!sectionsByType.contains("ORIGINATE")) {
        return NRobotPhoneCall::ECallStatus::InProgress;
    }

    auto phaseSectionIt = sectionsByType.find("PHASE_END");
    if (phaseSectionIt != sectionsByType.end()) {
        auto subsectionPtr = phaseSectionIt->second.GetValueByPath("action_args.state");
        if (subsectionPtr && subsectionPtr->IsString() && subsectionPtr->GetString() == "ERROR") {
            return NRobotPhoneCall::ECallStatus::Error;
        }
    }

    auto stateSectionIt = sectionsByType.find("SESSION_DUMP_STATE");
    if (stateSectionIt == sectionsByType.end()) {
        return NRobotPhoneCall::ECallStatus::InProgress;
    }

    if (!sectionsByType.contains("PLAY_PROMPT")) {
        return NRobotPhoneCall::ECallStatus::Unanswer;
    }

    if (stateSectionIt->second.GetValueByPath("action_args.state.variables.vm_messages")) {
        return NRobotPhoneCall::ECallStatus::OK;
    }

    return NRobotPhoneCall::ECallStatus::Error;
}

bool TFakeOctopusClient::Call(const TString& /*phoneNumber*/, const TString& /*helloPrompt*/, const TString& /*goodbyePrompt*/, TString& sessionId) const {
    sessionId = "abcdefgh";
    return true;
}

NRobotPhoneCall::ECallStatus TFakeOctopusClient::GetCallStatus(const TString& /*sessionId*/) const {
    TGuard g(Mutex);
    size_t actionIndex = CurrentResponseIndex;
    CurrentResponseIndex = (CurrentResponseIndex + 1) % Responses.size();
    return Responses[actionIndex];
}
