#include "client.h"

#include <drive/library/cpp/openssl/rsa.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/mediator/global_notifications/system_status.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/unistat/cache.h>

#include <util/charset/wide.h>
#include <util/folder/path.h>
#include <util/stream/file.h>
#include <util/stream/str.h>
#include <util/string/builder.h>

void NDrive::TParkingPaymentClient3::TBalance::FromJson(const NJson::TJsonValue& json) {
    Object = json;
    Currency = json["currency"].GetStringSafe();
    Amount = json["amount"].GetDoubleRobust();
}

void NDrive::TParkingPaymentClient3::TBalance::FromXml(const NXml::TNode& node) {
    auto fund = node.Node("funds", /* quiet = */ false);
    Amount = fund.Value<double>();
    Object = NJson::TMapBuilder("amount", Amount)("currency", Currency);
}

NJson::TJsonValue NDrive::TParkingPaymentClient3::TBalance::ToJson() const {
    return Object;
}

void NDrive::TParkingPaymentClient3::TCost::FromJson(const NJson::TJsonValue& json) {
    Currency = json["currency"].GetStringSafe();
    Amount = json["amount"].GetDoubleRobust();
}

NJson::TJsonValue NDrive::TParkingPaymentClient3::TCost::ToJson() const {
    NJson::TJsonValue result;
    result["currency"] = Currency;
    result["amount"] = Amount;
    return result;
}

void NDrive::TParkingPaymentClient3::TVehicle::FromJson(const NJson::TJsonValue& json) {
    LicensePlate = json["licensePlate"].GetStringSafe();
    Synonym = json["name"].GetStringSafe();
    Type = json["type"].GetStringSafe();
    if (json.Has("reference")) {
        Reference = json["reference"].GetStringSafe();
    }
}

NJson::TJsonValue NDrive::TParkingPaymentClient3::TVehicle::ToJson() const {
    NJson::TJsonValue result;
    result["licensePlate"] = LicensePlate;
    result["name"] = Synonym;
    result["type"] = Type;
    if (Reference) {
        result["reference"] = Reference;
    }
    return result;
}

void NDrive::TParkingPaymentClient3::TParking::FromJson(const NJson::TJsonValue& json) {
    ParkingId = json["id"].GetStringRobust();
    AggregatorId = json["aggregatorId"].GetIntegerRobust();
    Name = json["name"].GetString();
    Latitude = json["coordinates"]["latitude"].GetDouble();
    Longitude = json["coordinates"]["longitude"].GetDouble();
}

NJson::TJsonValue NDrive::TParkingPaymentClient3::TParking::ToJson() const {
    NJson::TJsonValue result;
    if (Address) {
        result["address"] = Address;
    }
    if (Attributes) {
        result["attributes"] = Attributes;
    }
    result["aggregatorId"] = AggregatorId;
    result["id"] = ParkingId;
    result["name"] = Name;
    result["coordinates"]["latitude"] = Latitude;
    result["coordinates"]["longitude"] = Longitude;
    return result;
}

void NDrive::TParkingPaymentClient3::TOffer::FromJson(const NJson::TJsonValue& json) {
    Id = json["sessionReference"].GetStringSafe();
    if (json.Has("parkingCost")) {
        Cost.FromJson(json["parkingCost"]);
    } else {
        Cost.FromJson(json["cost"]);
    }
}

NJson::TJsonValue NDrive::TParkingPaymentClient3::TOffer::ToJson() const {
    NJson::TJsonValue result;
    result["sessionReference"] = Id;
    result["cost"] = Cost.ToJson();
    result["parking"] = Parking.ToJson();
    result["vehicle"] = Vehicle.ToJson();
    return result;
}

void NDrive::TParkingPaymentClient3::TSession::FromJson(const NJson::TJsonValue& json) {
    Id = json["sessionReference"].GetString();
    ParkingStartTime = json["startTime"].GetStringSafe();
    StartTime = TInstant::ParseIso8601Deprecated(ParkingStartTime);
    ParkingEndTime = json["endTime"].GetStringSafe();
    EndTime = TInstant::ParseIso8601Deprecated(ParkingEndTime);
    Status = json["sessionStatus"].GetString();
    if (json.Has("parking")) {
        Parking.FromJson(json["parking"]);
    }
    if (json.Has("vehicle")) {
        Vehicle.FromJson(json["vehicle"]);
    }
}

NJson::TJsonValue NDrive::TParkingPaymentClient3::TSession::ToJson() const {
    NJson::TJsonValue result;
    if (Id) {
        result["sessionReference"] = Id;
    }
    if (Status) {
        result["sessionStatus"] = Status;
    }
    result["startTime"] = ParkingStartTime;
    result["endTime"] = ParkingEndTime;
    if (Parking.ParkingId) {
        result["parking"] = Parking.ToJson();
    }
    if (Vehicle) {
        result["vehicle"] = Vehicle.ToJson();
    }
    return result;
}

void NDrive::TParkingPaymentClient3::TSession::FromXml(const NXml::TNode& node) {
    auto parking = node.Node("parking", /* quiet = */ false);
    Id = parking.Node("sessionId", /* quiet = */ false).Value<TString>();
    ParkingStartTime = parking.Node("startTime", /* quiet = */ false).Value<TString>();
    StartTime = TInstant::ParseIso8601(ParkingStartTime) - TDuration::Hours(3);
    ParkingEndTime = parking.Node("stopTime", /* quiet = */ false).Value<TString>();
    EndTime = TInstant::ParseIso8601(ParkingEndTime) - TDuration::Hours(3);
}

NDrive::TParkingPaymentBase::TParkingPaymentBase(
    const TString& token,
    const TString& applicationId,
    const TString& host /*= "yoomoney.ru"*/,
    EAuthorizationType authorizationType,
    TAsyncDelivery::TPtr asyncDelivery /*= nullptr*/
)
    : ApplicationId(applicationId)
    , Token(token)
    , AuthorizationType(authorizationType)
    , AsyncDelivery(asyncDelivery ? asyncDelivery : MakeAtomicShared<TAsyncDelivery>())
    , ExternalAsyncDelivery(asyncDelivery)
    , Requester(MakeHolder<NNeh::THttpClient>(AsyncDelivery))
{
    if (!ExternalAsyncDelivery) {
        AsyncDelivery->Start(4, 4);
    }
    NSimpleMeta::TConfig config;
    config.SetMaxAttempts(1);
    config.SetGlobalTimeout(TDuration::Seconds(10));
    Requester->RegisterSource("money", host, 443, config, true);
}

NDrive::TParkingPaymentBase::~TParkingPaymentBase() {
    if (!ExternalAsyncDelivery) {
        AsyncDelivery->Stop();
    }
}

NThreading::TFuture<NJson::TJsonValue> NDrive::TParkingPaymentBase::AsyncFetch(const TString& url, const TString& method, const NJson::TJsonValue& data) const {
    NNeh::THttpRequest request;
    TString uri = url;
    request.SetUri(uri);
    request.AddHeader("Content-Type", "application/json");
    switch (AuthorizationType) {
        case EAuthorizationType::Parking:
            request.AddHeader("Parking-Authorization", Token);
            break;
        case EAuthorizationType::Basic:
            request.AddHeader("Authorization", "Basic " + Token);
            break;
        case EAuthorizationType::Bearer:
            request.AddHeader("Authorization", "Bearer " + Token);
            break;
    }
    request.SetRequestType(method);

    TString post = data.GetStringRobust();
    if (data.IsDefined()) {
        request.SetPostData(TBlob::FromStringSingleThreaded(post), method);
    }

    DEBUG_LOG << "Requesting " << uri << " " << post << Endl;
    NThreading::TFuture<NUtil::THttpReply> reply = Requester->SendAsync(request, Now() + TDuration::Seconds(10));
    return reply.Apply([uri, post, this](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            throw yexception() << "cannot fetch " << uri << " " << Token << " " << post << " : " << reply.Code() << " " << reply.Content();
        }
        DEBUG_LOG << "Received " << uri << " " << post << ": " << reply.Content() << Endl;

        NJson::TJsonValue result;
        NJson::ReadJsonFastTree(reply.Content(), &result, true);

        if (result.Has("error")) {
            throw yexception() << "error fetch " << uri << " " << Token << " " << post << " : " << reply.Code() << " " << result["error"].GetStringRobust();
        }

        return result;
    });
}

NJson::TJsonValue NDrive::TParkingPaymentBase::Fetch(const TString& url, const TString& method, const NJson::TJsonValue& data) const {
    return AsyncFetch(url, method, data).ExtractValueSync();
}

template <class T>
T NDrive::TParkingPaymentBase::ParseObject(const NJson::TJsonValue& response, const TString& path) const {
    const NJson::TJsonValue* object = response.GetValueByPath(path);
    if (!object) {
        throw yexception() << "path " << path << " is not found in " << response.GetStringRobust();
    }
    T result;
    result.FromJson(*object);
    return result;
}

template <class T>
TVector<T> NDrive::TParkingPaymentBase::ParseObjects(const NJson::TJsonValue& response, const TString& path) const {
    const NJson::TJsonValue* object = response.GetValueByPath(path);
    if (!object) {
        throw yexception() << "path " << path << " is not found in " << response.GetStringRobust();
    }
    if (!object->IsArray()) {
        throw yexception() << "path " << path << " should be an array in " << response.GetStringRobust();
    }
    TVector<T> result;
    for (auto&& i : object->GetArraySafe()) {
        result.emplace_back();
        result.back().FromJson(i);
    }
    return result;
}

NDrive::TParkingPaymentClient3::TBalance NDrive::TParkingPaymentClient3::GetBalance(const TParking& parking) const {
    NJson::TJsonValue request;
    request["aggregatorId"] = parking.AggregatorId;
    request["parkingId"] = parking.ParkingId;
    return ParseObject<TBalance>(Fetch("/api/parking/v3/get-parking-account-balance", "POST", request), "balanceDetails.accountBalance");
}

NDrive::TParkingPaymentClient3::TVehicle NDrive::TParkingPaymentClient3::AddVehicle(const TString& name, const TString& number) const {
    TVehicle vehicle(number, name);
    vehicle.Reference = Fetch("/api/parking/v3/add-vehicle", "POST", vehicle.ToJson())["reference"].GetStringRobust();
    return vehicle;
}

NDrive::TParkingPaymentClient3::TVehicles NDrive::TParkingPaymentClient3::GetVehicles() const {
    return ParseObjects<TVehicle>(Fetch("/api/parking/v3/get-vehicle-list", "POST", NJson::JSON_MAP), "vehicles");
}

void NDrive::TParkingPaymentClient3::DeleteVehicle(const TVehicle& vehicle) const {
    Fetch("/api/parking/v3/delete-vehicle", "POST", vehicle.ToJson());
}

NDrive::TParkingPaymentClient3::TSessions NDrive::TParkingPaymentClient3::GetSessions() const {
    return ParseObjects<TSession>(Fetch("/api/parking/v3/get-active-sessions", "POST", NJson::JSON_MAP), "sessions");
}

NDrive::TParkingPaymentClient3::TOffer NDrive::TParkingPaymentClient3::GetOffer(const TParking& parking, const TVehicle& vehicle, TDuration duration) const {
    NJson::TJsonValue data;
    data["duration"] = duration.Minutes();
    data["parking"] = parking.ToJson();
    data["vehicleReference"] = vehicle.Reference;
    data["notificationSettings"]["smsNotificationEnabled"] = false;
    data["notificationSettings"]["pushNotificationEnabled"] = false;
    if (ApplicationId) {
        data["applicationId"] = ApplicationId;
    }
    auto offer = ParseObject<TOffer>(Fetch("/api/parking/v3/get-prepay-parking-cost", "POST", data), "orderDetails");
    offer.Parking = parking;
    offer.Vehicle = vehicle;
    return offer;
}

NDrive::TParkingPaymentClient3::TSession NDrive::TParkingPaymentClient3::StartParking(const TOffer& offer, TDuration /*duration*/) const {
    NJson::TJsonValue data;
    data["sessionReference"] = offer.Id;
    auto result = ParseObject<TSession>(Fetch("/api/parking/v3/pay-parking", "POST", data), "sessionDetails");
    if (!result.Id) {
        result.Id = offer.Id;
    }
    if (!result.Vehicle) {
        result.Vehicle = offer.Vehicle;
    }
    return result;
}

NDrive::TParkingPaymentClient3::TRefund NDrive::TParkingPaymentClient3::StopParking(const TSessionReference& offerId) const {
    NJson::TJsonValue data;
    data["sessionReference"] = offerId;
    return ParseObject<TRefund>(Fetch("/api/parking/v3/stop-parking", "POST", data), "sessionDetails.refund");
}

NDrive::TParkingPaymentClient3::TVehicle NDrive::TParkingPaymentClient3::GetVehicleBySynonym(const TString& synonym) const {
    auto vehicles = GetVehicles();
    for (auto&& vehicle : vehicles) {
        if (vehicle.Synonym == synonym) {
            return vehicle;
        }
    }
    return {};
}

NDrive::TParkingPaymentClient3::TVehicle NDrive::TParkingPaymentClient3::GetVehicleByReference(const TString& refen) const {
    auto vehicles = GetVehicles();
    for (auto&& vehicle : vehicles) {
        if (vehicle.Reference == refen) {
            return vehicle;
        }
    }
    return{};
}

NDrive::TParkingPaymentClient3::TVehicle NDrive::TParkingPaymentClient3::AddVehicleRobust(const TString& synonym, const TString& number) const {
    auto vehicles = GetVehicles();
    for (auto&& vehicle : vehicles) {
        if (vehicle.LicensePlate == number) {
            return vehicle;
        }
    }
    try {
        return AddVehicle(synonym, number);
    } catch (const std::exception& e) {
        ERROR_LOG << "An exception occurred: " << FormatExc(e) << Endl;
        auto vehicles = GetVehicles();
        for (auto&& vehicle : vehicles) {
            if (vehicle.LicensePlate == number) {
                return vehicle;
            }
        }
        throw;
    }
}

void NDrive::TParkingPaymentClient3::DeleteVehicleRobust(const TVehicle& vehicle) const {
    auto vehicles = GetVehicles();
    for (auto&& candidate : vehicles) {
        if (candidate.Reference == vehicle.Reference) {
            DeleteVehicle(vehicle);
        }
    }
}

NDrive::TParkingPaymentClient3::TSession NDrive::TParkingPaymentClient3::StartParkingRobust(const TOffer& offer, TDuration duration) const {
    try {
        return StartParking(offer, duration);
    } catch (const std::exception& e) {
        ERROR_LOG << "An exception occurred: " << FormatExc(e) << Endl;
        auto sessions = GetSessions();
        for (auto&& session : sessions) {
            if (session.Id == offer.Id) {
                return session;
            }
        }
        throw;
    }
}

NDrive::TParkingPaymentClient3::TRefund NDrive::TParkingPaymentClient3::StopParkingRobust(const TSession& session) const {
    return StopParkingRobust(session.Id);
}

NDrive::TParkingPaymentClient3::TRefund NDrive::TParkingPaymentClient3::StopParkingRobust(const TSessionReference& offerId) const {
    try {
        return StopParking(offerId);
    } catch (const std::exception& e) {
        ERROR_LOG << "An exception occurred: " << FormatExc(e) << Endl;
        auto sessions = GetSessions();
        for (auto&& session : sessions) {
            if (session.Id == offerId) {
                throw;
            }
        }
        INFO_LOG << "Offer " << offerId << " is not found anymore" << Endl;
        return {};
    }
}

void NDrive::TFitDevParkingPaymentClient::TAccount::FromJson(const NJson::TJsonValue& value) {
    Id = value["id"].GetUIntegerSafe();
    Balance = value["balance"].GetDoubleSafe() / 100.0;
}

struct NDrive::TFitDevParkingPaymentClient::TCostImpl {
    i64 Sum;

    void FromJson(const NJson::TJsonValue& value) {
        Sum = value["sum"].GetIntegerSafe();
    }
};

struct NDrive::TFitDevParkingPaymentClient::TRefundImpl {
    i64 Sum;

    void FromJson(const NJson::TJsonValue& value) {
        Sum = value["returnSum"].GetIntegerSafe();
    }
};

struct NDrive::TFitDevParkingPaymentClient::TSessionImpl {
    i64 Id;
    TInstant Start;
    TInstant End;
    TString LicencePlate;
    TString ZoneNumber;

    void FromJson(const NJson::TJsonValue& value) {
        Id = value["id"].GetIntegerSafe();
        Start = TInstant::MilliSeconds(value["start"].GetUIntegerSafe());
        End = TInstant::MilliSeconds(value["end"].GetUIntegerSafe());
        LicencePlate = value["vrp"].GetString();
        ZoneNumber = value["zoneNumber"].GetString();
    }

    NDrive::TFitDevParkingPaymentClient::TSession ToSession() const {
        TSession result;
        result.Id = ToString(Id);
        result.StartTime = Start;
        result.ParkingStartTime = result.StartTime.ToString();
        result.EndTime = End;
        result.ParkingEndTime = result.EndTime.ToString();
        result.Parking.ParkingId = ZoneNumber;
        result.Vehicle.LicensePlate = ToCyrillic(LicencePlate);
        return result;
    }
};

NDrive::TFitDevParkingPaymentClient::TFitDevParkingPaymentClient(const TString& token, const TString& host, TAsyncDelivery::TPtr asyncDelivery)
    : TParkingPaymentBase(token, {}, host, EAuthorizationType::Basic, asyncDelivery)
{
}

NDrive::TFitDevParkingPaymentClient::TFitDevParkingPaymentClient(
    const TString& host,
    const TString& token,
    EAuthorizationType authorizationType,
    const TString& apiVersion,
    TAsyncDelivery::TPtr asyncDelivery
)
    : TParkingPaymentBase(token, {}, host, authorizationType, asyncDelivery)
    , ApiVersion(apiVersion)
    , AccountApiVersion(apiVersion)
{
}

NDrive::TFitDevParkingPaymentClient::TAccount NDrive::TFitDevParkingPaymentClient::GetAccount() const {
    auto response = Fetch("/api/" + AccountApiVersion + "/accounts/me", "GET", NJson::JSON_NULL);
    return ParseObject<TAccount>(response, "account");
}

NDrive::TFitDevParkingPaymentClient::TBalance NDrive::TFitDevParkingPaymentClient::GetBalance() const {
    auto account = GetAccount();
    TBalance balance;
    balance.Amount = account.Balance;
    return balance;
}

NDrive::TFitDevParkingPaymentClient::TCost NDrive::TFitDevParkingPaymentClient::GetCost(const TParking& parking, TDuration duration) const {
    NJson::TJsonValue data;
    data["zoneNumber"] = parking.ParkingId;
    data["duration"] = duration.Minutes();
    auto response = Fetch("/api/" + ApiVersion + "/accounts/me/reservations/getcost", "PUT", data);
    auto cost = ParseObject<TCostImpl>(response, "");
    TCost result;
    result.Amount = cost.Sum / 100.0;
    return result;
}

NDrive::TFitDevParkingPaymentClient::TOffer NDrive::TFitDevParkingPaymentClient::GetOffer(const TParking& parking, const TVehicle& vehicle, TDuration duration) const {
    TOffer result;
    result.Cost = GetCost(parking, duration);
    result.Parking = parking;
    result.Vehicle = vehicle;
    return result;
}

NDrive::TFitDevParkingPaymentClient::TSessions NDrive::TFitDevParkingPaymentClient::GetSessions() const {
    auto sessions = ParseObjects<TSessionImpl>(Fetch("/api/" + ApiVersion + "/accounts/me/reservations", "GET", NJson::JSON_NULL), "reservations");
    TSessions result;
    for (auto&& session : sessions) {
        result.push_back(session.ToSession());
    }
    return result;
}

NDrive::TFitDevParkingPaymentClient::TSession NDrive::TFitDevParkingPaymentClient::StartParking(const TOffer& offer, TDuration duration) const {
    NJson::TJsonValue data;
    data["zoneNumber"] = offer.Parking.ParkingId;
    data["vrp"] = offer.Vehicle.LicensePlate;
    data["vrpFormat"] = "local";
    data["duration"] = duration.Minutes();
    auto response = Fetch("/api/" + ApiVersion + "/accounts/me/reservations/start", "PUT", data);
    auto session = ParseObject<TSessionImpl>(response, "reservation");

    TSession result = session.ToSession();
    if (result.Parking) {
        Y_ENSURE(result.Parking.ParkingId == offer.Parking.ParkingId, result.Parking.ParkingId << ' ' << offer.Parking.ParkingId);
    }
    result.Parking = offer.Parking;
    if (result.Vehicle) {
        Y_ENSURE(result.Vehicle.LicensePlate == offer.Vehicle.LicensePlate, result.Vehicle.LicensePlate << ' ' << offer.Vehicle.LicensePlate);
    }
    result.Vehicle = offer.Vehicle;
    return result;
}

NDrive::TFitDevParkingPaymentClient::TRefund NDrive::TFitDevParkingPaymentClient::StopParking(const TSessionReference& offerId) const {
    NJson::TJsonValue data;
    data["reservationId"] = FromString<i64>(offerId);
    auto response = Fetch("/api/" + ApiVersion + "/accounts/me/reservations/cancel", "PUT", data);
    auto refund = ParseObject<TRefundImpl>(response, "");
    TRefund result;
    result.Object["amount"] = refund.Sum / 100.0;
    return result;
}

NDrive::TFitDevParkingPaymentClient::TSession NDrive::TFitDevParkingPaymentClient::StartParkingRobust(const TOffer& offer, TDuration duration) const {
    try {
        return StartParking(offer, duration);
    } catch (const std::exception& e) {
        ERROR_LOG << "An exception occurred: " << FormatExc(e) << Endl;
        auto sessions = GetSessions();
        for (auto&& session : sessions) {
            if (session.Vehicle.LicensePlate == offer.Vehicle.LicensePlate && session.Parking.ParkingId == offer.Parking.ParkingId) {
                return session;
            }
        }
        throw;
    }
}

NDrive::TFitDevParkingPaymentClient::TRefund NDrive::TFitDevParkingPaymentClient::StopParkingRobust(const TSession& session) const {
    return StopParkingRobust(session.Id);
}

NDrive::TFitDevParkingPaymentClient::TRefund NDrive::TFitDevParkingPaymentClient::StopParkingRobust(const TSessionReference& sessionId) const {
    try {
        return StopParking(sessionId);
    } catch (const std::exception& e) {
        ERROR_LOG << "An exception occurred: " << FormatExc(e) << Endl;
        auto sessions = GetSessions();
        for (auto&& session : sessions) {
            if (session.Id == sessionId) {
                throw;
            }
        }
        INFO_LOG << "Offer " << sessionId << " is not found anymore" << Endl;
        return {};
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TParkingPaymentClient3::TParking& object) {
    return object.ToJson();
}


void NDrive::TMskParkingPaymentConfig::Init(const TYandexConfig::Section* section) {
    Uri = section->GetDirectives().Value("Uri", Uri);
    AccountUriPath = section->GetDirectives().Value("AccountUriPath", AccountUriPath);
    ParkingUriPath = section->GetDirectives().Value("ParkingUriPath", ParkingUriPath);
    Token = section->GetDirectives().Value("Token", Token);
    TokenPath = section->GetDirectives().Value("TokenPath", TokenPath);
    AssertCorrectConfig(Token || TokenPath, "Need token or token path");
    if (!Token) {
        AssertCorrectConfig(TFsPath(TokenPath).Exists(), "Incorrect path for tanker's apikey in 'TokenPath'");
        TFileInput fi(TokenPath);
        Token = fi.ReadAll();
    }
    PartnerId = section->GetDirectives().Value("PartnerId", PartnerId);
    AccountToken = section->GetDirectives().Value("AccountToken", AccountToken);
    AccountTokenPath = section->GetDirectives().Value("AccountTokenPath", AccountTokenPath);
    AssertCorrectConfig(AccountToken || AccountTokenPath, "Need client token or client token path");
    if (!AccountToken) {
        AssertCorrectConfig(TFsPath(AccountTokenPath).Exists(), "Incorrect path for tanker's apikey in 'TokenPath'");
        TFileInput fi(AccountTokenPath);
        AccountToken = fi.ReadAll();
    }
    {
        const TYandexConfig::TSectionsMap children = section->GetAllChildren();
        auto it = children.find("RequestConfig");
        if (it != children.end()) {
            RequestConfig.InitFromSection(it->second);
        }
    }
    RequestTimeout  = section->GetDirectives().Value("RequestTimeout", RequestTimeout);
}

void NDrive::TMskParkingPaymentConfig::ToString(IOutputStream& os) const {
    os << "Uri: " << Uri << Endl;
    os << "AccountUriPath: " << AccountUriPath << Endl;
    os << "ParkingUriPath: " << ParkingUriPath << Endl;
    if (TokenPath) {
        os << "TokenPath: " << TokenPath << Endl;
    }
    os << "PartnerId: " << PartnerId << Endl;
    if (AccountTokenPath) {
        os << "AccountTokenPath: " << AccountTokenPath << Endl;
    }
    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;
    os << "RequestTimeout : " << RequestTimeout << Endl;
}

void NDrive::TMskParkingPaymentConfig::Authorize(NNeh::THttpRequest& request, const TMap<TString, TString>& cgi) const {
    auto signedCgi = cgi;
    signedCgi["subscriber"] = GetAccountToken();
    signedCgi["partner"] = GetPartnerId();
    signedCgi["time"] = ::ToString(Now().Seconds());
    signedCgi["secret"] = GetToken();
    auto confertPair = [](const std::pair<TString, TString>& item) { return item.first + "=" + item.second; };
    auto concatParams = [&confertPair](auto&& res, const std::pair<TString, TString>& item) {
        return std::move(res) + "&" + confertPair(item);
    };
    {
        TString params = Accumulate(std::next(signedCgi.begin()), signedCgi.end(), confertPair(*signedCgi.begin()), concatParams);
        TString hash = to_lower(HexEncode(NOpenssl::CalcSHA1(params)));
        signedCgi.erase("secret");
        signedCgi["hash"] = hash;
    }
    TString params = Accumulate(std::next(signedCgi.begin()), signedCgi.end(), confertPair(*signedCgi.begin()), concatParams);
    request.SetCgiData(params);
}

NDrive::TMskParkingPaymentConfig NDrive::TMskParkingPaymentConfig::ParseFromString(const TString& configStr) {
    TMskParkingPaymentConfig result;
    TAnyYandexConfig config;
    AssertCorrectConfig(config.ParseMemory(configStr.data()), "Fail to parse config:" + configStr);
    result.Init(config.GetRootSection());
    return result;
}


NDrive::TMskParkingPaymentClient::TMskParkingPaymentClient(const NDrive::TMskParkingPaymentConfig& config)
    : Config(config)
    , Client(new NNeh::THttpClient(Config.GetUri(), Config.GetRequestConfig()))
{
}

template <class TEntity>
class TApplier {
public:
    TApplier(const TString& requestName)
        : RequestName(requestName)
    {
    }

    NThreading::TFuture<TEntity> operator()(const NThreading::TFuture<NUtil::THttpReply>& response) const {
        const auto& report = response.GetValue();
        TUnistatSignalsCache::SignalAdd("ampp-" + RequestName +"-api-codes", ToString(report.Code()), 1);
        if (!report.HasReply()) {
            ythrow yexception() << "No reply: " << report.ErrorMessage();
        }
        if (!report.IsSuccessReply()) {
            ythrow yexception() << "Error getting reply: " << report.GetDebugReply();
        }
        NXml::TDocument xml(report.Content(), NXml::TDocument::String);
        if (xml.Root().Name() != "response" || xml.Root().Attr<bool>("errors")) {
            ythrow yexception() << "Error getting " << RequestName << " reply: " << report.GetDebugReply();
        }
        TEntity session;
        session.FromXml(xml.Root());
        return NThreading::MakeFuture(session);
    }

private:
    TString RequestName;
};

NThreading::TFuture<NDrive::TMskParkingPaymentClient::TBalance> NDrive::TMskParkingPaymentClient::GetBalance() const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetAccountUriPath());
    Config.Authorize(request, {{"action", "balance"}});
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TBalance>("balance"));
}

NThreading::TFuture<NDrive::TMskParkingPaymentClient::TSession> NDrive::TMskParkingPaymentClient::StartParking(const TString& carNumber, const TString& parkingId, const TDuration duration) const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetParkingUriPath());
    TMap<TString, TString> params;
    params["action"] = "start";
    params["carId"] = NDrive::ToLatin(carNumber);
    params["place"] = parkingId;
    params["duration"] = ToString(duration.Minutes());
    Config.Authorize(request, params);
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TSession>("parking_start"));
}

NThreading::TFuture<NDrive::TMskParkingPaymentClient::TSession> NDrive::TMskParkingPaymentClient::StopParking(const TString& carNumber) const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetParkingUriPath());
    TMap<TString, TString> params;
    params["action"] = "stop";
    params["carId"] = NDrive::ToLatin(carNumber);
    Config.Authorize(request, params);
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier<TSession>("parking_stop"));
}

const NDrive::TMskParkingPaymentConfig& NDrive::TMskParkingPaymentClient::GetConfig() const {
    return Config;
}

namespace {
    TMaybe<std::pair<TString, ui32>> GetZoneFromJson(const NJson::TJsonValue& value) {
        TString number = value["number"].GetStringSafe();
        if (!number) {
            return {};
        }
        for (const auto& item : value["prices"].GetArray()) {
            if (item["vehicleType"].GetStringSafe() == "car") {
                return std::make_pair<TString, ui32>(std::move(number), item["price"]["max"].GetIntegerSafe());
            }
        }
        return {};
    }
}

NThreading::TFuture<TMap<TString, ui32>> NDrive::TMskParkingPaymentClient::GetCosts() const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetParkingCostsPath());
    return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([path = Config.GetParkingCostsPath()](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<TMap<TString, ui32>> {
        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            throw yexception() << "cannot fetch " << path << ": " << reply.Code() << " " << reply.Content();
        }
        DEBUG_LOG << "Received " << path << ": " << reply.Content() << Endl;

        NJson::TJsonValue result;
        NJson::ReadJsonFastTree(reply.Content(), &result, true);

        if (result.Has("error")) {
            throw yexception() << "error fetch " << path << ": " << reply.Code() << " " << result["error"].GetStringRobust();
        }
        TMap<TString, ui32> map;
        for (const auto& json : result["zones"].GetArray()) {
            if (auto zone = GetZoneFromJson(json)) {
                map.emplace(*zone);
            }
        }
        return NThreading::MakeFuture(map);
    });
}
