#pragma once

#include <drive/library/cpp/car/number.h>

#include <rtline/util/network/neh.h>
#include <rtline/library/json/builder.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/xml/document/xml-document.h>

namespace NDrive {
    // a client for https://yoomoney.ru/api/parking/*
    class TParkingPaymentBase {
    public:
        enum class EAuthorizationType {
            Parking,
            Basic,
            Bearer,
        };

    public:
        TParkingPaymentBase(
            const TString& token,
            const TString& applicationId = {},
            const TString& host = "yoomoney.ru",
            EAuthorizationType authorizationType = EAuthorizationType::Parking,
            TAsyncDelivery::TPtr asyncDelivery = nullptr
        );
        ~TParkingPaymentBase();

    protected:
        NThreading::TFuture<NJson::TJsonValue> AsyncFetch(const TString& url, const TString& method, const NJson::TJsonValue& data) const;
        NJson::TJsonValue Fetch(const TString& url, const TString& method, const NJson::TJsonValue& data) const;

        template <class T>
        T ParseObject(const NJson::TJsonValue& response, const TString& path = "result") const;
        template <class T>
        TVector<T> ParseObjects(const NJson::TJsonValue& response, const TString& path = "result") const;

    protected:
        const TString ApplicationId;

    private:
        const TString Token;
        const EAuthorizationType AuthorizationType;

        TAsyncDelivery::TPtr AsyncDelivery;
        TAsyncDelivery::TPtr ExternalAsyncDelivery;
        THolder<NNeh::THttpClient> Requester;
    };

    class TParkingPaymentClient3 : public TParkingPaymentBase {
    public:
        using TSessionReference = TString;

        class TBalance {
        public:
            NJson::TJsonValue Object;
            TString Currency;
            double Amount = 0;

        public:
            TBalance() = default;

            void FromJson(const NJson::TJsonValue& json);
            void FromXml(const NXml::TNode& node);
            NJson::TJsonValue ToJson() const;
        };

        class TCost {
        public:
            TString Currency;
            double Amount = 0;

        public:
            TCost() = default;

            void FromJson(const NJson::TJsonValue& json);
            NJson::TJsonValue ToJson() const;
        };

        class TRefund {
        public:
            NJson::TJsonValue Object;

        public:
            void FromJson(const NJson::TJsonValue& json) {
                Object = json;
            }
            NJson::TJsonValue ToJson() const {
                return Object;
            }
        };

        class TVehicle {
        public:
            TString LicensePlate;
            TString Synonym;
            TString Type;
            TString Reference;

        public:
            TVehicle() = default;
            TVehicle(const TString& number, const TString& name, const TString& type = "B")
                : LicensePlate(number)
                , Synonym(name)
                , Type(type)
            {
            }

            explicit operator bool() const {
                return LicensePlate || Reference;
            }

            void FromJson(const NJson::TJsonValue& json);
            NJson::TJsonValue ToJson() const;
        };
        using TVehicles = TVector<TVehicle>;

        class TParking {
        public:
            TString Name;
            TString Address;
            TString Attributes;
            i64 AggregatorId = 0;
            TString ParkingId;
            double Latitude = 0;
            double Longitude = 0;

        public:
            TParking() = default;

            explicit operator bool() const {
                return !Name.empty();
            }

            void FromJson(const NJson::TJsonValue& json);
            NJson::TJsonValue ToJson() const;

            const TString& GetParkingName() const {
                return Name;
            }
        };
        using TParkings = TVector<TParking>;

        class TOffer {
        public:
            TSessionReference Id;
            TCost Cost;
            TVehicle Vehicle;
            TParking Parking;

        public:
            TOffer() = default;

            void FromJson(const NJson::TJsonValue& json);
            NJson::TJsonValue ToJson() const;
        };

        class TSession {
        public:
            TSessionReference Id;
            TString ParkingStartTime;
            TInstant StartTime;
            TString ParkingEndTime;
            TInstant EndTime;
            TParking Parking;
            TVehicle Vehicle;
            TString Status;

        public:
            TSession() = default;

            void FromJson(const NJson::TJsonValue& json);
            void FromXml(const NXml::TNode& node);
            NJson::TJsonValue ToJson() const;
        };
        using TSessions = TVector<TSession>;

    public:
        using TParkingPaymentBase::TParkingPaymentBase;

        TBalance GetBalance(const TParking& parking) const;

        TVehicle AddVehicle(const TString& name, const TString& number) const;
        TVehicles GetVehicles() const;
        void DeleteVehicle(const TVehicle& vehicle) const;

        TSessions GetSessions() const;
        TOffer GetOffer(const TParking& parking, const TVehicle& vehicle, TDuration duration) const;

        TSession StartParking(const TOffer& offer, TDuration duration) const;
        TRefund StopParking(const TSessionReference& offerId) const;

    public:
        TVehicle GetVehicleBySynonym(const TString& synonym) const;
        TVehicle GetVehicleByReference(const TString& refen) const;

        TVehicle AddVehicleRobust(const TString& synonym, const TString& number) const;
        void DeleteVehicleRobust(const TVehicle& vehicle) const;

        TSession StartParkingRobust(const TOffer& offer, TDuration duration) const;
        TRefund StopParkingRobust(const TSession& session) const;
        TRefund StopParkingRobust(const TSessionReference& offerId) const;
    };

    class TFitDevParkingPaymentClient : public TParkingPaymentBase {
    public:
        using TBalance = TParkingPaymentClient3::TBalance;
        using TCost = TParkingPaymentClient3::TCost;
        using TOffer = TParkingPaymentClient3::TOffer;
        using TParking = TParkingPaymentClient3::TParking;
        using TRefund = TParkingPaymentClient3::TRefund;
        using TSession = TParkingPaymentClient3::TSession;
        using TSessions = TParkingPaymentClient3::TSessions;
        using TSessionReference = TParkingPaymentClient3::TSessionReference;
        using TVehicle = TParkingPaymentClient3::TVehicle;

        class TAccount {
        public:
            ui64 Id;
            double Balance;

        public:
            void FromJson(const NJson::TJsonValue& json);
        };

    public:
        TFitDevParkingPaymentClient(const TString& token, const TString& host = "parkingkzn.ru", TAsyncDelivery::TPtr asyncDelivery = nullptr);

        TAccount GetAccount() const;
        TBalance GetBalance() const;
        TCost GetCost(const TParking& parking, TDuration duration) const;
        TOffer GetOffer(const TParking& parking, const TVehicle& vehicle, TDuration duration) const;
        TSessions GetSessions() const;

        TSession StartParking(const TOffer& offer, TDuration duration) const;
        TRefund StopParking(const TSessionReference& offerId) const;

    public:
        TSession StartParkingRobust(const TOffer& offer, TDuration duration) const;
        TRefund StopParkingRobust(const TSession& session) const;
        TRefund StopParkingRobust(const TSessionReference& sessionId) const;

    protected:
        TFitDevParkingPaymentClient(
            const TString& host,
            const TString& token,
            EAuthorizationType authorizationType,
            const TString& apiVersion,
            TAsyncDelivery::TPtr asyncDelivery = nullptr
        );

    private:
        struct TCostImpl;
        struct TRefundImpl;
        struct TSessionImpl;

    private:
        TString ApiVersion = "2.12";
        TString AccountApiVersion = "2.13";
    };

    class TMosParkingPaymentClient: public TFitDevParkingPaymentClient {
    public:
        TMosParkingPaymentClient(const TString& token, TAsyncDelivery::TPtr asyncDelivery = nullptr)
            : TFitDevParkingPaymentClient("lk.parking.mos.ru", token, EAuthorizationType::Bearer, "2.30", asyncDelivery)
        {
        }
    };

    class TSpbParkingPaymentClient: public TFitDevParkingPaymentClient {
    public:
        TSpbParkingPaymentClient(const TString& token, TAsyncDelivery::TPtr asyncDelivery = nullptr)
            : TFitDevParkingPaymentClient("parking.spb.ru", token, EAuthorizationType::Basic, "2.36", asyncDelivery)
        {
        }
    };

    class TMskParkingPaymentConfig {
    public:
        void Init(const TYandexConfig::Section* section);
        void ToString(IOutputStream& os) const;
        void Authorize(NNeh::THttpRequest& request, const TMap<TString, TString>& cgi) const;
        static TMskParkingPaymentConfig ParseFromString(const TString& configStr);

    private:
        R_READONLY(TString, Uri, "https://lk.parkingtest.ru/");
        R_READONLY(TString, Token);
        R_READONLY(TString, TokenPath);
        R_READONLY(TString, PartnerId, "Yandex");
        R_READONLY(TString, AccountUriPath, "paymentApi/api/1.0/account");
        R_READONLY(TString, ParkingUriPath, "paymentApi/api/1.0/parking");
        R_READONLY(TString, ParkingCostsPath, "api/2.45/zones");
        R_READONLY(TString, AccountToken);
        R_READONLY(TString, AccountTokenPath);
        R_READONLY(NSimpleMeta::TConfig, RequestConfig);
        R_READONLY(TDuration, RequestTimeout, TDuration::Seconds(1));
    };

    class TMskParkingPaymentClient {
    public:
        using TBalance = TParkingPaymentClient3::TBalance;
        using TSession = TParkingPaymentClient3::TSession;

    public:
        TMskParkingPaymentClient(const TMskParkingPaymentConfig& config);
        NThreading::TFuture<TBalance> GetBalance() const;
        NThreading::TFuture<TSession> StartParking(const TString& vehicleId, const TString& parkingId, const TDuration duration) const;
        NThreading::TFuture<TSession> StopParking(const TString& vehicleId) const;
        const TMskParkingPaymentConfig& GetConfig() const;
        NThreading::TFuture<TMap<TString, ui32>> GetCosts() const;

    private:
        TMskParkingPaymentConfig Config;
        THolder<NNeh::THttpClient> Client;
    };
}
