#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(ParkingPaymentClientSuite) {
    Y_UNIT_TEST(FitDev) {
        auto start = Now();
        auto token = GetEnv("PARKINGKZN_TOKEN");
        UNIT_ASSERT(token);
        NDrive::TFitDevParkingPaymentClient client(token);

        NDrive::TFitDevParkingPaymentClient::TParking parking;
        parking.ParkingId = "145";

        NDrive::TFitDevParkingPaymentClient::TVehicle vehicle;
        vehicle.LicensePlate = "т208мт799";

        TDuration duration = TDuration::Hours(1);

        NDrive::TFitDevParkingPaymentClient::TSessions sessions = client.GetSessions();
        for (auto&& session : sessions) {
            client.StopParking(session.Id);
        }

        NDrive::TFitDevParkingPaymentClient::TCost cost = client.GetCost(parking, duration);
        UNIT_ASSERT(cost.Amount > 0);
        NDrive::TFitDevParkingPaymentClient::TOffer offer = client.GetOffer(parking, vehicle, duration);
        UNIT_ASSERT_VALUES_EQUAL(offer.Vehicle.LicensePlate, vehicle.LicensePlate);
        UNIT_ASSERT_VALUES_EQUAL(offer.Parking.ParkingId, parking.ParkingId);
        UNIT_ASSERT_DOUBLES_EQUAL(offer.Cost.Amount, cost.Amount, 0.001);

        NDrive::TFitDevParkingPaymentClient::TSession session = client.StartParkingRobust(offer, duration);
        UNIT_ASSERT(FromString<i64>(session.Id) > 0);
        UNIT_ASSERT(session.StartTime > start);
        UNIT_ASSERT(session.EndTime > start + duration);
        UNIT_ASSERT_VALUES_EQUAL(session.Vehicle.LicensePlate, vehicle.LicensePlate);
        UNIT_ASSERT_VALUES_EQUAL(session.Parking.ParkingId, parking.ParkingId);

        NDrive::TFitDevParkingPaymentClient::TRefund refund = client.StopParkingRobust(session.Id);
    }

    Y_UNIT_TEST(FitDevBalance) {
        auto token = GetEnv("PARKINGKZN_TOKEN");
        UNIT_ASSERT(token);
        NDrive::TFitDevParkingPaymentClient client(token);

        auto balance = client.GetBalance();
        UNIT_ASSERT(balance.Amount > 0);
    }

    Y_UNIT_TEST(MoneyReadOnly) {
        auto token = GetEnv("YMPARKING_TOKEN", "10000000000000000000000000006326");
        NDrive::TParkingPaymentClient3 client(token);

        auto vehicles = client.GetVehicles();
        UNIT_ASSERT(!vehicles.empty());

        auto sessions = client.GetSessions();

        NJson::TJsonValue parkingJson;
        parkingJson["name"] = "fake";
        parkingJson["aggregatorId"] = 2;
        parkingJson["id"] = 4007;
        parkingJson["coordinates"]["latitude"] = 55.77878425;
        parkingJson["coordinates"]["longitude"] = 37.68308326;
        NDrive::TParkingPaymentClient3::TParking parking;
        parking.FromJson(parkingJson);

        auto balance = client.GetBalance(parking);
        UNIT_ASSERT(balance.Amount > 0);
    }

    Y_UNIT_TEST(CyrillicLicencePlate) {
        UNIT_ASSERT_VALUES_EQUAL(NDrive::ToCyrillic("T208MT799"), "т208мт799");
        UNIT_ASSERT_VALUES_EQUAL(NDrive::ToLatin("к000кк000"), "K000KK000");
    }

    NDrive::TMskParkingPaymentConfig GetConfig() {
        TStringBuilder config;
        if (auto token = GetEnv("MskApiToken")) {
            config << "Token: " << token << Endl;
        }
        if (auto token = GetEnv("MskAccountToken")) {
            config << "AccountToken: " << token << Endl;
        }
        if (auto uri = GetEnv("MskAccountUri")) {
            config << "Uri: " << uri << Endl;
        }
        if (auto partnerId = GetEnv("MskAccountPartner")) {
            config << "PartnerId: " << partnerId << Endl;
        }
        DEBUG_LOG << config << Endl;
        return NDrive::TMskParkingPaymentConfig::ParseFromString(config);
    }

    Y_UNIT_TEST(AmppApi) {
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        const auto config = GetConfig();
        const NDrive::TMskParkingPaymentClient client(config);
        {
            auto future = client.GetBalance();
            UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            DEBUG_LOG << "Got balance: " << future.GetValue().Amount << Endl;
        }
        const TString testCarId = "к000кк000";
        {
            const TString parkingId = "0302";
            const TDuration testDuration = TDuration::Hours(1);
            auto future = client.StartParking(testCarId, parkingId, testDuration);
            UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        }
        {
            auto future = client.StopParking(testCarId);
            UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        }
    }

    Y_UNIT_TEST(AmppApiGetCost) {
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        const auto config = GetConfig();
        const NDrive::TMskParkingPaymentClient client(config);
        auto future = client.GetCosts();
        UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
        UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        auto map = future.ExtractValue();
        UNIT_ASSERT(!map.empty());
        auto it = map.FindPtr("0302");
        UNIT_ASSERT(it);
        UNIT_ASSERT_VALUES_EQUAL(*it, 38000);
    }
}
