#include "money.h"

#include <drive/library/cpp/openssl/oio.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/xml/sax/simple.h>

#include <util/string/cast.h>

namespace {
    class TMoneyResponseParser : public NXml::ISimpleSaxHandler {
    public:
        void OnStartElement(const TStringBuf& name, const TAttr* attrs, size_t count) override {
            if (name != GetSectionName()) {
                return;
            }
            for (size_t i = 0; i < count; ++i) {
                const TAttr& attr = attrs[i];
                if ("status" == attr.Name) {
                    auto value = FromString<ui32>(attr.Value);
                    Code = static_cast<NDrive::TYandexMoneyClient::ECode>(value);
                    continue;
                }
                if ("error" == attr.Name) {
                    Error = FromString<i64>(attr.Value);
                    continue;
                }
                ProcessAttr(attr.Name, attr.Value);
            }
        }
        void OnEndElement(const TStringBuf& /*name*/) override {}
        void OnText(const TStringBuf& /*data*/) override {}

        virtual TString GetSectionName() const = 0;
        virtual void ProcessAttr(const TStringBuf name, const TStringBuf value) = 0;

    protected:
        NDrive::TYandexMoneyClient::ECode Code = NDrive::TYandexMoneyClient::Unknown;
        i64 Error = 0;
    };

    class TPaymentReservationResponseParser: public NXml::ISimpleSaxHandler {
    public:
        const NDrive::TYandexMoneyClient::TReservation GetReservation() const {
            NDrive::TYandexMoneyClient::TReservation reservation;
            reservation.ContractId = ContractId;
            reservation.Status = Status;
            return reservation;
        }

        void OnStartElement(const TStringBuf& name, const TAttr* attrs, size_t count) override {
            if (name != "paymentReservationResponse") {
                return;
            }
            for (size_t i = 0; i < count; ++i) {
                const TAttr& attr = attrs[i];
                if ("status" == attr.Name) {
                    Status = attr.Value;
                    continue;
                }
                if ("contractId" == attr.Name) {
                    ContractId = attr.Value;
                    continue;
                }
            }
        }
        void OnEndElement(const TStringBuf& /*name*/) override {
        }
        void OnText(const TStringBuf& /*data*/) override {
        }

    private:
        TString ContractId;
        TString Status;
    };

    class TMakeDepositionResponseParser : public TMoneyResponseParser {
    public:
        const NDrive::TYandexMoneyClient::TDeposit GetDeposit() const {
            NDrive::TYandexMoneyClient::TDeposit deposit;
            deposit.Code = Code;
            deposit.Error = Error;
            deposit.Message = Message;
            return deposit;
        }

        TString GetSectionName() const override {
            return "makeDepositionResponse";
        }

        void ProcessAttr(const TStringBuf name, const TStringBuf value) override {
            if (name == "techMessage") {
                Message = value;
            }
        }
    private:
        TString Message;
    };

    class TBalanceResponseParser : public TMoneyResponseParser {
    public:
        const NDrive::TYandexMoneyClient::TBalance GetBalance() const {
            NDrive::TYandexMoneyClient::TBalance balance;
            balance.Code = Code;
            balance.Error = Error;
            balance.Value = Value;
            return balance;
        }

        TString GetSectionName() const override {
            return "balanceResponse";
        }

        void ProcessAttr(const TStringBuf name, const TStringBuf value) override {
            if (name == "balance") {
                Value = FromString<double>(value);
            }
        }
    private:
        double Value = -1;
    };
}

namespace NDrive {
    TYandexMoneyClient::TYandexMoneyClient(const TOptions& options)
        : Options(options)
        , ReservationClient(options.ReservationHost, options.ReservationPort, options.Timeout, options.Timeout)
        , DepositionClient(options.DepositionHost, options.DepositionPort, options.Timeout, options.Timeout)
        , Certificate(NOpenssl::GetCertificateFromFile(options.CertificateFile))
        , PrivateKey(NOpenssl::GetPrivateKeyFromFile(options.PrivateKeyFile, options.PrivateKeyPassword))
    {
        Y_ENSURE(Certificate, "cannot open certificate " << Options.CertificateFile);
        Y_ENSURE(PrivateKey, "cannot open private key " << Options.PrivateKeyFile);
        ReservationClient.DisableVerificationForHttps();
        DepositionClient.DisableVerificationForHttps();
        TOpenSslClientIO::TOptions::TClientCert cert;
        cert.CertificateFile_ = Options.CertificateFile;
        cert.PrivateKeyFile_ = Options.PrivateKeyFile;
        cert.PrivateKeyPassword_ = Options.PrivateKeyPassword;
        ReservationClient.SetClientCertificate(cert);
        DepositionClient.SetClientCertificate(cert);
    }

    std::pair<ui32, TYandexMoneyClient::TReservation> TYandexMoneyClient::PaymentReservation(const TString& orderNumber, double sum) {
        TStringStream ss;
        ss << "shopId=" << Options.ShopId;
        ss << "&shopArticleId=" << Options.ShopArticleId;
        ss << "&sum=" << sum;
        ss << "&order_number=" << orderNumber;
        const TString& data = ss.Str();
        const TString& request = "/webservice/ext/api/paymentReservation";
        TKeepAliveHttpClient::THeaders headers;
        headers.emplace("Content-Type", "application/x-www-form-urlencoded");

        auto response = Fetch(ReservationClient, request, data, false, std::move(headers));
        TStringInput input(response.second);

        TPaymentReservationResponseParser parser;
        NXml::Parse(input, &parser);

        return std::make_pair(response.first, parser.GetReservation());
    }


    std::pair<ui32, TYandexMoneyClient::TDeposit> TYandexMoneyClient::MakeDeposition(const TString& contractId, const TString& accountId, const TDepositionOptions& options, double sum) {
        TStringStream ss;
        ss << "<makeDepositionRequest";
        ss << " agentId=" << '"' << options.AgentId << '"';
        ss << " clientOrderId=" << '"' << options.OrderId << '"';
        ss << " requestDT=" << '"' << options.Timestamp.FormatGmTime("%Y-%m-%dT%H:%M:%S") << '.' << Sprintf("%03dZ", options.Timestamp.MilliSecondsOfSecond()) << '"';
        ss << " dstAccount=" << '"' << contractId << '"';
        ss << " amount=" << '"' << Sprintf("%.2f", sum) << '"';
        ss << " currency=" << '"' << 643 << '"';
        ss << " contract=" << '"' << options.Contract << '"';
        ss << ">";
        ss << "<paymentParams>";
        ss << "<cps_ym_account>";
        ss << accountId;
        ss << "</cps_ym_account>";
        if (options.OfferAccepted) {
            ss << "<pof_offerAccepted>1</pof_offerAccepted>";
        }
        if (options.CustomerNumber) {
            ss << "<customerNumber>";
            ss << options.CustomerNumber;
            ss << "</customerNumber>";
        }
        ss << "</paymentParams>";
        ss << "</makeDepositionRequest>";

        const TString& request = "/webservice/deposition/api/makeDeposition";
        TKeepAliveHttpClient::THeaders headers;
        headers.emplace("Content-Type", "application/pkcs7-mime");

        auto response = Fetch(DepositionClient, request, ss.Str(), true, std::move(headers));
        TStringInput input(response.second);
        TMakeDepositionResponseParser parser;
        NXml::Parse(input, &parser);
        return std::make_pair(response.first, parser.GetDeposit());
    }

    std::pair<ui32, TYandexMoneyClient::TBalance> TYandexMoneyClient::Balance(ui64 clientOrderId) {
        auto now = Now();
        TStringStream ss;
        ss << "<balanceRequest";
        ss << " agentId=" << '"' << Options.AgentId << '"';
        ss << " clientOrderId=" << '"' << clientOrderId << '"';
        ss << " requestDT=" << '"' << now.FormatGmTime("%Y-%m-%dT%H:%M:%S") << '.' << Sprintf("%03dZ", now.MilliSecondsOfSecond()) << '"';
        ss << "/>";

        const TString& request = "/webservice/deposition/api/balance";
        TKeepAliveHttpClient::THeaders headers;
        headers.emplace("Content-Type", "application/pkcs7-mime");

        auto response = Fetch(DepositionClient, request, ss.Str(), true, std::move(headers));
        TStringInput input(response.second);

        TBalanceResponseParser parser;
        NXml::Parse(input, &parser);
        return std::make_pair(response.first, parser.GetBalance());
    }


    TYandexMoneyClient::TReservation TYandexMoneyClient::PaymentReservationSafe(const TString& orderNumber, double sum) {
        auto reply = PaymentReservation(orderNumber, sum);
        Y_ENSURE(!IsServerError(reply.first) && !IsUserError(reply.first), "fetch error: " << reply.first);
        Y_ENSURE(reply.second.Status == "SUCCESS", "bad PaymentReservation status: " << reply.second.Status);
        return reply.second;
    }


    TYandexMoneyClient::TDeposit TYandexMoneyClient::MakeDepositionSafe(const TString& contractId, const TString& accountId, const TDepositionOptions& options, double sum) {
        auto reply = MakeDeposition(contractId, accountId, options, sum);
        Y_ENSURE(!IsServerError(reply.first) && !IsUserError(reply.first), "fetch error: " << reply.first);
        return reply.second;
    }


    TYandexMoneyClient::TBalance TYandexMoneyClient::BalanceSafe(ui64 clientOrderId) {
        auto reply = Balance(clientOrderId);
        Y_ENSURE(!IsServerError(reply.first) && !IsUserError(reply.first), "fetch error: " << reply.first);
        return reply.second;
    }

    TString TYandexMoneyClient::SignRequestData(const TString& data) const {
        NOpenssl::TCMSPtr outgoing = NOpenssl::FormSignedCMS(data, Certificate, PrivateKey);
        Y_ENSURE(outgoing, "cannot sign package");
        return NOpenssl::PrintPEM(outgoing, data);
    }

    TString TYandexMoneyClient::DecodeSignedResponse(const TString& signedResponse) const {
        TString response;
        if (signedResponse.find("PKCS7") != TString::npos) {
            NOpenssl::TPKCS7Ptr incoming = NOpenssl::ReadSignedData(signedResponse);
            response = ToString(NOpenssl::GetData(incoming));
        } else if (signedResponse.find("CMS") != TString::npos) {
            NOpenssl::TCMSPtr incoming = NOpenssl::ReadSignedCMS(signedResponse);
            TString response = ToString(NOpenssl::GetData(incoming));
        } else {
            response = signedResponse;
        }
        return response;
    }

    std::pair<ui32, TString> TYandexMoneyClient::Fetch(TKeepAliveHttpClient& client, const TString& request, const TString& data, bool sign, TKeepAliveHttpClient::THeaders&& headers) const {
        TStringStream ss;
        DEBUG_LOG << "Requesting " << request << " " << data << Endl;
        TString signedReques = sign ? SignRequestData(data) : data;
        ui32 code = client.DoPost(request, signedReques, &ss, headers);
        TString response = sign ? DecodeSignedResponse(ss.Str()) : ss.Str();
        DEBUG_LOG << "Response " << request << " " << data << ": " << code << " " << response << Endl;
        return std::make_pair(code, response);
    }
}
