#pragma once

#include <drive/library/cpp/openssl/types.h>

#include <library/cpp/http/simple/http_client.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

namespace NDrive {
    class TYandexMoneyClient {
    public:
        enum ECode {
            Success = 0,
            Processing = 1,
            Declined = 3,
            Unknown = 128,
        };

        struct TOptions {
            TString ReservationHost = "https://shop.yookassa.ru";
            ui16 ReservationPort = 443;
            TString DepositionHost = "https://deposit.yookassa.ru";
            ui16 DepositionPort = 9094;
            TString CertificateFile;
            TString PrivateKeyFile;
            TString PrivateKeyPassword;
            TString AgentId;
            TString ShopId;
            TString ShopArticleId;
            TDuration Timeout = TDuration::Seconds(10);
        };

        struct TReservation {
            TString Status;
            TString ContractId;
        };

        struct TDeposit {
            ECode Code = Unknown;
            i64 Error = 0;
            TString Message;
        };

        struct TBalance : public TDeposit {
            double Value = -1;
        };

        struct TDepositionOptions {
            TString AgentId;
            TString CustomerNumber;
            ui64 OrderId;
            TString ParkingAccount;
            TString Contract;
            TInstant Timestamp = Now();
            bool OfferAccepted = true;

        public:
            TDepositionOptions(const TString& agentId, ui64 orderId)
                : AgentId(agentId)
                , OrderId(orderId)
            {}
        };

    public:
        TYandexMoneyClient(const TOptions& options);

        std::pair<ui32, TReservation> PaymentReservation(const TString& orderNumber, double sum);
        std::pair<ui32, TDeposit> MakeDeposition(const TString& contractId, const TString& accountId, const TDepositionOptions& options, double sum);
        std::pair<ui32, TBalance> Balance(ui64 clientOrderId);

        TReservation PaymentReservationSafe(const TString& orderNumber, double sum);
        TDeposit MakeDepositionSafe(const TString& contractId, const TString& accountId, const TDepositionOptions& options, double sum);
        TBalance BalanceSafe(ui64 clientOrderId);

    private:
        TString SignRequestData(const TString& data) const;
        TString DecodeSignedResponse(const TString& signedResponse) const;
        std::pair<ui32, TString> Fetch(TKeepAliveHttpClient& client, const TString& request, const TString& data, bool sign, TKeepAliveHttpClient::THeaders&& headers = {}) const;

    private:
        const TOptions Options;
        TKeepAliveHttpClient ReservationClient;
        TKeepAliveHttpClient DepositionClient;

        NOpenssl::TX509Ptr Certificate;
        NOpenssl::TEVP_PKEYPtr PrivateKey;
    };
}
