#include "client.h"

#include <util/string/cast.h>
#include <util/string/split.h>

namespace {
    constexpr ui32 ParkingObjectType = 2002;
    constexpr ui32 ParkingZoneObjectType = 2010;

    const TGeoRect KznRectangle = { 48.829293, 55.684190, 49.355482, 55.928310 };
    const TGeoRect SpbRectangle = { 29.94006121, 59.78484664, 30.69674456, 60.13056264 };

    i64 GetParkingAggregatorId(const NGraph::TRouter::TGeoObjectInfo& object) {
        for (auto&& coordinate : object.Coordinates) {
            if (KznRectangle.Contain(coordinate)) {
                return NDrive::TParkingListClient::KznAggregatorId;
            }
            if (SpbRectangle.Contain(coordinate)) {
                return NDrive::TParkingListClient::SpbAggregatorId;
            }
        }
        return NDrive::TParkingListClient::MskAggregatorId;
    }

    TString GetParkingName(const NGraph::TRouter::TGeoObjectInfo& object) {
        for (auto&& name : object.ParentNames) {
            if (name.Value) {
                return name.Value;
            }
        }
        return {};
    }

    TString GetParkingId(const NGraph::TRouter::TGeoObjectInfo& object) {
        for (auto&& name : object.ParentNames) {
            for (auto&& i : StringSplitter(name.Value).Split(' ')) {
                auto token = i.Token();
                if (IsNumber(token)) {
                    return TString(token);
                }
            }
        }
        return {};
    }
}

NThreading::TFuture<NDrive::TParkingListClient::TParkings> NDrive::TParkingListClient::GetParkings(double latitude, double longitude, double searchRadius) const {
    TGeoCoord coordinate = { longitude, latitude };

    NGraph::TRouter::TObjectRoutingOptions options;
    options.Perm = {};
    options.DistancePrecision = 250;
    options.Limit = searchRadius;
    options.Types = { ParkingObjectType };
    options.NameType = NGraph::NRouter::EObjectNameType::Official;
    auto objects = Router.GetObjects(coordinate, options);
    std::sort(objects.begin(), objects.end(), [](const NGraph::TRouter::TGeoObjectInfo& left, const NGraph::TRouter::TGeoObjectInfo& right) {
        return left.Distance < right.Distance;
    });

    TParkings result;
    TSet<TString> parkingNames;
    for (auto&& object : objects) {
        if (object.ParentType != ParkingZoneObjectType) {
            continue;
        }
        auto id = GetParkingId(object);
        if (!id) {
            continue;
        }
        if (!parkingNames.emplace(id).second) {
            continue;
        }

        TParking parking;
        parking.AggregatorId = GetParkingAggregatorId(object);
        parking.Name = GetParkingName(object);
        parking.ParkingId = id;
        if (!object.Coordinates.empty()) {
            parking.Latitude = object.Coordinates.front().Y;
            parking.Longitude = object.Coordinates.front().X;
        }
        result.push_back(std::move(parking));
    }
    return NThreading::MakeFuture(std::move(result));
}
