#include "client.h"

#include <drive/backend/logging/events.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/unistat/cache.h>

class TPassportClient::TCallback : public NNeh::THttpAsyncReport::THandlerCallback {
private:
    using TBase = NNeh::THttpAsyncReport::THandlerCallback;

private:
    R_READONLY(TString, SessionId);
    R_READONLY(TString, DeviceId);
    R_READONLY(TString, RequestType);

public:
    TCallback(IThreadPool& handler, const TString& sessionId, const TString& deviceId, const TString& reqType, NThreading::TPromise<void> promise)
        : TBase(handler)
        , SessionId(sessionId)
        , DeviceId(deviceId)
        , RequestType(reqType)
        , Promise(std::move(promise))
    {}

    void Process(void*) final {
        CHECK_WITH_LOG(Reports.size() == 1);
        auto result = Reports.front();
        //    TDuration dt = Now() - StartTime;

        TUnistatSignalsCache::SignalAdd("passport-session-reply_code", ::ToString(result.GetHttpCode()), 1);
        if (result.HasReport()) {
            NJson::TJsonValue json(NJson::JSON_NULL);
            if (!NJson::ReadJsonFastTree(*result.GetReport(), &json)) {
                ProcessError(result, RequestType);
                return;
            }

            if (json.Has("status") && json["status"].GetString() == "ok") {
                ProcessSuccess(result, RequestType);
            } else {
                ProcessError(result, RequestType);
            }
        }
    }

private:
    void ProcessError(const THttpReplyData<TString>& result, const TString& requestType) {
        NDrive::TEventLog::Log("PassportRequest", NJson::TMapBuilder
            ("http_code", result.GetHttpCode())
            ("reply", *result.GetReport())
            ("session_id", SessionId)
            ("device_id", DeviceId)
            ("req_type", requestType)
        );
        TUnistatSignalsCache::SignalAdd("passport-session-" + requestType, "error", 1);
        NJson::TJsonValue error;
        error["type"] = requestType;
        error["http_code"] = result.GetHttpCode();
        if (auto report = result.GetReport()) {
            error["report"] = NJson::ToJson(NJson::JsonString(*report));
        }
        Promise.SetException(error.GetStringRobust());
    }

    void ProcessSuccess(const THttpReplyData<TString>& result, const TString& requestType) {
        NDrive::TEventLog::Log("PassportRequest", NJson::TMapBuilder
            ("http_code", result.GetHttpCode())
            ("reply", *result.GetReport())
            ("session_id", SessionId)
            ("device_id", DeviceId)
            ("req_type", requestType)
        );
        TUnistatSignalsCache::SignalAdd("passport-session-" + requestType, "success", 1);
        Promise.SetValue();
    }

private:
    NThreading::TPromise<void> Promise;
};

void TPassportClientConfig::Init(const TYandexConfig::Section* section, const TMap<TString, NSimpleMeta::TConfig>* requestPolicy) {
    TSimpleAsyncRequestSender::TConfig::Init(section, requestPolicy);

    RequestTimeout = section->GetDirectives().Value<TDuration>("RequestTimeout", RequestTimeout);
    SelfClientId = section->GetDirectives().Value<ui32>("SelfClientId", SelfClientId);
    DestinationClientId = section->GetDirectives().Value<ui32>("DestinationClientId", DestinationClientId);
    Threads = section->GetDirectives().Value("Threads", Threads);
    StartApiPath = section->GetDirectives().Value("StartApiPath", StartApiPath);
    StopApiPath = section->GetDirectives().Value("StopApiPath", StopApiPath);
    AssertCorrectConfig(SelfClientId > 0, "SelfClientId required");
    AssertCorrectConfig(DestinationClientId > 0, "DestinationClientId required");
}

void TPassportClientConfig::ToString(IOutputStream& os) const {
    TSimpleAsyncRequestSender::TConfig::ToString(os);
    os << "Threads: " << Threads << Endl;
    os << "SelfClientId: " << SelfClientId << Endl;
    os << "DestinationClientId: " << DestinationClientId << Endl;
    os << "RequestTimeout: " << RequestTimeout << Endl;
    os << "StartApiPath: " << StartApiPath << Endl;
    os << "StopApiPath: " << StopApiPath << Endl;
}

NThreading::TFuture<void> TPassportClient::StartSessionAsync(const TString& userTicket, const TString& clientIp, const TString& sessionId, const TString& deviceId) const {
    TStringStream request;
    request << "drive_session_id=" << sessionId;
    request << "&drive_device_id= " << deviceId;

    auto req = CreateRequest(userTicket, clientIp, request.Str());
    req.SetUri(Config.GetStartApiPath() + "?consumer=" + Config.GetConsumer());

    const TInstant deadline = Now() + Config.GetRequestTimeout();
    auto promise = NThreading::NewPromise<void>();
    auto guard = Impl->Send(req, deadline, new TCallback(RepliesHandler, sessionId, deviceId, "start", promise));
    return promise.GetFuture();
}


NThreading::TFuture<void> TPassportClient::StopSessionAsync(const TString& userTicket, const TString& clientIp, const TString& sessionId, const TString& deviceId) const {
    TStringStream request;
    request << "drive_session_id=" << sessionId;
    request << "&drive_device_id= " << deviceId;

    auto req = CreateRequest(userTicket, clientIp, request.Str());
    req.SetUri(Config.GetStopApiPath() + "?consumer=" + Config.GetConsumer());

    const TInstant deadline = Now() + Config.GetRequestTimeout();
    auto promise = NThreading::NewPromise<void>();
    auto guard = Impl->Send(req, deadline, new TCallback(RepliesHandler, sessionId, deviceId, "stop", promise));
    return promise.GetFuture();
}

NNeh::THttpRequest TPassportClient::CreateRequest(const TString& userTicket, const TString& clientIp, const TString& postData /*=""*/) const {
    NNeh::THttpRequest req;
    req.AddHeader("X-Ya-User-Ticket", userTicket);
    req.AddHeader("Ya-Consumer-Client-Ip", clientIp);
    req.AddHeader("Content-Type", "application/x-www-form-urlencoded");
    try {
        req.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Config.GetDestinationClientId()));
    } catch (const std::exception& e) {
        ERROR_LOG << FormatExc(e) << Endl;
        req.AddHeader("X-Ya-Service-Ticket", "");
    }

    if (!postData) {
        req.SetRequestType("GET");
    } else {
        req.SetPostData(TBlob::FromString(postData)).SetRequestType("POST");
    }
    return req;
}
