#include "client.h"

#include <rtline/library/unistat/cache.h>

#include <library/cpp/string_utils/quote/quote.h>


namespace {
    class TApplier {
    private:
        TString Action;

    public:
        TApplier(const TString& action)
            : Action(action)
        {
        }

        auto operator()(const NThreading::TFuture<NUtil::THttpReply>& r) const {
            const auto& report = r.GetValue();
            TUnistatSignalsCache::SignalAdd("redis_proxy-" + Action + "-codes", ToString(report.Code()), 1);
            if (auto code = report.Code(); report.IsSuccessReply() || code == 201) {
                return report.Content();
            }
            throw yexception() << report.Serialize().GetStringRobust();
        }
    };
}

namespace NDrive::NRedis {

    TClient::TClient(const TConfig& config)
        : Config(config)
        , Client(MakeAtomicShared<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig()))
    {
    }

    const TConfig& TClient::GetConfig() const {
        return Config;
    }

    NThreading::TFuture<TString> TClient::Get(const TString& key) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/" + UrlEscapeRet(key, true));
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier("get"));
    }

    NThreading::TFuture<TString> TClient::Set(const TString& key, const TString& value, const TDuration ttl) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/" + UrlEscapeRet(key, true));
        request.SetPostData(value);
        if (ttl) {
            request.SetCgiData("ttl_seconds=" + ToString(ttl.Seconds()));
        }
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply(TApplier("set"));
    }

    NThreading::TFuture<bool> TClient::Del(const TString& key) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/" + UrlEscapeRet(key, true));
        request.SetRequestType("DELETE");
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout())
            .Apply(TApplier("del"))
            .Apply([](const NThreading::TFuture<TString>& r) { return IsTrue(r.GetValue()); });
    }
}
