#include <drive/library/cpp/redis/client.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/vector.h>
#include <util/string/builder.h>
#include <util/system/env.h>


Y_UNIT_TEST_SUITE(RedisProxySuite) {
    TMaybe<NDrive::NRedis::TConfig> GetConfig() {
        const TString uri = GetEnv("RedisUri");
        if (!uri) {
            return {};
        }
        auto config = TStringBuilder()
            << "Uri: " << uri << Endl
            << "<RequestConfig>" << Endl
            << "    MaxAttempts: 1" << Endl
            << "</RequestConfig>" << Endl
            << "RequestTimeout: 1s" << Endl;
        if (auto path = GetEnv("RedisUriPath")) {
            config << "UriPath: " << path << Endl;
        }
        DEBUG_LOG << config << Endl;
        auto result = NDrive::NRedis::TConfig::ParseFromString(config);
        UNIT_ASSERT(!!result);
        return *result;
    }

    Y_UNIT_TEST(Simple) {
        auto config = GetConfig();
        if (!config) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        const NDrive::NRedis::TClient client(*config);
        const TString testKey = GetEnv("RedisTestKey", "test_key");
        if (IsTrue(GetEnv("RedisCleanOnStart"))) {
            UNIT_ASSERT(client.Del(testKey).Wait(config->GetRequestTimeout()));
            auto future = client.Get(testKey);
            UNIT_ASSERT(future.Wait(config->GetRequestTimeout()));
            UNIT_ASSERT(!future.HasValue());
        }
        const TString testValue = GetEnv("RedisTestValue", "test_value");
        {
            auto future = client.Set(testKey, testValue);
            UNIT_ASSERT(future.Wait(config->GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            UNIT_ASSERT_VALUES_EQUAL(future.GetValue(), testValue);
        }
        {
            auto future = client.Get(testKey);
            UNIT_ASSERT(future.Wait(config->GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            UNIT_ASSERT_VALUES_EQUAL(future.GetValue(), testValue);
        }
    }

    Y_UNIT_TEST(TTL) {
        auto config = GetConfig();
        if (!config) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        const NDrive::NRedis::TClient client(*config);
        const TString testKey = GetEnv("RedisTestKey", "test_key");
        const TString testValue = GetEnv("RedisTestValue", "test_value");
        {
            auto future = client.Set(testKey, testValue, TDuration::Seconds(5));
            UNIT_ASSERT(future.Wait(config->GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            UNIT_ASSERT_VALUES_EQUAL(future.GetValue(), testValue);
        }
        {
            auto future = client.Get(testKey);
            UNIT_ASSERT(future.Wait(config->GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            UNIT_ASSERT_VALUES_EQUAL(future.GetValue(), testValue);
        }
        Sleep(TDuration::Seconds(7));
        {
            auto future = client.Get(testKey);
            UNIT_ASSERT(future.Wait(config->GetRequestTimeout()));
            UNIT_ASSERT(!future.HasValue());
        }
    }
}
