#include "client.h"

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>

#include <library/cpp/json/json_reader.h>

#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/string_utils/url/url.h>

#include <util/string/builder.h>
#include <util/string/vector.h>


namespace {
    template <class TResult>
    class TDefautResultConstructor {
    public:
        TResult operator()() const {
            return TResult();
        }
    };

    class TCalculateKaskoResult {
        TString RequiredPack;
    public:
        TCalculateKaskoResult(const TString& requiredPack)
            : RequiredPack(requiredPack)
        {
        }

        NDrive::NRenins::NKasko::TCalculateOrderResult operator()() const {
            return NDrive::NRenins::NKasko::TCalculateOrderResult(RequiredPack);
        }
    };

    template <class TResult, class TResultConstructor = TDefautResultConstructor<TResult>>
    class TApplier {
    private:
        TString Action;
        TResultConstructor ResultConstructor;

    public:
        TApplier(const TString& action, TResultConstructor&& resultConstructor = {})
            : Action(action)
            , ResultConstructor(resultConstructor)
        {
        }

        auto operator()(const NThreading::TFuture<NUtil::THttpReply>& r) const {
            const auto& report = r.GetValue();
            TUnistatSignalsCache::SignalAdd("renins-" + Action + "-codes", ToString(report.Code()), 1);
            NJson::TJsonValue rawResult;
            if (!NJson::ReadJsonFastTree(report.EnsureSuccessfulReply().Content(), &rawResult)) {
                ythrow yexception() << "Error getting objects(" << Action << "): " << report.GetDebugReply();
            } else if (!rawResult["isSuccess"].GetBooleanRobust()) {
                for (auto&& message : rawResult["messages"].GetArray()) {
                    if (message["level"].GetString() == "Error") {
                        ythrow NDrive::NRenins::TKaskoException(message["code"].GetString(), message["text"].GetString());
                    }
                }
                ythrow yexception() << "Error getting objects(" << Action << ") by renins flag: " << report.GetDebugReply();
            }
            auto result = ResultConstructor();
            if (!result.Parse(rawResult)) {
                ythrow yexception() << "Fail to parse reply(" << Action << "): " << rawResult.GetStringRobust();
            }
            return NThreading::MakeFuture(std::move(result));
        }
    };
}

namespace NDrive::NRenins {

    TInsuranceClient::TInsuranceClient(const TReninsConfig& config)
        : Config(config)
        , Client(MakeAtomicShared<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig()))
    {
    }

    const TReninsConfig& TInsuranceClient::GetConfig() const {
        return Config;
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceClient::Calculate(const NJson::TJsonValue& context) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/calculate");
        request.SetContentType("application/json-patch+json");
        request.AddHeader("accept", "text/plain");
        request.SetPostData(NJson::TMapBuilder("version", "1.0")("model", context));
        Config.Authorize(request);
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceClient::Order(const TString& orderId, const NJson::TJsonValue& context) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/import");
        request.SetContentType("application/json-patch+json");
        request.AddHeader("accept", "text/plain");
        request.SetCgiData("accountNumber=" + orderId);
        request.SetPostData(NJson::TMapBuilder("version", "1.0")("model", context));
        Config.Authorize(request);
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceClient::GetStatus(const TString& orderId) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/getAccountStatus");
        request.SetContentType("application/json-patch+json");
        request.AddHeader("accept", "text/plain");
        request.SetCgiData("accountNumber=" + orderId);
        Config.Authorize(request);
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceClient::GetPaymentLink(const TMainOrderData& orderData) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/getPaymentLink");
        request.SetContentType("application/json-patch+json");
        request.AddHeader("accept", "text/plain");
        request.SetCgiData(TStringBuilder()
            << "accountNumber=" + orderData.OrderId
            << "&successUrl=" + UrlEscapeRet(orderData.SucceedPaymentUrl, true)
            << "&failUrl=" + UrlEscapeRet(orderData.FailedPaymentUrl, true)
        );
        Config.Authorize(request);
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceClient::GetDocument(const TString& orderId, const TString& dealId) const {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUriPath() + "/printDocument");
        request.SetContentType("application/json-patch+json");
        request.AddHeader("accept", "text/plain");
        request.SetCgiData(TStringBuilder()
            << "accountNumber=" + orderId
            << "&dealId=" + dealId
            << "&failUrl=PolicyOriginal"
        );
        Config.Authorize(request);
        DEBUG_LOG << request.GetDebugRequest() << Endl;
        return Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    }

    TInsuranceSyncClient::TInsuranceSyncClient(const TReninsConfig& config)
        : Config(config)
    {
        TStringBuf scheme;
        TStringBuf host;
        if (TryGetSchemeHostAndPort(Config.GetUri(), scheme, host, Port)) {
            Https = scheme == "https://";
            Host = host;
        }
    }

    const TReninsConfig& TInsuranceSyncClient::GetConfig() const {
        return Config;
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceSyncClient::Calculate(const NJson::TJsonValue& context) const {
        const TString command = TStringBuilder()
            << Config.GetUriPath()
            << "/calculate";
        NJson::TJsonValue json = NJson::TMapBuilder("version", "1.0")("model", context);
        NUtil::THttpRequest request(command, json.GetStringRobust());
        request.SetIsHttps(IsHttps());
        request.SetContentType("application/json-patch+json");
        request.SetHeader("accept", "text/plain");
        request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
        request.SetHeader("Authorization", "Bearer " + Config.GetToken());
        return NThreading::MakeFuture(request.Execute(GetHost(), GetPort()));
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceSyncClient::Order(const TString& orderId, const NJson::TJsonValue& context) const {
        const TString command = TStringBuilder()
            << Config.GetUriPath()
            << "/import"
            << "?accountNumber=" << orderId;
        NJson::TJsonValue json = NJson::TMapBuilder("version", "1.0")("model", context);
        NUtil::THttpRequest request(command, json.GetStringRobust());
        request.SetIsHttps(IsHttps());
        request.SetContentType("application/json-patch+json");
        request.SetHeader("accept", "text/plain");
        request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
        request.SetHeader("Authorization", "Bearer " + Config.GetToken());
        return NThreading::MakeFuture(request.Execute(GetHost(), GetPort()));
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceSyncClient::GetStatus(const TString& orderId) const {
        const TString command = TStringBuilder()
            << Config.GetUriPath()
            << "/getAccountStatus"
            << "?accountNumber=" << orderId;
        NUtil::THttpRequest request(command);
        request.SetIsHttps(IsHttps());
        request.SetContentType("application/json-patch+json");
        request.SetHeader("accept", "text/plain");
        request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
        request.SetHeader("Authorization", "Bearer " + Config.GetToken());
        return NThreading::MakeFuture(request.Execute(GetHost(), GetPort()));
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceSyncClient::GetPaymentLink(const TMainOrderData& orderData) const {
        const TString command = TStringBuilder()
            << Config.GetUriPath()
            << "/getPaymentLink"
            << "?accountNumber=" << orderData.OrderId
            << "&successUrl=" << UrlEscapeRet(orderData.SucceedPaymentUrl, true)
            << "&failUrl=" << UrlEscapeRet(orderData.FailedPaymentUrl, true);
        NUtil::THttpRequest request(command);
        request.SetIsHttps(IsHttps());
        request.SetContentType("application/json-patch+json");
        request.SetHeader("accept", "text/plain");
        request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
        request.SetHeader("Authorization", "Bearer " + Config.GetToken());
        return NThreading::MakeFuture(request.Execute(GetHost(), GetPort()));
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceSyncClient::GetDocument(const TString& orderId, const TString& dealId) const {
        const TString command = TStringBuilder()
            << Config.GetUriPath()
            << "/printDocument"
            << "?accountNumber=" << orderId
            << "&dealId=" + dealId
            << "&printDocumentType=PolicyOriginal";
        NUtil::THttpRequest request(command);
        request.SetIsHttps(IsHttps());
        request.SetContentType("application/json-patch+json");
        request.SetHeader("accept", "text/plain");
        request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
        request.SetHeader("Authorization", "Bearer " + Config.GetToken());
        return NThreading::MakeFuture(request.Execute(GetHost(), GetPort()));
    }

    NThreading::TFuture<NUtil::THttpReply> TInsuranceSyncClient::GetAutocodeData(const TString& carNumber) const {
        const TString command = TStringBuilder()
            << Config.GetUriPath()
            << "/getAvtoCod"
            << "?vehicleNumber=" << UrlEscapeRet(carNumber)
            << "&vehicleNumberType=RegNumber";
        NUtil::THttpRequest request(command, "{}");
        request.SetIsHttps(IsHttps());
        request.SetHeader("accept", "text/plain");
        request.SetTimeout(Config.GetRequestTimeout().MilliSeconds());
        request.SetHeader("Authorization", "Bearer " + Config.GetToken());
        return NThreading::MakeFuture(request.Execute(GetHost(), GetPort()));
    }

    namespace NOsago {
        NThreading::TFuture<TCalculateOrderResult> TClient::Calculate(const TCalculateOrder& context) const {
            return TBase::Calculate(context.GetReport()).Apply(TApplier<TCalculateOrderResult>("osago-calculate"));
        }
    }

    namespace NKasko {
        NThreading::TFuture<TCalculateOrderResult> TClient::Calculate(const TCalculateOrder& context, const TString& requiredPack) const {
            auto payload = context.GetReport();
            payload.InsertValue("isLead", true);
            return TBase::Calculate(payload).Apply(TApplier<TCalculateOrderResult, TCalculateKaskoResult>("kasko-calculate", TCalculateKaskoResult(requiredPack)));
        }

        NThreading::TFuture<TOrderStatus> TClient::Order(const TStartOrderData& context) const {
            return TBase::Order(context.GetOrderData().OrderId, context.GetReport()).Apply(TApplier<TOrderStatus>("kasko-import"));
        }

        NThreading::TFuture<TOrderStatus> TClient::GetStatus(const TString& orderId) const {
            return TBase::GetStatus(orderId).Apply(TApplier<TOrderStatus>("kasko-status"));
        }

        NThreading::TFuture<TString> TClient::GetPaymentLink(const TMainOrderData& orderData) const {
            return TBase::GetPaymentLink(orderData)
                .Apply(TApplier<TPaymentLink>("kasko-payment"))
                .Apply([](const NThreading::TFuture<TPaymentLink>& l) { return l.GetValue().GetLink(); });
        }

        NThreading::TFuture<TString> TClient::GetDocument(const TString& orderId, const TString& dealId) const {
            return TBase::GetDocument(orderId, dealId).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
                const auto& report = r.GetValue();
                TUnistatSignalsCache::SignalAdd("rennis-kasko-document-codes", ToString(report.Code()), 1);
                return std::move(report.EnsureSuccessfulReply().Content());
            });
        }

        NThreading::TFuture<TAutocodeCarData> TClient::GetAutocodeData(const TString& carNumber) const {
            return TBase::GetAutocodeData(carNumber).Apply(TApplier<TAutocodeCarData>("kasko-autocode"));
        }
    }
}
