#include "entities.h"

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/datetime.h>


namespace {
    const ui32 OWNER_POS = 2;
}


template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::TUserMainData& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "name", object.GetName());
    NJson::InsertField(json, "lastName", object.GetLastName());
    NJson::InsertField(json, "dateOfBirth", object.GetDateOfBirth().FormatLocalTime("%d.%m.%Y"));
    if (object.HasMiddleName()) {
        NJson::InsertField(json, "middleName", object.GetMiddleNameRef());
    }
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::TPersonPassport& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "number", object.Number);
    NJson::InsertField(json, "series", object.Series);
    NJson::InsertField(json, "issuedBy", object.IssuedBy);
    NJson::InsertField(json, "dateIssue", object.DateIssue.FormatLocalTime("%d.%m.%Y"));
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::TMainOrderData& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "onlinePaymentSuccessURL", object.SucceedPaymentUrl);
    NJson::InsertField(json, "onlinePaymentFailURL", object.FailedPaymentUrl);
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NOsago::TDriverData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetUserData());
    NJson::InsertField(json, "driverLicenseSeries", object.GetLicenseSeries());
    NJson::InsertField(json, "driverLicenseNumber", object.GetLicenseNumber());
    NJson::InsertField(json, "drivingExperienceDateStart", object.GetExperienceDateStart().FormatLocalTime("%d.%m.%Y"));
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NKasko::TDriverData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetUserData());
    NJson::InsertField(json, "drivingExperienceDateStart", object.GetExperienceDateStart().FormatLocalTime("%d.%m.%Y"));
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::TAddressData& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "country", object.GetCountryDef(""));
    NJson::InsertField(json, "region", object.GetRegionDef(""));
    NJson::InsertField(json, "city", object.GetCityDef(""));
    NJson::InsertField(json, "street", object.GetStreetDef(""));
    NJson::InsertField(json, "house", object.GetHouseDef(""));
    NJson::InsertField(json, "kladrId", TString()); // rennis bug
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NOsago::TPersonData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetUserData()) ;
    NJson::InsertField(json, "hasPersonalDataAgreement", object.GetPersonalAgreement());
    NJson::InsertField(json, "regionOfResidenceKladrId", object.GetRegoinKladrIdDef(""));
    NJson::InsertField(json, "addresses", object.GetAddresses());
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::TMainAutoData& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "registrationNumber", object.GetNumber());
    NJson::InsertField(json, "autoTypeCode", object.GetTypeCode());
    NJson::InsertField(json, "autoBrandCode", object.GetBrandCode());
    NJson::InsertField(json, "autoModelCode", object.GetModelCode());
    NJson::InsertField(json, "autoYearOfIssueCode", object.GetYearOfIssueCode());
    NJson::InsertField(json, "autoEnginePowerCode", object.GetEnginePowerCode());
    NJson::InsertNonNull(json, "bankOfCredit", object.GetBankOfCredit());
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NOsago::TAutoData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetMainAutoData());
    NJson::InsertField(json, "vin", object.GetVin());
    NJson::InsertField(json, "hasTrailer", object.GethasTrailer());
    NJson::InsertField(json, "isTransitAuto", object.GetTransitAuto());
    NJson::InsertField(json, "passengersQuantity", object.GetPassengersQuantity());
    NJson::InsertField(json, "autoEnginePowerCode", object.GetEnginePowerCode());
    NJson::InsertField(json, "purposeOfUsingCode", object.GetUsingCode());
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NKasko::TAutoData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetMainAutoData());
    NJson::InsertField(json, "autoBodyTypeCode", object.GetBodyTypeCode());
    NJson::InsertField(json, "autoCost", object.GetCost());
    NJson::InsertField(json, "hasMileage", object.GetMileage() > 0);
    NJson::InsertNonNull(json, "mileage", object.GetMileage());
    NJson::InsertField(json, "inCredit", object.GetInCredit());
    NJson::InsertField(json, "vin", object.GetVin());
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NKasko::TPersonData& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "addresses", object.GetAddresses());
    NJson::InsertField(json, "dateOfBirth", object.GetDateOfBirth().FormatLocalTime("%d.%m.%Y"));
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NKasko::TDriverOrderStartData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetUserData());
    NJson::InsertField(json, "driverLicenseSeries", object.GetLicenseSeries());
    NJson::InsertField(json, "driverLicenseNumber", object.GetLicenseNumber());
    NJson::InsertField(json, "drivingExperienceDateStart", object.GetExperienceDateStart().FormatLocalTime("%d.%m.%Y"));
    NJson::InsertField(json, "isMaried", object.IsMaried());
    NJson::InsertField(json, "sexCode", object.GetSexCode());
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NKasko::TPersonOrderStartData& object) {
    NJson::TJsonValue json = NJson::ToJson(object.GetUserData());
    NJson::InsertField(json, "phone1", object.GetPhone());
    NJson::InsertField(json, "email1", object.GetEmail());
    NJson::InsertField(json, "identityDocument", object.GetPassport());
    NJson::InsertField(json, "addresses", object.GetAddresses());
    return json;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NRenins::NKasko::TCarOrderStartData& object) {
    NJson::TJsonValue json;
    json.InsertValue("registrationNumber", object.GetNumber());
    json.InsertValue("vin", object.GetVin());
    auto& sts = json.InsertValue("tsIdentityCard", NJson::TMapBuilder("tsIdentityCardTypeCode", "sts"));
    sts.InsertValue("number", object.GetStsNumber());
    sts.InsertValue("series", object.GetStsSeries());
    sts.InsertValue("dateOfIssued", object.GetStsIssuedDate().FormatLocalTime("%d.%m.%Y"));
    return json;
}

namespace NDrive::NRenins {
    TUserMainData::TUserMainData(const TString& name, const TString& lastName, const TInstant birthDate)
        : Name(name)
        , LastName(lastName)
        , DateOfBirth(birthDate)
    {
    }

    NJson::TJsonValue TUserMainData::GetReport() const {
        NJson::TJsonValue json;
        NJson::InsertField(json, "name", GetName());
        NJson::InsertField(json, "last_name", GetLastName());
        NJson::InsertField(json, "date_of_birth", GetDateOfBirth().FormatLocalTime("%d.%m.%Y"));
        NJson::InsertNonNull(json, "middle_name", GetMiddleNameRef());
        return json;
    }

    bool TPaymentLink::Parse(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "paymentUrl", Link);
    }

    NJson::TJsonValue IInsuranceOrder::GetReport() const {
        NJson::TJsonValue model = GetModel();
        auto& deal = model.InsertValue("deals", NJson::JSON_ARRAY).AppendValue(GetDeal());
        auto& insuranceObjects = deal.InsertValue("insuranceObjects", NJson::JSON_ARRAY);
        auto& insuranceObject = insuranceObjects.AppendValue(NJson::JSON_MAP);

        auto& dealsObjects = model.InsertValue("dealsObjects", NJson::JSON_ARRAY);
        auto& car = dealsObjects.AppendValue(NJson::JSON_MAP);
        car.InsertValue("id", 1);
        car.InsertValue("type", "auto");
        NJson::InsertField(car, "auto", GetCarData());
        insuranceObject.InsertValue("auto", NJson::JSON_MAP).InsertValue("dealObjects", 1);

        ui32 ownersPos = OWNER_POS;
        for (const auto& item : GetOwnersData()) {
            auto& driver = dealsObjects.AppendValue(NJson::JSON_MAP);
            driver.InsertValue("id", ownersPos++);
            driver.InsertValue("type", "person");
            NJson::InsertField(driver, "person", item);
        }
        auto& owners = insuranceObject.InsertValue("owners", NJson::JSON_ARRAY);
        for (ui32 i = OWNER_POS; i < ownersPos; i++) {
            owners.AppendValue(NJson::TMapBuilder("dealObjects", i));
        }

        ui32 driversPos = ownersPos;
        for (const auto& item : GetDriversData()) {
            auto& driver = dealsObjects.AppendValue(NJson::JSON_MAP);
            driver.InsertValue("id", driversPos++);
            driver.InsertValue("type", "driver");
            NJson::InsertField(driver, "driver", item);
        }
        auto& drivers = deal.InsertValue("drivers", NJson::JSON_ARRAY);
        for (ui32 i = ownersPos; i < driversPos; i++) {
            drivers.AppendValue(NJson::TMapBuilder("dealObjects", i));
        }
        return model;
    }

    NJson::TJsonValue IInsuranceOrder::GetModel() const {
        return NJson::JSON_MAP;
    }

    NJson::TJsonValue IInsuranceOrder::GetDeal() const {
        return NJson::TMapBuilder("isMultidrive", Multidrive);
    }

    bool IInsuranceOrderResult::Parse(const NJson::TJsonValue& json) {
        double price = 0;
        if (!NJson::ParseField(json["model"]["price"], price, /* required = */ true)) {
            return false;
        }
        Price = (ui32)(price * 100);
        return NJson::ParseField(json["model"]["accountNumber"], AccountNumber, /* required = */ true)
            && DoParse(json);
    }

    bool TAutocodeCarData::Parse(const NJson::TJsonValue& json) {
        const auto& techData = json["data"]["techData"];
        const auto& identifiers = json["data"]["identifiers"];
        return NJson::ParseField(techData["brandCode"], BrandCode)
            && NJson::ParseField(techData["modelCode"], ModelCode)
            && NJson::ParseField(techData["year"], Year)
            && NJson::ParseField(techData["power"], EnginePowerCode)

            && NJson::ParseField(identifiers["vin"], Vin)
            && NJson::ParseField(identifiers["sts"], Sts)
            && NJson::ParseField(identifiers["regNumber"], LatinNumber, /* required = */ true)
        ;
    }

    NJson::TJsonValue TAutocodeCarData::GetReport() const {
        NJson::TJsonValue json;
        NJson::InsertNonNull(json, "auto_brand_code", BrandCode);
        NJson::InsertNonNull(json, "auto_model_code", ModelCode);
        NJson::InsertNonNull(json, "auto_year_of_issue_code", Year);
        NJson::InsertNonNull(json, "engine_power_code", EnginePowerCode);

        NJson::InsertNonNull(json, "vin", Vin);
        NJson::InsertNonNull(json, "sts_number", Sts);
        NJson::InsertNonNull(json, "registration_number", LatinNumber);
        return json;
    }

    namespace NOsago {
        NJson::TJsonValue TCalculateOrder::GetDeal() const {
            NJson::TJsonValue deal = IInsuranceOrder::GetDeal();
            deal.InsertValue("dealType", "eOSAGO");
            deal.InsertValue("insurancePeriodTypeCode", PeriodType);
            deal.InsertValue("dateStart", DateStart.FormatLocalTime("%d.%m.%Y"));
            return deal;
        }

        NJson::TJsonValue TCalculateOrder::GetCarData() const {
            return NJson::ToJson(AutoData);
        }

        TVector<NJson::TJsonValue> TCalculateOrder::GetOwnersData() const {
            TVector<NJson::TJsonValue> result;
            Transform(Owners.cbegin(), Owners.cend(), std::back_inserter(result), [](const auto& owner) {
                return NJson::ToJson(owner);
            });
            return result;
        }

        TVector<NJson::TJsonValue> TCalculateOrder::GetDriversData() const {
            TVector<NJson::TJsonValue> result;
            Transform(Drivers.cbegin(), Drivers.cend(), std::back_inserter(result), [](const auto& driver) {
                return NJson::ToJson(driver);
            });
            return result;
        }

        bool TCalculateOrderResult::DoParse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json["model"]["price"], MutablePrice(), /* required = */ true);
        }
    }

    namespace NKasko {
        NJson::TJsonValue TCalculateOrder::GetDeal() const {
            NJson::TJsonValue deal = IInsuranceOrder::GetDeal();
            deal.InsertValue("dealType", "Kasko");
            deal.InsertValue("isAccurateCalculation", false);
            deal.InsertValue("hasPriceDeltas", false);
            deal.InsertValue("insurant", NJson::TMapBuilder("dealObjects", OWNER_POS));
            if (UserScore) {
                deal.InsertValue("scoreExt", *UserScore);
            }
            if (Conditions.IsDefined()) {
                deal.InsertValue("conditions", Conditions);
            }
            return deal;
        }

        NJson::TJsonValue TCalculateOrder::GetCarData() const {
            return NJson::ToJson(AutoData);
        }

        NJson::TJsonValue TCarOrderStartData::GetReport() const {
            NJson::TJsonValue json;
            json.InsertValue("registration_number", GetNumber());
            json.InsertValue("vin", GetVin());
            auto& sts = json.InsertValue("ts_identity_card", NJson::JSON_MAP);
            sts.InsertValue("ts_identity_card_type_code", "sts");
            sts.InsertValue("number", GetStsNumber());
            sts.InsertValue("series", GetStsSeries());
            sts.InsertValue("date_of_issued", GetStsIssuedDate().Seconds());
            return json;
        }

        TVector<NJson::TJsonValue> TCalculateOrder::GetOwnersData() const {
            TVector<NJson::TJsonValue> result;
            Transform(Owners.cbegin(), Owners.cend(), std::back_inserter(result), [](const auto& owner) {
                return NJson::ToJson(owner);
            });
            return result;
        }

        TVector<NJson::TJsonValue> TCalculateOrder::GetDriversData() const {
            TVector<NJson::TJsonValue> result;
            Transform(Drivers.cbegin(), Drivers.cend(), std::back_inserter(result), [](const auto& driver) {
                return NJson::ToJson(driver);
            });
            return result;
        }

        TCalculateOrderResult::TCalculateOrderResult(const TString& requiredPack)
            : IInsuranceOrderResult()
            , RequiredPack(requiredPack)
        {
        }

        bool TCalculateOrderResult::DoParse(const NJson::TJsonValue& json) {
            for (auto&& deal : json["model"]["deals"].GetArray()) {
                for (auto&& packet : deal["packets"].GetArray()) {
                    if (packet["packageName"].GetStringRobust() == RequiredPack) {
                        double price = 0;
                        if (!NJson::ParseField(packet["packagePrice"], price, /* required = */ true)) {
                            return false;
                        }
                        SetPrice(price * 100);
                        return true;
                    }
                }
            }
            return false;
        }

        NJson::TJsonValue TStartOrderData::GetModel() const {
            return NJson::ToJson(OrderData);
        }

        NJson::TJsonValue TStartOrderData::GetDeal() const {
            NJson::TJsonValue deal;
            deal.InsertValue("dealType", "Kasko");
            deal.InsertValue("conditions", NJson::JSON_MAP).InsertValue("packetName", PackName);
            deal.InsertValue("dateSign", Now().FormatLocalTime("%d.%m.%Y"));
            deal.InsertValue("dateStart", DateStart.FormatLocalTime("%d.%m.%Y"));
            if (DateEnd) {
                deal.InsertValue("dateEnd", DateEnd.FormatLocalTime("%d.%m.%Y"));
            } else {
                deal.InsertValue("dateEnd", AddMonths(DateStart, 12).FormatLocalTime("%d.%m.%Y"));
            }
            deal.InsertValue("insurant", NJson::TMapBuilder("dealObjects", OWNER_POS));
            if (Conditions.IsDefined()) {
                deal.InsertValue("conditions", Conditions);
            }
            return deal;
        }

        NJson::TJsonValue TStartOrderData::GetCarData() const {
            return NJson::ToJson(Car);
        }

        TVector<NJson::TJsonValue> TStartOrderData::GetOwnersData() const {
            TVector<NJson::TJsonValue> result;
            Transform(Owners.cbegin(), Owners.cend(), std::back_inserter(result), [](const auto& owner) {
                return NJson::ToJson(owner);
            });
            return result;
        }

        NJson::TJsonValue TDriverOrderStartData::GetReport() const {
            NJson::TJsonValue json = GetUserData().GetReport();
            NJson::InsertField(json, "driver_license", GetLicenseSeries() + GetLicenseNumber());
            NJson::InsertField(json, "driving_experience_date_start", GetExperienceDateStart().FormatLocalTime("%d.%m.%Y"));
            return json;
        }

        TVector<NJson::TJsonValue> TStartOrderData::GetDriversData() const {
            TVector<NJson::TJsonValue> result;
            Transform(Drivers.cbegin(), Drivers.cend(), std::back_inserter(result), [](const auto& driver) {
                return NJson::ToJson(driver);
            });
            return result;
        }

        bool TOrderStatus::Parse(const NJson::TJsonValue& json) {
            const auto& model = json["model"];
            if (model["deals"].GetArray().empty()) {
                return false;
            }
            for (auto&& payment : model["payments"].GetArray()) {
                if (!NJson::ParseField(payment, "status", NJson::Stringify(PaymentStatus), /* required = */ false)) {
                    return false;
                }
                if (PaymentStatus & SucceedPaymentStatuses) {
                    break;
                }
            }
            {
                double price = 0;
                if (!NJson::ParseField(model, "price", price, /* required = */ false)) {
                    return false;
                }
                Price = (ui32)price * 100;
            }
            const auto& deal = model["deals"].GetArray().front();
            if (deal.Has("price")) {
                if (auto price = deal["price"].GetDoubleRobust(); price > 0) {
                    Price = (ui32)(price * 100);
                }
            }
            if (json["onlinePaymentUrl"].IsDefined()
                && json["onlinePaymentUrl"].IsString()
                && !NJson::ParseField(json, "onlinePaymentUrl", PaymentLink, /* required = */ false))
            {
                return false;
            }
            return NJson::ParseField(model, "accountStatus", NJson::Stringify(AccountStatus), /* required = */ true)
                && NJson::ParseField(deal, "psoStatus", NJson::Stringify(PsoStatus), /* required = */ false)
                && NJson::ParseField(deal, "dealId", DealId, /* required = */ false)
                && NJson::ParseField(deal, "policyNumber", PolicyNumber, /* required = */ false);
        }

        bool TOrderStatus::IsAvailable() const {
            return AccountStatus != EAccountStatus::Unknown
                || PsoStatus != EPsoStatus::Unknown
                || PaymentStatus != EPaymentStatus::Unknown
                || Price > 0;
        }
    }
}
