#pragma once

#include <rtline/util/types/accessor.h>

#include <library/cpp/json/writer/json_value.h>


namespace NDrive::NRenins {
    struct TPersonPassport {
        TString Number;
        TString Series;
        TString IssuedBy;
        TInstant DateIssue;
    };

    struct TMainOrderData {
        TString OrderId;
        TString SucceedPaymentUrl;
        TString FailedPaymentUrl;
    };

    struct TPaymentLink {
        R_FIELD(TString, Link);
    public:
        bool Parse(const NJson::TJsonValue& json);
    };

    class TUserMainData {
        R_FIELD(TString, Name);
        R_FIELD(TString, LastName);
        R_FIELD(TInstant, DateOfBirth);
        R_OPTIONAL(TString, MiddleName);

    public:
        TUserMainData() = default;
        TUserMainData(const TString& name, const TString& lastName, const TInstant birthDate);

        NJson::TJsonValue GetReport() const;
    };

    class TAddressData {
        R_OPTIONAL(TString, Country);
        R_OPTIONAL(TString, Region);
        R_OPTIONAL(TString, City);
        R_OPTIONAL(TString, Street);
        R_OPTIONAL(TString, House);
    };

    class TMainAutoData {
        R_FIELD(TString, Number);
        R_FIELD(TString, TypeCode);
        R_FIELD(TString, BrandCode);
        R_FIELD(TString, ModelCode);
        R_FIELD(TString, YearOfIssueCode);
        R_FIELD(TString, EnginePowerCode);
        R_FIELD(TString, BankOfCredit);
    };

    class IInsuranceOrder {
        R_FIELD(bool, Multidrive, false);

    public:
        virtual ~IInsuranceOrder() = default;
        NJson::TJsonValue GetReport() const;

    protected:
        virtual NJson::TJsonValue GetModel() const;
        virtual NJson::TJsonValue GetDeal() const;
        virtual NJson::TJsonValue GetCarData() const = 0;
        virtual TVector<NJson::TJsonValue> GetOwnersData() const = 0;
        virtual TVector<NJson::TJsonValue> GetDriversData() const = 0;
    };

    class IInsuranceOrderResult {
        R_FIELD(TString, AccountNumber);
        R_FIELD(ui32, Price, 0);

    public:
        virtual ~IInsuranceOrderResult() = default;
        bool Parse(const NJson::TJsonValue& json);

    private:
        virtual bool DoParse(const NJson::TJsonValue& json) = 0;
    };

    class TAutocodeCarData {
        R_FIELD(TString, LatinNumber);
        R_FIELD(TString, Sts);
        R_FIELD(TString, Vin);
        R_FIELD(TString, BrandCode);
        R_FIELD(TString, ModelCode);
        R_FIELD(TString, Year);
        R_FIELD(TString, EnginePowerCode);

    public:
        bool Parse(const NJson::TJsonValue& json);
        NJson::TJsonValue GetReport() const;
    };

    namespace NOsago {
        class TDriverData {
            R_FIELD(TUserMainData, UserData);
            R_FIELD(TString, LicenseSeries);
            R_FIELD(TString, LicenseNumber);
            R_FIELD(TInstant, ExperienceDateStart, TInstant::Zero());
        };

        class TPersonData {
            R_FIELD(TUserMainData, UserData);
            R_OPTIONAL(TString, RegoinKladrId);
            R_FIELD(TVector<TAddressData>, Addresses);
            R_FIELD(bool, PersonalAgreement, true);
        };

        class TAutoData {
            R_FIELD(TMainAutoData, MainAutoData);
            R_FIELD(TString, Vin);
            R_FIELD(bool, hasTrailer, false);
            R_FIELD(bool, TransitAuto, false);
            R_FIELD(ui32, PassengersQuantity, 0);
            R_FIELD(TString, EnginePowerCode);
            R_FIELD(TString, UsingCode);
        };

        class TCalculateOrder : public IInsuranceOrder {
            R_FIELD(TInstant, DateStart, TInstant::Zero());
            R_FIELD(TString, PeriodType);
            R_FIELD(TVector<TPersonData>, Owners);
            R_FIELD(TVector<TDriverData>, Drivers);
            R_FIELD(TAutoData, AutoData);

        private:
            virtual NJson::TJsonValue GetDeal() const override;
            virtual NJson::TJsonValue GetCarData() const override;
            virtual TVector<NJson::TJsonValue> GetOwnersData() const override;
            virtual TVector<NJson::TJsonValue> GetDriversData() const override;
        };

        class TCalculateOrderResult : public IInsuranceOrderResult {
        public:
            bool DoParse(const NJson::TJsonValue& json) override;
        };
    }

    namespace NKasko {
        class TDriverData {
            R_FIELD(TUserMainData, UserData);
            R_FIELD(TInstant, ExperienceDateStart, TInstant::Zero());
        };

        class TPersonData {
            R_FIELD(TVector<TAddressData>, Addresses);
            R_FIELD(TInstant, DateOfBirth);
        };

        class TAutoData {
            R_FIELD(TMainAutoData, MainAutoData);
            R_FIELD(TString, BodyTypeCode);
            R_FIELD(bool, InCredit, false);
            R_FIELD(ui64, Mileage, 0);
            R_FIELD(ui64, Cost, 0);
            R_FIELD(TString, Vin);
        };

        class TCalculateOrder : public IInsuranceOrder {
            R_FIELD(TVector<TPersonData>, Owners);
            R_FIELD(TVector<TDriverData>, Drivers);
            R_FIELD(TAutoData, AutoData);
            R_OPTIONAL(double, UserScore);
            R_FIELD(NJson::TJsonValue, Conditions);

        private:
            virtual NJson::TJsonValue GetDeal() const override;
            virtual NJson::TJsonValue GetCarData() const override;
            virtual TVector<NJson::TJsonValue> GetOwnersData() const override;
            virtual TVector<NJson::TJsonValue> GetDriversData() const override;
        };

        class TCalculateOrderResult : public IInsuranceOrderResult {
            const TString RequiredPack;

        public:
            TCalculateOrderResult(const TString& requiredPack);
            bool DoParse(const NJson::TJsonValue& json) override;
        };

        class TDriverOrderStartData {
            R_FIELD(TUserMainData, UserData);
            R_FIELD(TString, LicenseSeries);
            R_FIELD(TString, LicenseNumber);
            R_FIELD(TInstant, ExperienceDateStart, TInstant::Zero());
            R_FIELD(TString, SexCode);
            R_FIELD(bool, Maried, false);

        public:
            NJson::TJsonValue GetReport() const;
        };

        class TPersonOrderStartData {
            R_FIELD(TString, Phone);
            R_FIELD(TString, Email);
            R_FIELD(TUserMainData, UserData);
            R_FIELD(TPersonPassport, Passport);
            R_FIELD(TVector<TAddressData>, Addresses);
        };

        class TCarOrderStartData {
            R_FIELD(TString, Number);
            R_FIELD(TString, StsNumber);
            R_FIELD(TString, StsSeries);
            R_FIELD(TInstant, StsIssuedDate, TInstant::Zero());
            R_FIELD(TString, Vin);

        public:
            NJson::TJsonValue GetReport() const;
        };

        class TStartOrderData : public IInsuranceOrder {
            R_FIELD(TString, PackName);
            R_FIELD(TInstant, DateStart, TInstant::Zero());
            R_FIELD(TInstant, DateEnd, TInstant::Zero());
            R_FIELD(TMainOrderData, OrderData);
            R_FIELD(TCarOrderStartData, Car);
            R_FIELD(TVector<TDriverOrderStartData>, Drivers);
            R_FIELD(TVector<TPersonOrderStartData>, Owners);
            R_FIELD(NJson::TJsonValue, Conditions);

        private:
            virtual NJson::TJsonValue GetModel() const override;
            virtual NJson::TJsonValue GetDeal() const override;
            virtual NJson::TJsonValue GetCarData() const override;
            virtual TVector<NJson::TJsonValue> GetOwnersData() const override;
            virtual TVector<NJson::TJsonValue> GetDriversData() const override;
        };

        enum class EAccountStatus {
            Unknown,
            InWork,
            ReadyForSign,
            Registration,
        };

        enum class EPsoStatus {
            Unknown             /* "unknown" */,
            Draft               /* "draft" */,
            Sending             /* "sending" */,
            Preexpertise        /* "preexpertise" */,
            Expertise           /* "expertise" */,
            Repeat              /* "repeat" */,
            Clientagreement     /* "clientagreement" */,
            Contract            /* "contract" */,
            DenyExpertise       /* "deny_expertise" */,
        };

        enum EPaymentStatus: ui32 {
            Unknown = 0,
            ManualApproveWaiting = 1 << 0,
            ManuallyApproved = 1 << 1,
            Paid = 1 << 2,
            PaidIncorrect = 1 << 3,
            Special = 1 << 4,
        };

        static constexpr ui32 SucceedPaymentStatuses = EPaymentStatus::ManuallyApproved | EPaymentStatus::Paid | EPaymentStatus::Special;

        class TOrderStatus {
            R_FIELD(EAccountStatus, AccountStatus);
            R_FIELD(EPsoStatus, PsoStatus);
            R_FIELD(EPaymentStatus, PaymentStatus);
            R_FIELD(TString, PaymentLink);
            R_FIELD(TString, DealId);
            R_FIELD(TString, PolicyNumber);
            R_OPTIONAL(ui32, Price, 0);

        public:
            bool Parse(const NJson::TJsonValue& json);
            bool IsAvailable() const;
        };
    }
}
