#include <drive/library/cpp/renins/client.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/vector.h>
#include <util/string/builder.h>
#include <util/system/env.h>


namespace {
    const TString TestAddres = "Москва Академика Королева 3";
}

Y_UNIT_TEST_SUITE(ReninsSuite) {
    NDrive::NRenins::TReninsConfig GetInsuranceConfig(const TString& tokenKey) {
        const TString token = GetEnv(tokenKey);
        UNIT_ASSERT(token);
        auto config = TStringBuilder()
            << "<RequestConfig>" << Endl
            << "    MaxAttempts: 1" << Endl
            << "</RequestConfig>" << Endl
            << "RequestTimeout: 60s" << Endl
            << "Token: " << token << Endl;
        if (tokenKey == "KaskoApiToken") {
            config << "UriPath: link2api/DiKaskoInsurance/1.0" << Endl;
        }
        DEBUG_LOG << config << Endl;
        auto result = NDrive::NRenins::TReninsConfig::ParseFromString(config);
        UNIT_ASSERT(!!result);
        return *result;
    }

    NDrive::NRenins::TAddressData BuildAddressData() {
        NDrive::NRenins::TAddressData address;
        address.SetCountry("Россия");
        address.SetRegion("г Москва");
        address.SetCity("Москва");
        address.SetStreet("Академика Королева");
        address.SetHouse("3");
        return address;
    }

    void BuildOrder(NDrive::NRenins::NOsago::TCalculateOrder& order) {
        {
            auto& carData = order.MutableAutoData();
            carData.SetVin("21316546546565654");
            carData.SetPassengersQuantity(5);
            carData.SetEnginePowerCode("61");
            carData.SetUsingCode("PRIVATE");

            auto& simpleCarData = carData.MutableMainAutoData();
            simpleCarData.SetNumber("O136MO40");
            simpleCarData.SetTypeCode("lts");
            simpleCarData.SetBrandCode("honda");
            simpleCarData.SetModelCode("accord");
            simpleCarData.SetYearOfIssueCode("2014");
            simpleCarData.SetEnginePowerCode("180");
        }
        {
            NDrive::NRenins::NOsago::TPersonData owner;
            owner.SetUserData({"Максим", "Якимчук", TInstant::Seconds(0)});
            owner.MutableAddresses().emplace_back(BuildAddressData());
            owner.SetRegoinKladrId("7700000000000");
            order.MutableOwners().emplace_back(owner);
        }
        {
            NDrive::NRenins::NOsago::TDriverData driver;
            driver.SetUserData({"Максим", "Якимчук", TInstant::Zero()});
            driver.SetLicenseSeries("1231");
            driver.SetLicenseNumber("123123");
            driver.SetExperienceDateStart(TInstant::Seconds(1287552813));
            order.MutableDrivers().emplace_back(driver);
        }
    }

    void BuildOrder(NDrive::NRenins::NKasko::TCalculateOrder& order) {
        {
            auto& carData = order.MutableAutoData();
            carData.SetCost(997000);
            carData.SetMileage(1500);
            carData.SetInCredit(false);
            carData.SetBodyTypeCode("s");
            carData.SetVin("21316546546565654");

            auto& simpleCarData = carData.MutableMainAutoData();
            simpleCarData.SetNumber("O136MO40");
            simpleCarData.SetTypeCode("lts");
            simpleCarData.SetBrandCode("honda");
            simpleCarData.SetModelCode("accord");
            simpleCarData.SetYearOfIssueCode("2014");
            simpleCarData.SetEnginePowerCode("180");
        }
        order.MutableOwners().emplace_back(NDrive::NRenins::NKasko::TPersonData()
            .SetAddresses(TVector{ BuildAddressData() })
            .SetDateOfBirth(Now() - TDuration::Days(365) * 20)
        );
        {
            NDrive::NRenins::NKasko::TDriverData driver;
            driver.SetUserData({"Максим", "Якимчук", TInstant::Zero()});
            driver.SetExperienceDateStart(TInstant::Seconds(1287552813));
            order.MutableDrivers().emplace_back(driver);
        }
    }

    Y_UNIT_TEST(CalculateOsago) {
        if (!GetEnv("OsagoApiToken")) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        NDrive::NRenins::NOsago::TCalculateOrder order;
        order.SetPeriodType("oneYear");
        order.SetDateStart(Now() + TDuration::Days(31));
        BuildOrder(order);
        auto config = GetInsuranceConfig("OsagoApiToken");
        const NDrive::NRenins::NOsago::TClient client(config);
        auto future = client.Calculate(order);
        UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
        UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        if (auto sum = GetEnv("TestSum", "")) {
            UNIT_ASSERT_EQUAL(ToString(future.GetValue().GetPrice()), sum);
        }
    }

    Y_UNIT_TEST(CalculateKasko) {
        if (!GetEnv("KaskoApiToken")) {
            return;
        }
        UNIT_ASSERT(GetEnv("KaskoRequiredPack"));
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        NDrive::NRenins::NKasko::TCalculateOrder order;
        BuildOrder(order);
        auto config = GetInsuranceConfig("KaskoApiToken");
        const NDrive::NRenins::NKasko::TClient client(config);
        auto future = client.Calculate(order, GetEnv("KaskoRequiredPack"));
        UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
        UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        if (auto sum = GetEnv("TestSum", "")) {
            UNIT_ASSERT_EQUAL(ToString(future.GetValue().GetPrice()), sum);
        }
    }

    void BuildStartOrder(NDrive::NRenins::NKasko::TStartOrderData& order) {
        const TString packName = GetEnv("KaskoRequiredPack");
        UNIT_ASSERT(packName);
        order.SetPackName(packName);
        order.SetDateStart(Now() + TDuration::Days(5));
        {
            auto& orderData = order.MutableOrderData();
            orderData.OrderId = GetEnv("KaskoOrderId");
            UNIT_ASSERT(orderData.OrderId);
            orderData.SucceedPaymentUrl = "https://ya.ru";
            orderData.FailedPaymentUrl = "https://ya.ru";
        }
        {
            auto& carData = order.MutableCar();
            carData.SetNumber("O136MO40");
            carData.SetStsIssuedDate(TInstant::Seconds(1287552813));
            carData.SetStsNumber("123456");
            carData.SetStsSeries("1234");
            carData.SetVin("12345678901234567");
        }
        {
            NDrive::NRenins::NKasko::TDriverOrderStartData driver;
            driver.SetUserData({"Максим", "Якимчук", TInstant::Zero()});
            driver.SetLicenseSeries("1231");
            driver.SetLicenseNumber("123123");
            driver.SetExperienceDateStart(TInstant::Seconds(1287552813));
            driver.SetSexCode("M");
            order.MutableDrivers().emplace_back(std::move(driver));
        }
        {
            NDrive::NRenins::NKasko::TPersonOrderStartData owner;
            owner.SetUserData({"Максим", "Якимчук", TInstant::Zero()});
            owner.MutableUserData().SetMiddleName("Викторович");
            owner.SetPhone("78005553535");
            owner.SetEmail("testPartner@renins.com");
            owner.MutableAddresses().emplace_back(BuildAddressData());
            owner.SetPassport({"123456", "1234", "", TInstant::Zero()});
            order.MutableOwners().emplace_back(std::move(owner));
        }
    }

    Y_UNIT_TEST(StartKaskoOrder) {
        if (!GetEnv("KaskoApiToken")) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        NDrive::NRenins::NKasko::TStartOrderData order;
        BuildStartOrder(order);
        auto config = GetInsuranceConfig("KaskoApiToken");
        const NDrive::NRenins::NKasko::TClient client(config);
        auto future = client.Order(order);
        UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
        UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
    }

    Y_UNIT_TEST(GetKaskoOrderStatus) {
        if (!GetEnv("KaskoApiToken")) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        NDrive::NRenins::NKasko::TStartOrderData order;
        TString orderId = GetEnv("KaskoOrderId");
        UNIT_ASSERT(orderId);
        auto config = GetInsuranceConfig("KaskoApiToken");
        const NDrive::NRenins::NKasko::TClient client(config);
        auto future = client.GetStatus(orderId);
        UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
        UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        if (auto status = GetEnv("TestPsoStatus", "")) {
            UNIT_ASSERT_EQUAL(ToString(future.GetValue().GetPsoStatus()), status);
        } else if (auto status = GetEnv("TestAccountStatus", "")) {
            UNIT_ASSERT_EQUAL(ToString(future.GetValue().GetAccountStatus()), status);
        }
    }

    Y_UNIT_TEST(GetKaskoAutocode) {
        if (!GetEnv("KaskoApiToken")) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        NDrive::NRenins::NKasko::TStartOrderData order;
        const TString carNumber = GetEnv("CarNumber", "О462НУ40");
        auto config = GetInsuranceConfig("KaskoApiToken");
        const NDrive::NRenins::NKasko::TClient client(config);
        auto future = client.GetAutocodeData(carNumber);
        UNIT_ASSERT(future.Wait(config.GetRequestTimeout()));
        UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
        auto data = future.GetValue();
        UNIT_ASSERT_EQUAL_C(data.GetLatinNumber(), carNumber, data.GetReport());
    }

    NJson::TJsonValue GetEnvJson(const TString& key) {
        TString env = GetEnv(key);
        UNIT_ASSERT(env);
        NJson::TJsonValue json;
        UNIT_ASSERT(NJson::ReadJsonFastTree(env, &json));
        return json;
    }

    class TTestKaskoClient : public NDrive::NRenins::TInsuranceSyncClient {
    private:
        using TBase = NDrive::NRenins::TInsuranceSyncClient;

    public:
        using TBase::TBase;

        TString CalculateOrderId() const {
            auto future = Calculate(GetEnvJson("CalculateData"));
            UNIT_ASSERT(future.Wait(GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto reply = future.GetValue();
            INFO_LOG << reply.GetDebugReply() << Endl;
            UNIT_ASSERT(reply.IsSuccessReply());
            NJson::TJsonValue json;
            UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &json));
            TString orderId = json["model"]["accountNumber"].GetString();
            UNIT_ASSERT(orderId);
            return orderId;
        }

        void OrderInsurance(const TString& orderId) const {
            auto future = Order(orderId, GetEnvJson("OrderData"));
            UNIT_ASSERT(future.Wait(GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto reply = future.GetValue();
            INFO_LOG << reply.GetDebugReply() << Endl;
            UNIT_ASSERT(reply.IsSuccessReply());
        }

        void CheckStatus(const TString& orderId) const {
            auto future = GetStatus(orderId);
            UNIT_ASSERT(future.Wait(GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto reply = future.GetValue();
            INFO_LOG << reply.GetDebugReply() << Endl;
            UNIT_ASSERT(reply.IsSuccessReply());
        }

        void CheckPaymentLink(const TString& orderId) const {
            NDrive::NRenins::TMainOrderData order;
            order.OrderId = orderId;
            order.SucceedPaymentUrl = "https://yandex.ru";
            order.FailedPaymentUrl = "https://ya.ru";
            auto future = GetPaymentLink(order);
            UNIT_ASSERT(future.Wait(GetConfig().GetRequestTimeout()));
            UNIT_ASSERT_C(future.HasValue(), NThreading::GetExceptionMessage(future));
            auto reply = future.GetValue();
            INFO_LOG << reply.GetDebugReply() << Endl;
            UNIT_ASSERT(reply.IsSuccessReply());
        }
    };

    Y_UNIT_TEST(KaskoSimpleRequest) {
        if (!GetEnv("KaskoApiToken")) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        auto config = GetInsuranceConfig("KaskoApiToken");
        const TTestKaskoClient client(config);
        auto orderId = client.CalculateOrderId();
        client.OrderInsurance(orderId);
        client.CheckStatus(orderId);
        UNIT_ASSERT(false); // for logs
    }

    Y_UNIT_TEST(KaskoSimplePaymentLink) {
        if (!GetEnv("KaskoApiToken")) {
            return;
        }
        DoInitGlobalLog("console", FromString<ui32>(GetEnv("LogLevel", "6")), false, false);
        auto config = GetInsuranceConfig("KaskoApiToken");
        const TTestKaskoClient client(config);
        TString orderId = GetEnv("KaskoOrderId");
        client.CheckPaymentLink(orderId);
        UNIT_ASSERT(false); // for logs
    }
}
