#include "client.h"

#include <library/cpp/json/json_reader.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <util/string/builder.h>

namespace NDrive {

    TRtmrSurgeClient::TRtmrSurgeClient(TOptions options/* = Default<TOptions>()*/)
        : Client(new NNeh::THttpClient(options.Endpoint))
        , Options(options)
    {
    }

    NThreading::TFuture<TMaybe<TRtmrSurge>> TRtmrSurgeClient::GetSurge(const TString& key) const {
        auto applier = [](const NThreading::TFuture<NJson::TJsonValue>& r) -> TMaybe<TRtmrSurge> {
            const auto& parsed = r.GetValue();
            const auto& chunks = parsed["Chunks"];
            Y_ENSURE(!chunks.IsDefined() || chunks.IsArray(), "Chunks must be an array: " << chunks.GetStringRobust());
            TVector<std::pair<TString, TString>> values;
            for (auto&& chunk : chunks.GetArray()) {
                const auto& entries = chunk["Entries"];
                Y_ENSURE(!entries.IsDefined() || entries.IsArray(), "Entries must be an array: " << entries.GetStringRobust());
                for (auto&& entry : entries.GetArray()) {
                    const auto& subkey = entry["SubKey"].GetString();
                    const auto& value = entry["Value"].GetString();
                    if (value) {
                        values.emplace_back(subkey, value);
                    }
                }
            }
            if (values.empty()) {
                return {};
            }
            std::sort(values.begin(), values.end());
            auto value = NJson::ReadJsonFastTree(values.back().second);
            TRtmrSurge surge;
            surge.Prediction = NJson::FromJson<double>(value["prediction"]);
            surge.FloatFeatures = NJson::FromJson<TVector<float>>(value["float_features"]);
            TUnistatSignalsCache::SignalAdd("rtmr_client-surge", "response-success", 1);
            return surge;
        };
        return GetTableValue(Options.Path, key, Options.MaxSubkeys).Apply(applier);
    }

    NThreading::TFuture<TMaybe<TRtmrExtendedSurge>> TRtmrSurgeClient::GetExtendedSurge(const TString& key) const {
        auto applier = [](const NThreading::TFuture<NJson::TJsonValue>& r) -> TMaybe<TRtmrExtendedSurge> {
            const auto& parsed = r.GetValue();
            const auto& chunks = parsed["Chunks"];
            Y_ENSURE(!chunks.IsDefined() || chunks.IsArray(), "Chunks must be an array: " << chunks.GetStringRobust());
            TVector<std::pair<TString, TString>> values;
            for (auto&& chunk : chunks.GetArray()) {
                const auto& entries = chunk["Entries"];
                Y_ENSURE(!entries.IsDefined() || entries.IsArray(), "Entries must be an array: " << entries.GetStringRobust());
                for (auto&& entry : entries.GetArray()) {
                    const auto& subkey = entry["SubKey"].GetString();
                    const auto& value = entry["Value"].GetString();
                    if (value) {
                        values.emplace_back(subkey, value);
                    }
                }
            }
            if (values.empty()) {
                return {};
            }
            std::sort(values.begin(), values.end());
            auto value = NJson::ReadJsonFastTree(values.back().second);
            TRtmrExtendedSurge surge;
            surge.FloatFeatures = NJson::FromJson<TVector<float>>(value["float_features"]);
            NJson::ReadField(value["predictions"]["sessions_60_min"], NJson::Dictionary(surge.Predictions60min));
            NJson::ReadField(value["predictions"]["sessions_120_min"], NJson::Dictionary(surge.Predictions120min));
            NJson::ReadField(value["predictions"]["sessions_180_min"], NJson::Dictionary(surge.Predictions180min));
            TUnistatSignalsCache::SignalAdd("rtmr_client-extended_surge", "response-success", 1);
            return surge;
        };
        return GetTableValue(Options.ExtendedPath, key, Options.MaxSubkeys).Apply(applier);
    }

    NThreading::TFuture<NJson::TJsonValue> TRtmrSurgeClient::GetTableValue(const TString& path, const TString& key, int maxRecords/* = 1000*/) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/v2/data.json");
        request.SetCgiData(TStringBuilder()
            << "client" << '=' << Options.Client << '&'
            << "id" << '=' << key << '&'
            << "maxrecords" << '=' << maxRecords << '&'
            << "path" << '=' << path
        );
        TInstant start = Now();
        auto reply = Yensured(Client)->SendAsync(request, start + Options.Timeout);
        return reply.Apply([start](const NThreading::TFuture<NUtil::THttpReply>& r) {
            const TDuration duration = Now() - start;
            const NUtil::THttpReply& reply = r.GetValue();
            if (reply.IsUserError()) {
                TUnistatSignalsCache::SignalAdd("rtmr_client", "response-user_error", 1);
            }
            if (reply.IsServerError()) {
                TUnistatSignalsCache::SignalAdd("rtmr_client", "response-server_error", 1);
            }
            reply.EnsureSuccessfulReply();
            TUnistatSignalsCache::SignalAdd("rtmr_client", "response-success", 1);
            TUnistatSignalsCache::SignalHistogram(
                "rtmr_client", "response-times", duration.MilliSeconds(),
                NRTLineHistogramSignals::IntervalsRTLineReply
            );
            return NJson::ReadJsonFastTree(reply.Content());
        });
    }

    TString GetRtmrAreaKey(const TString& name) {
        return "area:" + name;
    }

}
