#include <drive/library/cpp/samsara/client.h>

#include <util/generic/queue.h>

bool TSamsaraClient::GetQueues(TVector<TSamsaraQueue>& results, TMessagesCollector& errors, const TSamsaraQueue::TId& queueId) const {
    NJson::TJsonValue reply;
    auto request = CreateGetQueuesRequest(queueId);
    if (!SendRequest(ESamsaraOperationType::GetQueues, request, reply, errors) || !reply.IsArray()) {
        return false;
    }
    for (const auto& item : reply.GetArray()) {
        TSamsaraQueue info;
        if (!info.DeserializeFromJson(item)) {
            return false;
        }
        results.push_back(std::move(info));
    }
    return true;
}

bool TSamsaraClient::GetNestedQueueIds(const TSamsaraQueue::TId& rootQueueId, TVector<TSamsaraQueue::TId>& results, TMessagesCollector& errors) const {
    TVector<TSamsaraQueue> queues;
    if (!GetQueues(queues, errors)) {
        return false;
    }

    TMap<TSamsaraQueue::TId, TVector<TSamsaraQueue::TId>> dependencies;
    for (const auto& queue : queues) {
        const auto parentId = queue.GetParentQueueId();
        if (parentId != TSamsaraQueue::UndefinedQueueId) {
            dependencies[parentId].push_back(queue.GetId());
        }
    }

    TQueue<TSamsaraQueue::TId> queueIdsToProcess;
    queueIdsToProcess.push(rootQueueId);

    while (!!queueIdsToProcess) {
        const TSamsaraQueue::TId queueId = queueIdsToProcess.front();

        results.push_back(queueId);

        auto nestedQueueIds = dependencies.FindPtr(queueId);
        if (nestedQueueIds != nullptr) {
            for (const auto& queueId : *nestedQueueIds) {
                queueIdsToProcess.push(queueId);
            }
        }

        queueIdsToProcess.pop();
    }

    return true;
}

bool TSamsaraClient::QueryUpdatedTickets(const TSamsaraQueue::TId& queueId, const TInstant& minTs, const TInstant& maxTs, TVector<TTicketUpdateInfo>& results, TMessagesCollector& errors) const {
    NJson::TJsonValue reply;
    auto request = CreateQueryUpdatedTicketsRequest(queueId, minTs, maxTs);
    if (!SendRequest(ESamsaraOperationType::QueryUpdatedTickets, request, reply, errors) || !reply.IsArray()) {
        return false;
    }
    for (const auto& item : reply.GetArray()) {
        TTicketUpdateInfo info;
        if (!info.DeserializeFromJson(item)) {
            return false;
        }
        results.push_back(std::move(info));
    }
    return true;
}

bool TSamsaraClient::GetTicket(const TTicket::TId& ticketId, TSamsaraTicket& result, TMessagesCollector& errors) const {
    NJson::TJsonValue reply;
    if (!SendRequest(ESamsaraOperationType::GetTicket, CreateGetTicketRequest(ticketId), reply, errors)) {
        return false;
    }
    return result.DeserializeFromJson(reply);
}

bool TSamsaraClient::GetMultipleTickets(const TVector<TTicket::TId>& ticketIds, TVector<TSamsaraTicket>& results, TMessagesCollector& errors) const {
    NJson::TJsonValue reply;
    auto request = CreateGetMultipleTicketsRequest(ticketIds);
    if (!SendRequest(ESamsaraOperationType::GetMultipleTickets, request, reply, errors) || !reply.IsArray()) {
        return false;
    }
    for (const auto& item : reply.GetArray()) {
        TSamsaraTicket ticket;
        if (!ticket.DeserializeFromJson(item)) {
            return false;
        }
        results.push_back(std::move(ticket));
    }
    return true;
}

NNeh::THttpRequest TSamsaraClient::CreateGetQueuesRequest(const TSamsaraQueue::TId& queueId) const {
    auto request = CreateCommonRequest("api/v2/queues/dashboardQueue");
    if (queueId != TSamsaraQueue::UndefinedQueueId) {
        request.SetCgiData("queueIds=" + ToString(queueId));
    }
    return request;
}

NNeh::THttpRequest TSamsaraClient::CreateGetTicketRequest(const TTicket::TId& ticketId) const {
    return CreateCommonRequest("api/v2/tickets/" + ticketId);
}

NNeh::THttpRequest TSamsaraClient::CreateGetMultipleTicketsRequest(const TVector<TTicket::TId>& ticketIds) const {
    NJson::TJsonValue data(NJson::JSON_ARRAY);
    for (const auto& id : ticketIds) {
        data.AppendValue(id);
    }
    return CreateCommonRequest("api/v2/tickets/multi", ERequestMethod::POST, data, ERequestContentType::Json);
}

NNeh::THttpRequest TSamsaraClient::CreateQueryUpdatedTicketsRequest(const TSamsaraQueue::TId& queueId, const TInstant& minTs, const TInstant& maxTs) const {
    TString cgiData = TStringBuilder() << "queueId=" << queueId
                                       << "&minTs=" << minTs.MilliSeconds()
                                       << "&maxTs=" << maxTs.MilliSeconds();
    return CreateCommonRequest("api/v2/tickets/queryUpdated").SetCgiData(cgiData);
}
