#include <drive/library/cpp/samsara/entity.h>

const TSamsaraQueue::TId TSamsaraQueue::UndefinedQueueId = 0;
const TSamsaraQueue::TId TSamsaraTag::UndefinedTagId = 0;
const TSamsaraQueue::TId TSamsaraArticle::UndefinedArticleId = 0;

template <typename T>
static inline bool DeserializeContainer(const NJson::TJsonValue& data, const TString& field, TVector<T>& results, const bool mustBe = false) {
    if (!data.Has(field) || !data[field].IsArray()) {
        return !mustBe;
    }
    for (const auto& item : data[field].GetArray()) {
        T instance;
        if (!instance.DeserializeFromJson(item)) {
            return false;
        }
        results.push_back(std::move(instance));
    }
    return true;
}

static inline bool ReadInstantMilliseconds(const NJson::TJsonValue& data, const TString& field, TInstant& result, const bool mustBe = false) {
    if (!data.Has(field) || !data[field].IsUInteger()) {
        return !mustBe;
    }
    result = TInstant::MilliSeconds(data[field].GetUInteger());
    return true;
}

bool TSamsaraQueue::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_UINT(data, "id", Id);
    if (!data.Has("parentQueueId")) {
        ParentQueueId = UndefinedQueueId;
    } else {
        JREAD_UINT(data, "parentQueueId", ParentQueueId);
    }
    JREAD_STRING(data, "name", Name);
    JREAD_STRING(data, "state", State);
    JREAD_UINT_NULLABLE_OPT(data, "openTicketsCount", OpenTicketsCount);
    return true;
}

bool TSamsaraTag::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_UINT(data, "id", Id);
    if (!data.Has("parentTagId")) {
        ParentTagId = UndefinedTagId;
    } else {
        JREAD_UINT(data, "parentTagId", ParentTagId);
    }
    JREAD_STRING(data, "name", Name);
    JREAD_STRING(data, "state", State);
    return true;
}

bool TSamsaraArticleRecepient::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_STRING_OPT(data, "login", Login);
    JREAD_STRING_OPT(data, "name", Name);
    JREAD_STRING_OPT(data, "email", Email);
    return true;
}

bool TSamsaraArticleAttachment::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_STRING_OPT(data, "name", Name);
    JREAD_STRING_OPT(data, "contentType", ContentType);
    JREAD_INT_OPT(data, "size", Size);
    JREAD_STRING_OPT(data, "uri", Uri);
    JREAD_STRING_OPT(data, "extId", ExternalId);
    JREAD_STRING_OPT(data, "type", Type);
    return true;
}

bool TSamsaraArticleBodyPart::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_STRING_OPT(data, "type", Type);
    for (const auto& [fieldName, fieldValue] : data["customData"].GetMap()) {
        CustomData.emplace(fieldName, fieldValue.GetStringRobust());
    }
    JREAD_STRING_OPT(data, "content", Content);
    JREAD_STRING_OPT(data, "contentType", ContentType);
    return true;
}

bool TSamsaraArticle::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_STRING_OPT(data, "type", Type);
    JREAD_UINT(data, "id", Id);

    if (data.Has("parentArticleId") && !data["parentArticleId"].IsNull()) {
        JREAD_UINT(data, "parentArticleId", ParentArticleId);
    } else {
        ParentArticleId = UndefinedArticleId;
    }

    if (!DeserializeContainer(data, "tags", Tags)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "linkedStTickets", LinkedStartrekTickets)) {
        return false;
    }

    if (!From.DeserializeFromJson(data["from"])) {
        return false;
    }
    if (!DeserializeContainer(data, "to", To)) {
        return false;
    }
    if (!DeserializeContainer(data, "cc", Cc)) {
        return false;
    }
    if (!DeserializeContainer(data, "bcc", Bcc)) {
        return false;
    }
    if (!DeserializeContainer(data, "replyTo", ReplyTo)) {
        return false;
    }

    if (!ReadInstantMilliseconds(data, "createdTs", CreatedTs)) {
        return false;
    }

    JREAD_STRING_OPT(data, "subject", Subject);
    JREAD_STRING_OPT(data, "body", Body);

    if (!DeserializeContainer(data, "attachments", Attachments)) {
        return false;
    }

    for (const auto& [fieldName, fieldValue] : data["customData"].GetMap()) {
        CustomData.emplace(fieldName, fieldValue.GetStringRobust());
    }

    JREAD_BOOL_OPT(data, "messageBodyTrimmed", MessageBodyTrimmed);

    if (!DeserializeContainer(data, "bodyParts", BodyParts)) {
        return false;
    }

    return true;
}

bool TSamsaraTicket::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_STRING(data, "ticketNumber", TicketNumber);
    JREAD_STRING(data, "subject", Subject);
    JREAD_STRING(data, "status", Status);
    JREAD_STRING(data, "resolution", Resolution);
    JREAD_STRING(data, "priority", Priority);

    if (!DeserializeContainer(data, "articles", Articles)) {
        return false;
    }

    if (!ReadInstantMilliseconds(data, "createdTs", CreatedTs, /* mustBe = */ true)) {
        return false;
    }
    if (!ReadInstantMilliseconds(data, "updatedTs", UpdatedTs)) {
        return false;
    }
    if (!ReadInstantMilliseconds(data, "openSinceTs", OpenSinceTs)) {
        return false;
    }
    if (!ReadInstantMilliseconds(data, "reopenTs", ReopenTs)) {
        return false;
    }

    if (!DeserializeContainer(data, "tags", Tags)) {
        return false;
    }

    JREAD_UINT(data, "queueId", QueueId);

    JREAD_STRING_OPT(data, "createdBy", CreatedBy);
    JREAD_STRING_OPT(data, "createdFor", CreatedFor);
    JREAD_STRING_OPT(data, "updatedBy", UpdatedBy);

    if (data.Has("lastArticleId") && !data["lastArticleId"].IsNull()) {
        JREAD_UINT(data, "lastArticleId", LastArticleId);
    } else {
        LastArticleId = TArticle::UndefinedArticleId;
    }
    if (data.Has("rootArticleId") && !data["rootArticleId"].IsNull()) {
        JREAD_UINT(data, "rootArticleId", RootArticleId);
    } else {
        RootArticleId = TArticle::UndefinedArticleId;
    }

    JREAD_STRING_NULLABLE_OPT(data, "channel", Channel);

    if (!TJsonProcessor::ReadContainer(data, "linkedTickets", LinkedTickets)) {
        return false;
    }

    JREAD_STRING(data, "id", Id);
    return true;
}
