#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/string/builder.h>
#include <util/system/env.h>
#include <util/generic/vector.h>
#include <util/string/join.h>

#include <drive/library/cpp/samsara/client.h>
#include <drive/library/cpp/samsara/config.h>

namespace {
    TSamsaraClientConfig GetSamsaraConfig() {
        auto config = TSamsaraClientConfig::ParseFromString(
            TStringBuilder()
            << "Host: samsara.yandex-team.ru" << Endl
            << "Port: 443" << Endl
            << "IsHttps: true" << Endl
            << "Token: " << GetEnv("TESTING_SAMSARA_TOKEN") << Endl
            << "TokenPath: " << GetEnv("TESTING_SAMSARA_TOKEN_PATH") << Endl
            << "RequestTimeout: 5s" << Endl
            << "<RequestConfig>" << Endl
            << "    MaxAttempts: 1" << Endl
            << "</RequestConfig>" << Endl
        );
        return config;
    }

    TSamsaraClient GetSamsaraClient() {
        return TSamsaraClient(GetSamsaraConfig());
    }
}

Y_UNIT_TEST_SUITE(SamsaraClient) {
    Y_UNIT_TEST(GetQueues) {
        const TSamsaraQueue::TId driveRootQueueId = 4212;

        auto client = GetSamsaraClient();

        TVector<TSamsaraQueue::TId> results;
        TMessagesCollector errors;

        UNIT_ASSERT_C(client.GetNestedQueueIds(driveRootQueueId, results, errors), errors.GetStringReport());
        UNIT_ASSERT_GE(results.size(), 1);
    }

    Y_UNIT_TEST(QueryUpdatedTickets) {
        const TSamsaraQueue::TId driveMainQueueId = 4005;

        const TInstant since = TInstant::ParseIso8601("2019-11-01T09:00:00+0300");  // 1572588000000
        const TInstant until = since + TDuration::Seconds(20);

        auto client = GetSamsaraClient();

        TVector<TSamsaraClient::TTicketUpdateInfo> result;
        TMessagesCollector errors;

        UNIT_ASSERT_C(client.QueryUpdatedTickets(driveMainQueueId, since, until, result, errors), errors.GetStringReport());
        UNIT_ASSERT_VALUES_EQUAL(result.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(result.front().GetTicketId(), 104657500);
        UNIT_ASSERT_VALUES_EQUAL(result.front().GetArticleId(), TSamsaraArticle::UndefinedArticleId);
    }

    Y_UNIT_TEST(GetTicket) {
        const TSamsaraTicket::TId ticketId = "101105003";

        auto client = GetSamsaraClient();

        TSamsaraTicket result;
        TMessagesCollector errors;

        UNIT_ASSERT_C(client.GetTicket(ticketId, result, errors), errors.GetStringReport());
        UNIT_ASSERT_STRINGS_EQUAL(result.GetId(), ticketId);
        UNIT_ASSERT_VALUES_EQUAL(result.GetCreatedTs(), TInstant::MilliSeconds(1549205018832));
    }

    Y_UNIT_TEST(GetMultipleTickets) {
        const TVector<TSamsaraTicket::TId> ticketIds = {"101105003"};

        auto client = GetSamsaraClient();

        TVector<TSamsaraTicket> results;
        TMessagesCollector errors;

        UNIT_ASSERT_C(client.GetMultipleTickets(ticketIds, results, errors), errors.GetStringReport());
        UNIT_ASSERT_VALUES_EQUAL(results.size(), 1);
        UNIT_ASSERT_STRINGS_EQUAL(results.front().GetId(), ticketIds.front());
    }
}
