#include "client.h"

#include <rtline/library/unistat/cache.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TSaturnClient::TDebtScoringOptions& object) {
    NJson::TJsonValue result;
    result["reqid"] = object.ReqId;
    result["puid"] = object.PassportUid;
    if (object.FormulaId) {
        result["formula_id"] = object.FormulaId;
    }
    result["service"] = object.Service;
    if (object.Amount) {
        result["amount"] = *object.Amount;
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TSaturnClient::TResult& result) {
    return NJson::ParseField(value["reqid"], result.ReqId) &&
        NJson::ParseField(value["puid"], result.PassportUid) &&
        NJson::ParseField(value["score"], result.Score) &&
        NJson::ParseField(value["score_raw"], result.ScoreRaw) &&
        NJson::ParseField(value["formula_id"], result.FormulaId) &&
        NJson::ParseField(value["formula_description"], result.FormulaDescription) &&
        NJson::ParseField(value["data_source"], result.DataSource) &&
        NJson::ParseField(value["status"], result.Status);
}

namespace NDrive {

    TSaturnClient::TSaturnClient(const TOptions& options, TMaybe<TTvmAuth> tvmAuth)
        : Client(new NNeh::THttpClient(options.Endpoint))
        , Options(options)
        , TvmAuth(tvmAuth)
    {
    }

    NThreading::TFuture<TSaturnClient::TResult> TSaturnClient::GetDebtScoring(const TDebtScoringOptions& options) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/v1/drive/search");
        if (TvmAuth) {
            TvmAuth->UpdateRequest(request);
        }
        request.SetPostData(NJson::ToJson(options));
        TUnistatSignalsCache::SignalAdd("saturn_client", "request", 1);
        TInstant start = Now();
        auto reply = Yensured(Client)->SendAsync(request, start + Options.RequestTimeout);
        return reply.Apply([start](const NThreading::TFuture<NNeh::THttpReply>& r) {
            const TDuration duration = Now() - start;
            const NUtil::THttpReply& reply = r.GetValue();
            if (reply.IsUserError()) {
                TUnistatSignalsCache::SignalAdd("saturn_client", "response-user_error", 1);
            }
            if (reply.IsServerError()) {
                TUnistatSignalsCache::SignalAdd("saturn_client", "response-server_error", 1);
            }
            reply.EnsureSuccessfulReply();
            TUnistatSignalsCache::SignalAdd("saturn_client", "response-success", 1);
            TUnistatSignalsCache::SignalHistogram(
                "saturn_client", "response-times", duration.MilliSeconds(),
                NRTLineHistogramSignals::IntervalsRTLineReply
            );
            const auto response = NJson::ToJson(NJson::JsonString(reply.Content()));
            return NJson::FromJson<TResult>(response);
        });
    }

}
