#pragma once

#include <rtline/library/json/cast.h>
#include <rtline/library/rect_hash/rect_hash.h>

#include <util/ysaveload.h>

namespace NDrive {
    template <class T>
    class TGeoSchedule {
    public:
        using TObjectPtr = TAtomicSharedPtr<T>;
        using TObjectConstPtr = TAtomicSharedPtr<const T>;

    private:
        struct THashable {
            TGeoRect Rectangle;
            TObjectPtr Object;

            const TGeoRect& GetRect() const {
                return Rectangle;
            }
            inline bool operator==(const THashable& other) const {
                return Object.Get() == other.Object.Get();
            }
            inline bool operator<(const THashable& other) const {
                return Object.Get() < other.Object.Get();
            }

            void Load(IInputStream* in) {
                ::Load(in, Rectangle.Min.X);
                ::Load(in, Rectangle.Min.Y);
                ::Load(in, Rectangle.Max.X);
                ::Load(in, Rectangle.Max.Y);
                Object = MakeAtomicShared<T>();
                ::Load(in, *Object);
            }
            void Save(IOutputStream* out) const {
                ::Save(out, Rectangle.Min.X);
                ::Save(out, Rectangle.Min.Y);
                ::Save(out, Rectangle.Max.X);
                ::Save(out, Rectangle.Max.Y);
                ::Save(out, *Object);
            }
        };

    public:
        TGeoSchedule()
            : RectHash({ -180, -90, 180, 90 })
        {
        }

        void Add(const TGeoRect& rectangle, TObjectPtr object) {
            THashable hashable;
            hashable.Rectangle = rectangle;
            hashable.Object = object;
            Y_ENSURE(RectHash.AddObject(std::move(hashable)));
        }

        TObjectConstPtr Get(const TGeoCoord& coordinate) const {
            auto p = RectHash.GetObject(coordinate);
            return p ? p->Object : nullptr;
        }

        void Load(IInputStream* in) {
            TVector<THashable> objects;
            ::Load(in, objects);
            for (auto&& i : objects) {
                Add(i.Rectangle, i.Object);
            }
        }
        void Save(IOutputStream* out) const {
            ::Save(out, RectHash.GetObjects(true));
        }

    private:
        TRectHash<TGeoCoord, THashable> RectHash;
    };
}

