#pragma once

#include <rtline/library/json/cast.h>
#include <rtline/util/types/coverage.h>

#include <util/datetime/base.h>

namespace NDrive {
    template <class T>
    class TTemporalSchedule {
    public:
        using TCoverage = NUtil::TCoverage<ui64, T>;

    public:
        TTemporalSchedule(TInterval<ui64> range)
            : Coverage(range.GetMin(), range.GetMax())
        {
        }
        TTemporalSchedule(TDuration duration = TDuration::Days(7))
            : Coverage(0, Key(duration) - 1)
        {
        }

        void Add(TInterval<ui64> range, T object) {
            Coverage.AddHandler(range, object);
        }
        void Add(TDuration offset, TDuration duration, T object) {
            TInterval<ui64> range(Key(offset), Key(offset + duration) - 1);
            Add(range, object);
        }

        const TCoverage& GetCoverage() const {
            return Coverage;
        }

        const T* Get(TDuration offset) const {
            auto p = Coverage.lower_bound(Key(offset));
            if (p == Coverage.end()) {
                return nullptr;
            }
            const auto& elements = p->second;
            Y_ASSERT(elements.size() < 2);
            if (elements.empty()) {
                return nullptr;
            }
            return &elements[0];
        }

        void Load(IInputStream* in) {
            TMap<TInterval<ui64>, T> data;
            ::Load(in, data);
            for (auto&&[interval, handler] : data) {
                Coverage.AddHandler(interval, handler);
            }
        }
        void Save(IOutputStream* out) const {
            TMap<TInterval<ui64>, T> data;
            for (auto&&[interval, handlers] : Coverage) {
                Y_ASSERT(handlers.size() < 2);
                if (handlers.empty()) {
                    continue;
                }
                data[interval] = handlers[0];
            }
            ::Save(out, data);
        }

    private:
        ui64 Key(TDuration value) const {
            return value.Seconds();
        }

    private:
        TCoverage Coverage;
    };
}

namespace NJson {
    template <class T>
    TJsonValue ToJson(const TInterval<T>& object) {
        TJsonValue result;
        result.AppendValue(ToJson(object.GetMin()));
        result.AppendValue(ToJson(object.GetMax()));
        return result;
    }

    template <class T>
    bool TryFromJson(const TJsonValue& value, TInterval<T>& result) {
        if (!value.IsArray()) {
            return false;
        }
        const auto& arr = value.GetArray();
        if (arr.size() != 2) {
            return false;
        }
        auto min = TryFromJson<T>(arr[0]);
        auto max = TryFromJson<T>(arr[1]);
        if (!min || !max) {
            return false;
        }
        result = { *min, *max };
        return true;
    }

    template <class T>
    TJsonValue ToJson(const NDrive::TTemporalSchedule<T>& object) {
        const auto& coverage = object.GetCoverage();
        NJson::TJsonValue result;
        result.InsertValue("range", ToJson(
            NUtil::TInterval<ui64>(coverage.GetMin(), coverage.GetMax())
        ));

        auto& cover = result.InsertValue("coverage", NJson::JSON_ARRAY);
        for (auto&&[interval, handlers] : coverage) {
            Y_ASSERT(handlers.size() < 2);
            if (handlers.empty()) {
                continue;
            }
            TJsonValue element;
            element["handler"] = ToJson(handlers[0]);
            element["range"] = ToJson(interval);
            cover.AppendValue(std::move(element));
        }
        return result;
    }

    template <class T>
    bool TryFromJson(const TJsonValue& value, NDrive::TTemporalSchedule<T>& result) {
        auto range = TryFromJson<NUtil::TInterval<ui64>>(value["range"]);
        if (!range) {
            return false;
        }
        NDrive::TTemporalSchedule<T> r(*range);
        const auto& elements = value["elements"];
        if (!elements.IsArray()) {
            return false;
        }
        for (auto&& element : elements.GetArray()) {
            auto interval = TryFromJson<TInterval<ui64>>(element["range"]);
            auto handler = TryFromJson<T>(element["handler"]);
            if (!interval || !handler) {
                return false;
            }
            r.Add(*interval, *handler);
        }
        result = r;
        return true;
    }
}
