#include "scheme.h"

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/parse.h>

#include <util/generic/guid.h>
#include <util/string/join.h>

namespace NJson {
    template <>
    TJsonValue ToJson(const NDrive::TFSVariants::TCompoundVariant& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const TJsonValue& data, NDrive::TFSVariants::TCompoundVariant& result) {
        return result.DeserializeFromJson(data);
    }
}

namespace NDrive {
    ISchemeElement::TFactory::TRegistrator<TFSIgnore> TFSIgnore::Registrator(EElementType::Ignore);
    ISchemeElement::TFactory::TRegistrator<TFSSeparator> TFSSeparator::Registrator(EElementType::Separator);
    ISchemeElement::TFactory::TRegistrator<TFSString> TFSString::Registrator(EElementType::String);
    ISchemeElement::TFactory::TRegistrator<TFSText> TFSText::Registrator(EElementType::Text);
    ISchemeElement::TFactory::TRegistrator<TFSJson> TFSJson::Registrator(EElementType::Json);
    ISchemeElement::TFactory::TRegistrator<TFSArray> TFSArray::Registrator(EElementType::Array);
    ISchemeElement::TFactory::TRegistrator<TFSStructure> TFSStructure::Registrator(EElementType::Structure);
    ISchemeElement::TFactory::TRegistrator<TFSVariants> TFSVariants::Registrator(EElementType::Variants);
    ISchemeElement::TFactory::TRegistrator<TFSVariable> TFSVariable::Registrator(EElementType::Variable);
    ISchemeElement::TFactory::TRegistrator<TFSBoolean> TFSBoolean::Registrator(EElementType::Boolean);
    ISchemeElement::TFactory::TRegistrator<TFSNumeric> TFSNumeric::Registrator(EElementType::Numeric);
    ISchemeElement::TFactory::TRegistrator<TFSDuration> TFSDuration::Registrator(EElementType::Duration);

    const TString ISchemeElement::DefaultTabName = "default";

    ISchemeElement::ISchemeElement(const TString& fieldName, const TString& description, const ui32 idx)
        : FieldName(fieldName)
        , Description(description)
    {
        if (idx != Max<ui32>()) {
            OrderIdx = idx;
        }
    }

    void ISchemeElement::AddHidingRule(IHidingRule::TPtr rule) {
        HidingRules.emplace_back(std::move(rule));
    }

    NJson::TJsonValue ISchemeElement::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        if (traits & EReportTraits::TypeField) {
            result.InsertValue("type", ::ToString(GetType()));
        }
        if (GetRegisterType() != GetType() && (traits & EReportTraits::InternalTypeField)) {
            result.InsertValue("internal_type", ::ToString(GetRegisterType()));
        }
        if (traits & EReportTraits::TabNameField) {
            JWRITE_DEF(result, "tab_name", GetTabName(), DefaultTabName);
        }
        if (traits & EReportTraits::ReadOnlyField) {
            result.InsertValue("read_only", ReadOnly);
        }
        if (!!Deprecated && *Deprecated) {
            result.InsertValue("deprecated", *Deprecated);
        }
        if (!!Description && (traits & EReportTraits::DescriptionField)) {
            result.InsertValue("display_name", Description);
        }
        if (!!Tooltip && (traits & EReportTraits::TooltipField)) {
            result.InsertValue("description", Tooltip);
        }
        if (Required && (traits & EReportTraits::RequiredField)) {
            result.InsertValue("required", Required);
        }
        if (!HidingRules.empty() && (traits & EReportTraits::HidingField)) {
            auto& rules = result.InsertValue("hide", NJson::JSON_ARRAY);
            for (auto&& rule : HidingRules) {
                if (rule){
                    rules.AppendValue(rule->SerializeToJson());
                }
            }
        }
        return result;
    }

    ISchemeElement::TPtr ISchemeElement::ConstructFromJson(const NJson::TJsonValue& json, const TString& fieldName) {
        ISchemeElement::TPtr element;
        EElementType type;
        bool found = false;
        if (json.Has("internal_type")) {
            found = TryFromString(json["internal_type"].GetString(), type);
        }
        if (!found && (!json.Has("type") || !TryFromString(json["type"].GetString(), type))) {
            element = MakeAtomicShared<TScheme>();
        } else {
            element = TFactory::MakeHolder(type, fieldName);
        }

        if (!element) {
            ERROR_LOG << "unknown type " << type << Endl;
            return nullptr;
        }
        if (!element->DeserializeFromJson(json)) {
            ERROR_LOG << "Cannot deserialize from " << json.GetStringRobust() << Endl;
            return nullptr;
        }
        return element;
    }

    bool ISchemeElement::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_STRING_OPT(json, "tab_name", TabName);
        JREAD_BOOL_OPT(json, "read_only", ReadOnly);
        JREAD_STRING_OPT(json, "display_name", Description);
        JREAD_STRING_OPT(json, "description", Tooltip);
        JREAD_BOOL_OPT(json, "required", Required);
        TJsonProcessor::Read(json, "deprecated", Deprecated);
        return true;
    }

    bool ISchemeElement::ValidateJson(const NJson::TJsonValue& json, const TString& path) const {
        TString error;
        if (!DoValidateJson(json, path, error)) {
            if (!!error) {
                ERROR_LOG << json.GetStringRobust() << " by path " << path << " " << error << Endl;
            }
            return false;
        }
        return true;
    }

    void ISchemeElement::MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        PrintValue(json, buf, os);
        PrintDescription(buf, os);
    }

    void ISchemeElement::PrintDescription(NJsonWriter::TBuf& buf, IOutputStream& os) const {
        buf.FlushTo(&os);
        os << " // (" << ::ToString(GetRegisterType()) << (Required ? "" : ", optional") << ")";
        if (Description && (Description != FieldName)) {
            os << " " << Description;
        }
        auto additionalDescription = AdditionalDescription();
        if (additionalDescription) {
            os << " " << additionalDescription;
        }
    }

    void ISchemeElement::PrintValue(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        if (!ValidateJson(json)) {
            ythrow yexception() << "incorrect json";
        }
        AddValueToDescription(json, buf, os);
    }

    void ISchemeElement::AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& /*os*/) const {
        buf.WriteJsonValue(&json, true);
    }

    TString ISchemeElement::AdditionalDescription() const {
        return TString();
    }

    NJson::TJsonValue IBaseDefaultSchemeElement::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);

        auto defaultValue = GetDefaultValueView();
        if (defaultValue.IsDefined()) {
            NJson::InsertField(result, "default", defaultValue);
        }

        return result;
    }

    bool IBaseDefaultSchemeElement::DeserializeFromJson(const NJson::TJsonValue& json) {
        return TBase::DeserializeFromJson(json) && SetDefaultValueView(json["default"]);
    }

    void TFSIgnore::MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        if (json.IsArray() || json.IsMap()) {
            PrintDescription(buf, os);
            PrintValue(json, buf, os);
        } else {
            PrintValue(json, buf, os);
            PrintDescription(buf, os);
        }
    }

    bool TFSIgnore::DoValidateJson(const NJson::TJsonValue& /*json*/, const TString& /*path*/, TString& /*error*/) const {
        return true;
    }

    bool TFSSeparator::DoValidateJson(const NJson::TJsonValue& /*json*/, const TString& /*path*/, TString& /*error*/) const {
        return true;
    }

    bool TFSString::TValidator::DeserializeFromJson(const NJson::TJsonValue& json) {
        TJsonProcessor::Read(json, "regex", Regex);
        TJsonProcessor::Read(json, "error_string", ErrorString);
        return HasRegex() || HasErrorString();
    }

    NJson::TJsonValue TFSString::TValidator::SerializeToJson() const {
        NJson::TJsonValue validatorResult;
        if (HasRegex() && GetRegexRef()) {
            validatorResult.InsertValue("regex", GetRegexRef());
        }

        if (HasErrorString() && GetErrorStringRef()) {
            validatorResult.InsertValue("error_string", GetErrorStringRef());
        }

        return validatorResult;
    }

    NJson::TJsonValue TFSString::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);
        NJson::InsertNonNull(result, "maxLength", MaxLength);
        if (Visual) {
            result.InsertValue("visual", ::ToString(*Visual));
        }
        if (Validator) {
            result.InsertValue("validator", Validator->SerializeToJson());
        }
        return result;
    }

    bool TFSString::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (json.Has("visual")) {
            EVisualType visualType;
            if (!NJson::ParseField(json["visual"], NJson::Stringify(visualType))) {
                return false;
            }
            Visual = visualType;
        }

        TValidator v;
        if (v.DeserializeFromJson(json["validator"])) {
            SetValidator(v);
        }

        return NJson::TryFromJson(json["maxLength"], MaxLength)
            && TBase::DeserializeFromJson(json);
    }

    bool TFSString::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (!json.IsString()) {
            error = "is not a STRING";
            return false;
        }
        if (Visual) {
            TGUID guid;
            bool rightType = false;
            switch (*Visual) {
            case EVisualType::GUID:
                rightType = GetGuid(json.GetString(), guid);
                break;
            case EVisualType::UUID:
                [[fallthrough]];
            case EVisualType::ObjectId:
                rightType = GetUuid(json.GetString(), guid);
                break;
            default:
                break;
            }
            if (!rightType) {
                error = "is not a "+ ::ToString(*Visual);
                return false;
            }
        }
        if (MaxLength) {
            if (json.GetString().size() > *MaxLength) {
                error = "max length exceeded: " + ::ToString(json.GetString().size()) + " > " + ::ToString(*MaxLength);
                return false;
            }
        }
        return true;
    }

    bool TFSText::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (!json.IsString()) {
            error = "is not a STRING";
        }
        return true;
    }

    bool TFSJson::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (!json.IsString()) {
            error = "is not a STRING";
            return false;
        }
        NJson::TJsonValue result;
        if (!NJson::ReadJsonTree(json.GetString(), &result)) {
            error = "is not a JSON";
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFSJson::GetDefaultValueView() const {
        if (!HasDefault()) {
            return {};
        }
        NJson::TJsonValue result;
        if (NJson::ReadJsonFastTree(GetDefaultUnsafe(), &result)) {
            return result;
        }
        return GetDefaultUnsafe();
    }

    TFSVariants::TCompoundVariant::TCompoundVariant(const TString& value)
        : TCompoundVariant(value, value)
    {
    }

    TFSVariants::TCompoundVariant::TCompoundVariant(const TString& value, const TString& text, const TString& description)
        : Value(value)
        , Text(text)
        , Description(description)
    {
    }

    bool TFSVariants::TCompoundVariant::operator < (const TCompoundVariant& other) const {
        return Value < other.Value;
    }

    NJson::TJsonValue TFSVariants::GetDefaultValueView() const {
        if (!Defaults) {
            return {};
        }
        return (MultiSelect) ? NJson::ToJson(Defaults) : NJson::ToJson(Defaults.back());
    }

    bool TFSVariants::SetDefaultValueView(const NJson::TJsonValue& view) {
        if (!NJson::ParseField(view, Defaults)) {
            TString defaultValue;
            if (!NJson::ParseField(view, defaultValue)) {
                return false;
            }
            Defaults = {defaultValue};
        }
        return true;
    }

    NJson::TJsonValue TFSVariants::TCompoundVariant::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "value", Value);
        NJson::InsertField(result, "text", Text);
        NJson::InsertNonNull(result, "description", Description);
        return result;
    }

    bool TFSVariants::TCompoundVariant::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["value"], Value, /* required = */ true) &&
               NJson::ParseField(data["text"], Text) &&
               NJson::ParseField(data["description"], Description);
    }

    TFSVariants& TFSVariants::SetDefault(const TString& value) {
        return SetDefaults(TVector<TString>{value});
    }

    NJson::TJsonValue TFSVariants::SerializeToJson(TReportTraits traits /*= ReportAll*/) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);

        // insert one of variants fields (they are maintained consistent)
        if (CompoundVariants) {
            NJson::InsertField(result, "variants", CompoundVariants);
        } else {
            NJson::InsertField(result, "variants", Variants);
        }

        NJson::InsertField(result, "editable", Editable);
        NJson::InsertField(result, "multi_select", MultiSelect);
        NJson::InsertNonNull(result, "reference", Reference);
        NJson::InsertNonNull(result, "refresh_on_update", RefreshOnUpdate);

        return result;
    }

    bool TFSVariants::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (!NJson::ParseField(json["variants"], Variants, true)) {
            if (!NJson::ParseField(json["variants"], CompoundVariants, true)) {  // NB. compound types could be parsed incorrectly!
                return false;
            }
            for (auto&& compoundVariant: CompoundVariants) {
                Variants.emplace(compoundVariant.GetValue());
            }
        }
        return NJson::ParseField(json["multi_select"], MultiSelect) &&
               NJson::ParseField(json["editable"], Editable) &&
               NJson::ParseField(json["reference"], Reference) &&
               TBase::DeserializeFromJson(json);
    }

    bool TFSVariants::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (MultiSelect) {
            if (!json.IsArray()) {
                error = "is not an ARRAY";
                return false;
            }
            const auto& jsonArray = json.GetArray();
            for (size_t i = 0; i < jsonArray.size(); ++i) {
                const auto& elem = jsonArray[i];
                if (!elem.IsString()) {
                    error = "[" + ToString(i) + "] is not a STRING";
                    return false;
                }
                if (!Editable && !Variants.contains(elem.GetString())) {
                    error = "[" + ToString(i) + "]" + elem.GetString() + " not in " + JoinSeq(", ", Variants);
                    return false;
                }
            }
        } else {
            if (!json.IsString()) {
                error = "is not a STRING";
                return false;
            }
            if (!Editable && !Variants.contains(json.GetString())) {
                error = "not in " + JoinSeq(", ", Variants);
                return false;
            }
        }
        return true;
    }

    TString TFSVariants::AdditionalDescription() const{
        return "(variants: " + JoinSeq(", ", Variants) + ")";
    }

    TFSVariable::TFSVariable(const TString& fieldName, const TString& description, const ui32 idx)
        : TBase(fieldName, description, idx)
    {
        Condition.SetDescription(description);  // NB. all properties must be applied to the nested condition instead
    }

    NJson::TJsonValue TFSVariable::SerializeToJson(TReportTraits traits) const {
        auto result = TBase::SerializeToJson(traits);  // only type and order are serialized actually

        auto& controlField = result.InsertValue("control_field", NJson::JSON_MAP);
        controlField.InsertValue(GetFieldName(), Condition.SerializeToJson(traits));

        // defaults and variants are "schemes" itself

        if (!!DefaultValue) {
            result.InsertValue("default_fields", DefaultValue->SerializeToJson(traits));
        }

        auto& variantsMapping = result.InsertValue("variants_fields", NJson::JSON_MAP);
        for (auto&& [name, item] : ValueMapping) {
            variantsMapping.InsertValue(name, item->SerializeToJson());
        }

        return result;
    }

    bool TFSVariable::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (!TBase::DeserializeFromJson(json)) {
            return false;
        }

        if (!json["control_field"][GetFieldName()].IsDefined() || !Condition.DeserializeFromJson(json["control_field"][GetFieldName()])) {
            return false;
        }

        if (json.Has("default_fields")) {
            DefaultValue = ISchemeElement::ConstructFromJson(json["default_fields"], "");
        }

        for (auto&& [name, serializedItemStructure] : json["variants_fields"].GetMap()) {
            auto item = ISchemeElement::ConstructFromJson(serializedItemStructure, "");
            if (!Condition.GetVariants().contains(name) || !item) {
                return false;
            }
            ValueMapping.emplace(name, std::move(item));
        }

        return true;
    }

    bool TFSVariable::DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const {
        if (!json.Has(GetFieldName()) || !json[GetFieldName()].IsString()) {
            return true;  // empty
        }

        TString itemName = json[GetFieldName()].GetString();
        if (!Condition.GetVariants().contains(itemName)) {
            error = "invalid conditional type " + itemName;
            return false;
        }

        bool schemeCheckResult = false;

        auto itemSchemePair = ValueMapping.find(itemName);
        if (itemSchemePair != ValueMapping.end()) {
            schemeCheckResult = (itemSchemePair->second)->ValidateJson(json, path + "/" + itemName);
        } else if (!!DefaultValue) {
            schemeCheckResult = DefaultValue->ValidateJson(json, path + "/default");
        } else {
            schemeCheckResult = true;
        }

        return schemeCheckResult;
    }

    NJson::TJsonValue TFSDuration::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);
        if (!!Min) {
            result.InsertValue("min", Min->Seconds());
        }
        result.InsertValue("visual", "duration");
        if (!!Max) {
            result.InsertValue("max", Max->Seconds());
        }
        return result;
    }

    bool TFSDuration::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_DURATION_OPT(json, "min", Min);
        JREAD_DURATION_OPT(json, "max", Max);
        return TBase::DeserializeFromJson(json);
    }

    NJson::TJsonValue TFSDuration::GetDefaultValueView() const {
        return HasDefault() ? TJsonProcessor::FormatDurationString(GetDefaultUnsafe()) : NJson::TJsonValue();
    }

    bool TFSDuration::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (!json.IsString()) {
            error = "is not a STRING";
            return false;
        }
        TDuration result;
        if (!TDuration::TryParse(json.GetString(), result)) {
            error = "is not a Duration";
            return false;
        }
        if (!!Min && result < Min) {
            error = ToString(result) + " < " + ToString(Min);
            return false;
        }
        if (!!Max && result > Max) {
            error = ToString(result) + " > " + ToString(Max);
            return false;
        }
        return true;
    }

    TString TFSDuration::AdditionalDescription() const {
        if (Min || Max) {
            return "(" + (!!Min ? " min: " + TJsonProcessor::FormatDurationString(*Min) : "") + (!!Max ? " max: " + TJsonProcessor::FormatDurationString(*Max) : "") + ")";
        }
        return TString();
    }

    NJson::TJsonValue TFSNumeric::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);
        if (!!Min) {
            result.InsertValue("min", *Min);
        }
        if (!!Max) {
            result.InsertValue("max", *Max);
        }
        if (Visual) {
            result.InsertValue("visual", ::ToString(Visual));
        }
        if (traits & EReportTraits::PrecisionField) {
            result.InsertValue("precision", Precision);
        }
        return result;
    }

    bool TFSNumeric::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_DURATION_OPT(json, "min", Min);
        JREAD_DURATION_OPT(json, "max", Max);
        JREAD_FROM_STRING_OPT(json, "visual", Visual);
        return TBase::DeserializeFromJson(json);
    }

    TString TFSNumeric::AdditionalDescription() const {
        if (Min || Max) {
            return "(" + (!!Min ? " min: " + ToString(*Min) : "") + (!!Max ? " max: " + ToString(*Max) : "") + ")";
        }
        return TString();
    }

    bool TFSNumeric::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (!json.IsDouble()) {
            error = "is not a Double";
            return false;
        }

        double result = json.GetDouble();
        if (!!Min && result < Min) {
            error = ToString(result) + " < " + ToString(Min);
            return false;
        }
        if (!!Max && result > Max) {
            error = ToString(result) + " > " + ToString(Max);
            return false;
        }
        if ((Visual == EVisualType::DateTime || Visual == EVisualType::Money) && result < 0) {
            error = ToString(result) + " < 0";
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFSStructure::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);
        CHECK_WITH_LOG(!!StructureElement);
        if (VisualType != EVisualType::Default) {
            TJsonProcessor::WriteAsString(result, "visual", VisualType);
        }
        result.InsertValue("structure", StructureElement->SerializeToJson(traits));
        return result;
    }

    bool TFSStructure::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (!json.Has("structure")) {
            return false;
        }
        if (!TJsonProcessor::ReadFromString("visual", json, VisualType, false)) {
            return false;
        }
        return TBase::DeserializeFromJson(json) && (StructureElement = ISchemeElement::ConstructFromJson(json["structure"], ""));
    }

    void TFSStructure::MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        PrintDescription(buf, os);
        PrintValue(json, buf, os);
    }

    bool TFSStructure::DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& /*error*/) const {
        CHECK_WITH_LOG(!!StructureElement);
        return StructureElement->ValidateJson(json, path);
    }

    void TFSStructure::AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        if (!StructureElement) {
            ythrow yexception() << "incorrect scheme description";
        }
        StructureElement->MakeJsonDescription(json, buf, os);
    }

    bool TFSBoolean::DoValidateJson(const NJson::TJsonValue& json, const TString& /*path*/, TString& error) const {
        if (!json.IsBoolean()) {
            error = "is not a Boolean";
        }
        return json.IsBoolean();
    }

    NJson::TJsonValue TFSArray::GetDefaultValueView() const {
        return (Defaults) ? NJson::ToJson(Defaults) : NJson::TJsonValue();
    }

    bool TFSArray::SetDefaultValueView(const NJson::TJsonValue& view) {
        return NJson::ParseField(view, Defaults);
    }

    NJson::TJsonValue TFSArray::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = TBase::SerializeToJson(traits);
        CHECK_WITH_LOG(!!ArrayTypeElement);
        NJson::InsertField(result, "array_type", ArrayTypeElement->SerializeToJson(traits));
        return result;
    }

    bool TFSArray::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (!json.Has("array_type")) {
            return false;
        }
        return TBase::DeserializeFromJson(json) &&
               (ArrayTypeElement = ISchemeElement::ConstructFromJson(json["array_type"], ""));
    }

    bool TFSArray::DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const {
        if (!json.IsArray()) {
            error = "is not an ARRAY";
            return false;
        }
        CHECK_WITH_LOG(!!ArrayTypeElement);
        const auto& jsonArray = json.GetArray();
        for (size_t i = 0; i < jsonArray.size(); ++i) {
            if (!ArrayTypeElement->ValidateJson(jsonArray[i], path + "/[" + ToString(i) + "]")) {
                return false;
            }
        }
        return true;
    }

    void TFSArray::MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        PrintDescription(buf, os);
        PrintValue(json, buf, os);
    }

    void TFSArray::AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        if (!ArrayTypeElement) {
            ythrow yexception() << "incorrect scheme: unknown element type";
        }
        buf.BeginList();
        if (!json.GetArray().empty()) {
            ArrayTypeElement->MakeJsonDescription(json.GetArray().front(), buf, os);
            if (json.GetArray().size() > 1) {
                buf.FlushTo(&os);
                os << "," << Endl;
                for (size_t i = 1; i < buf.State().Stack.size(); ++i) {
                    os << "  ";
                }
                os << "...";
            }
        }
        buf.EndList();
    }

    TSchemeTabGuard::TSchemeTabGuard(const TString& tabName, TScheme& ownedScheme)
        : OwnedScheme(ownedScheme) {
        OwnedScheme.SetCurrentTab(tabName);
    }

    TSchemeTabGuard::~TSchemeTabGuard() {
        OwnedScheme.SetCurrentTab(ISchemeElement::DefaultTabName);
    }

    bool TScheme::IsEmpty() const {
        return Elements.empty();
    }

    bool TScheme::HasField(const TString& fieldName) const {
        return ElementNames.contains(fieldName);
    }

    const TSet<TString>& TScheme::GetElementNames() const {
        return ElementNames;
    }

    const TVector<ISchemeElement::TPtr>& TScheme::GetElements() const {
        return Elements;
    }

    ISchemeElement::TPtr TScheme::Get(const TString& fName) const {
        for (auto&& element : Elements) {
            if (element && element->GetFieldName() == fName) {
                return element;
            }
        }
        return nullptr;
    }

    ISchemeElement::TPtr TScheme::AddFromJson(const TString& fieldName, const NJson::TJsonValue& json) {
        Y_ASSERT(!!fieldName);
        auto newElement = ISchemeElement::ConstructFromJson(json, fieldName);
        if (!newElement) {
            WARNING_LOG << "cannot deserialize field from json " << fieldName << Endl;
            return nullptr;
        }

        size_t inSchemeElementIndex = std::numeric_limits<size_t>::max();
        for (size_t i = 0; i < Elements.size(); ++i) {
            if (Elements[i] && Elements[i]->GetFieldName() == fieldName) {
                inSchemeElementIndex = i;
                break;
            }
        }

        //replace current with income
        if (inSchemeElementIndex != std::numeric_limits<size_t>::max()) {
            if (Elements[inSchemeElementIndex]->GetType() == newElement->GetType()) {
                Elements[inSchemeElementIndex]->DeserializeFromJson(json);
            } else {
                WARNING_LOG << "replace with different type " << fieldName << Endl;
                Elements[inSchemeElementIndex].Reset(newElement);
            }

            return Elements[inSchemeElementIndex];
        }

        newElement->SetFieldName(fieldName);
        Elements.emplace_back(newElement);
        ElementNames.emplace(fieldName);
        return newElement;
    }

    bool TScheme::Patch(const TString& fieldName, const ISchemeElement::TPtr& schemePatch) {
        if (auto inSchemeElement = Get(fieldName)) {
            if (schemePatch && schemePatch->GetType() == inSchemeElement->GetType()) {
                if (schemePatch->GetRequired()) {
                    inSchemeElement->SetRequired(true);
                }
                if (schemePatch->GetReadOnly()) {
                    inSchemeElement->SetReadOnly(true);
                }
                if (schemePatch->GetType() == EElementType::String) {
                    auto stringPatch = std::dynamic_pointer_cast<TFSString>(schemePatch);
                    auto stringToPatch = std::dynamic_pointer_cast<TFSString>(inSchemeElement);
                    if (!!stringPatch && !!stringToPatch && stringPatch->GetMaxLengthDef(0)) {
                        uint newLength = stringToPatch->GetMaxLengthDef(0);
                        if (newLength == 0 && stringPatch->GetMaxLengthDef(0) > 0) {
                            newLength = stringPatch->GetMaxLengthDef(0);
                        } else if (newLength != 0 && stringPatch->GetMaxLengthDef(0) > 0) {
                            newLength = std::min(newLength, stringPatch->GetMaxLengthDef(0));
                        }
                        stringToPatch->SetMaxLength(newLength);
                    }
                }

                return true;
            }
        }

        return false;
    }

    void TScheme::AddToSchemeFromJson(const NJson::TJsonValue& json, const TSet<TString>& skipFields) {
        auto schemeDescriptions = json.GetArray();
        if (schemeDescriptions.empty()) {
            return;
        }
        for (auto&& desc : schemeDescriptions) {
            if (desc.IsDefined() && desc.IsMap() && desc.GetMap().contains("name") && desc.GetMap().contains("description")) {
                auto name = desc.GetMap().at("name").GetString();
                if (skipFields.contains(name)) {
                    continue;
                }
                AddFromJson(name, desc.GetMap().at("description"));
            }
        }
    }

    void NDrive::TScheme::MergeScheme(const NDrive::TScheme& src) {
        for (auto&& name : src.GetElementNames()) {
            if (!HasField(name)) {
                Extend(src.Get(name));
                continue;
            }

            if (!Patch(name, src.Get(name))) {
                WARNING_LOG << "Cannot patch field " << name;
            }
        }
    }

    void TScheme::Reorder(const TVector<TString>& order) {
        TMap<TString, size_t> orderMap;
        for (size_t i = 0; i < order.size(); ++i) {
            orderMap[order[i]] = i;
        }
        size_t i = 0;
        while (i < Elements.size()) {
            //Elements[orderMap.at(name)]->SetOrderIdx(i + 1);
            auto name = Elements[i]->GetFieldName();
            if (orderMap.find(name) != orderMap.end() && orderMap.at(name) < Elements.size()) {
                if (size_t index = orderMap.at(name); index != i) {
                    std::swap(Elements[i], Elements[index]);
                    continue;
                }
            }
            i++;
        }
    }

    bool TScheme::Extend(const ISchemeElement::TPtr& element) {
        if (element && ElementNames.insert(element->GetFieldName()).second) {
            Elements.push_back(element);
            return true;
        }
        return false;
    }

    bool TScheme::AddPattern(ISchemeElement::TPtr element) {
        if (!element) {
            ERROR_LOG << "element undefined" << Endl;
            return false;
        }
        TString name = element->GetFieldName();
        TString regExprStr = name;
        if (!regExprStr.StartsWith('^')) {
            regExprStr = "^" + regExprStr;
        }
        if (!regExprStr.EndsWith('$')) {
            regExprStr += "$";
        }
        TRegExMatch regEx(regExprStr);
        if (name.empty() || !regEx.IsCompiled() || !ElementNames.emplace(name).second) {
            ERROR_LOG << "incorrect name " << name << Endl;
            return false;
        }
        PatternElements.emplace_back(regEx, element);
        element->SetTabName(CurrentTab);
        return true;
    }

    TScheme& TScheme::Extend(const TScheme& other) {
        CHECK_WITH_LOG(IsIntersectionEmpty(ElementNames, other.ElementNames));
        for (auto&& itemName: other.ElementNames) {
            ElementNames.insert(itemName);
        }
        for (auto&& itemPtr: other.Elements) {
            Elements.push_back(itemPtr);
        }
        for (auto&& patternItemPair: other.PatternElements) {
            PatternElements.push_back(patternItemPair);
        }
        return *this;
    }

    TScheme& TScheme::ExtendSafe(const TScheme& other) {
        for (auto&& itemPtr: other.Elements) {
            if (itemPtr && ElementNames.insert(itemPtr->GetFieldName()).second) {
                Elements.push_back(itemPtr);
            }
        }
        for (auto&& patternItemPair: other.PatternElements) {
            PatternElements.push_back(patternItemPair);
        }
        return *this;
    }

    TScheme& TScheme::Remove(const TString& fName) {
        if (ElementNames.erase(fName)) {
            for (auto it = Elements.begin(); it != Elements.end();) {
                if ((*it)->GetFieldName() == fName) {
                    it = Elements.erase(it);
                } else {
                    ++it;
                }
            }
        }
        return *this;
    }

    NJson::TJsonValue TScheme::SerializeToJson(TReportTraits traits) const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        SerializeToJson(result, traits);
        return result;
    }

    void TScheme::SerializeToJson(NJson::TJsonValue& result, TReportTraits traits) const {
        ui32 idx = 1;
        for (auto&& i : Elements) {
            CHECK_WITH_LOG(!!i->GetFieldName());
            NJson::TJsonValue item = i->SerializeToJson(traits);
            if (traits & EReportTraits::OrderField) {
                if (i->HasOrderIdx()) {
                    item.InsertValue("order", ++idx + i->GetOrderIdxUnsafe() * 1000);
                } else {
                    item.InsertValue("order", ++idx);
                }
            }
            JWRITE(result, i->GetFieldName(), std::move(item));
        }
        for (auto&& i : PatternElements) {
            CHECK_WITH_LOG(!!i.second->GetFieldName());
            NJson::TJsonValue item = i.second->SerializeToJson(traits);
            JWRITE(result, i.second->GetFieldName(), std::move(item));
        }
        if (AdditionalProperties && (traits & EReportTraits::AdditionalProperties)) {
            JWRITE(result, "__additional_properties", AdditionalProperties);
        }
    }

    bool TScheme::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (!json.IsMap()) {
            ERROR_LOG << json.GetStringRobust() << " not a Map" << Endl;
            return false;
        }
        for (auto&&[key, valueJson] : json.GetMap()) {
            if (key == "__additional_properties") {
                JREAD_BOOL(json, "__additional_properties", AdditionalProperties);
                continue;
            }
            if (!AddPattern(ISchemeElement::ConstructFromJson(valueJson, key))) {
                return false;
            }
        }
        return true;
    }

    TString TScheme::MakeJsonDescription(const NJson::TJsonValue& json) const {
        TStringStream ss;
        NJsonWriter::TBuf buf;
        buf.SetIndentSpaces(2);
        MakeJsonDescription(json, buf, ss);
        buf.FlushTo(&ss);
        return ss.Str();
    }

    void TScheme::MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        PrintValue(json, buf, os);
    }

    void TScheme::AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const {
        buf.BeginObject();
        for (auto&& i : Elements) {
            TString key = i->GetFieldName();
            if (!key) {
                ythrow yexception() << "incorrect scheme: empty field name";
            }
            NJson::TJsonValue jsonElem;
            if (json.GetValue(key, &jsonElem)) {
                buf.WriteKey(key);
                i->MakeJsonDescription(jsonElem, buf, os);
            }
        }
        TVector<TString> keys = MakeVector(NContainer::Keys(json.GetMap()));
        for (auto&& i : PatternElements) {
            for (const auto& key : keys) {
                if (i.first.Match(key.data())) {
                    buf.WriteKey(key);
                    i.second->MakeJsonDescription(json[key], buf, os);
                }
            }
        }
        buf.EndObject();
    }

    bool TScheme::DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const {
        if (!json.IsMap()) {
            error = "is not a Map";
            return false;
        }
        TSet<TString> keys = MakeSet(NContainer::Keys(json.GetMap()));
        for (auto&& i : Elements) {
            CHECK_WITH_LOG(!!i->GetFieldName());
            keys.erase(i->GetFieldName());
            NJson::TJsonValue jsonElem;
            if (!json.GetValue(i->GetFieldName(), &jsonElem) || json[i->GetFieldName()].IsNull()) {
                if (i->GetRequired()) {
                    error = "required parameter " + i->GetFieldName() + " not found";
                    return false;
                } else {
                    continue;
                }
            }
            if (!i->ValidateJson(jsonElem, path + "/" + i->GetFieldName())) {
                return false;
            }
        }
        TSet<TString> matchedKeys;
        for (auto&& i : PatternElements) {
            bool found = false;
            for (const auto& key : keys) {
                if (i.first.Match(key.data())) {
                    if (!matchedKeys.emplace(key).second) {
                        error = "find two pattern for key: " + key;
                        return false;
                    }
                    if (json[key].IsNull()) {
                        if (i.second->GetRequired()) {
                            error = "required parameter " + i.second->GetFieldName() + " is null";
                            return false;
                        }
                        continue;
                    }
                    found = true;
                    if (!i.second->ValidateJson(json[key], path + "/" + key)) {
                        return false;
                    }
                }
            }
            if (!found && i.second->GetRequired()) {
                error = "required parameter " + i.second->GetFieldName() + " not found";
                return false;
            }
        }
        for (const auto& key : matchedKeys) {
            keys.erase(key);
        }
        if (keys && !AdditionalProperties) {
            error = "find unknown keys: " + JoinSeq(",", keys);
            return false;
        }
        return true;
    }
}
