#pragma once

#include <library/cpp/json/json_value.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/regex/pcre/regexp.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/cast.h>
#include <util/generic/map.h>
#include <util/generic/serialized_enum.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>

namespace NDrive {
    enum class EElementType {
        Ignore /* "ignore" */,
        Separator /* "separator" */,

        Boolean /* "bool" */,
        Numeric /* "numeric" */,
        Duration /* "duration" */,

        String /* "string" */,
        Text /* "text" */,
        Json /* "json" */,

        Array /* "array_types" */,
        Structure /* "structure" */ ,

        Variants /* "variants" */,
        Variable /* "variable" */,
    };

    class ISchemeElement {
    public:
        class IHidingRule {
        public:
            using TPtr = TAtomicSharedPtr<IHidingRule>;

        public:
            virtual ~IHidingRule() = default;
            virtual NJson::TJsonValue SerializeToJson() const = 0;
        };

    public:
        using TReportTraits = ui32;

        enum EReportTraits : TReportTraits {
            OrderField = 1 << 0,
            TabNameField = 1 << 1,
            ReadOnlyField = 1 << 2,
            TypeField = 1 << 3,
            InternalTypeField = 1 << 4,
            RequiredField = 1 << 5,
            DescriptionField = 1 << 6,
            PrecisionField = 1 << 7,
            AdditionalProperties = 1 << 8,
            TooltipField = 1 << 9,
            HidingField = 1 << 10,
        };

        static constexpr TReportTraits ReportAll = Max<TReportTraits>();
        static constexpr TReportTraits ReportFrontend = (EReportTraits::OrderField |
                                                         EReportTraits::TabNameField |
                                                         EReportTraits::ReadOnlyField |
                                                         EReportTraits::TypeField |
                                                         EReportTraits::RequiredField |
                                                         EReportTraits::DescriptionField |
                                                         EReportTraits::TooltipField |
                                                         EReportTraits::PrecisionField);
        static constexpr TReportTraits ReportValidation = (EReportTraits::TypeField |
                                                           EReportTraits::InternalTypeField |
                                                           EReportTraits::RequiredField |
                                                           EReportTraits::DescriptionField |
                                                           EReportTraits::AdditionalProperties);

        static const TString DefaultTabName;

    public:
        using TPtr = TAtomicSharedPtr<ISchemeElement>;
        using TFactory = NObjectFactory::TParametrizedObjectFactory<ISchemeElement, EElementType, TString>;

        ISchemeElement(const TString& fieldName = "", const TString& description = "", const ui32 idx = Max<ui32>());
        virtual ~ISchemeElement() = default;

        auto& SetDeprecated() {
            Deprecated = true;
            return *this;
        }

        virtual EElementType GetType() const = 0;

        virtual EElementType GetRegisterType() const {
            return GetType();
        }

        virtual NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const;

        static ISchemeElement::TPtr ConstructFromJson(const NJson::TJsonValue& json, const TString& fieldName);
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json);

        bool ValidateJson(const NJson::TJsonValue& json, const TString& path = "") const;

        virtual void MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const = 0;

        void PrintDescription(NJsonWriter::TBuf& buf, IOutputStream& os) const;
        void PrintValue(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const;

        virtual void AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const;
        virtual TString AdditionalDescription() const;

        void AddHidingRule(IHidingRule::TPtr rule);

    private:
        R_FIELD(TString, FieldName);
        R_FIELD(TString, Description);
        R_FIELD(TString, Tooltip);

        R_OPTIONAL(ui32, OrderIdx);
        R_FIELD(TString, TabName, DefaultTabName);

        R_FIELD(bool, Required, false);
        R_FIELD(bool, ReadOnly, false);
        R_OPTIONAL(bool, Deprecated);

        TVector<IHidingRule::TPtr> HidingRules;
    };

    class IBaseDefaultSchemeElement: public ISchemeElement {
        using TBase = ISchemeElement;

    public:
        using TBase::TBase;

        virtual NJson::TJsonValue GetDefaultValueView() const = 0;
        virtual bool SetDefaultValueView(const NJson::TJsonValue& view) = 0;

        virtual NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
    };

    template <typename T>
    class IDefaultSchemeElement: public IBaseDefaultSchemeElement {
        using TBase = IBaseDefaultSchemeElement;

    public:
        class THidingFieldRule : public IHidingRule {
            T Value;
            TString Field;

        public:
            THidingFieldRule(const T& value, const TString& field)
                : Value(value)
                , Field(field)
            {
            }

            NJson::TJsonValue SerializeToJson() const override {
                NJson::TJsonValue result;
                NJson::InsertField(result, "value", Value);
                NJson::InsertField(result, "field", Field);
                return result;
            }
        };

    public:
        using TBase::TBase;

        virtual NJson::TJsonValue GetDefaultValueView() const override {
            return NJson::ToJson(Default);
        }

        virtual bool SetDefaultValueView(const NJson::TJsonValue& view) override {
            return NJson::ParseField(view, Default);
        }

        IDefaultSchemeElement<T>& AddHidingRule(const T& value, const TString& field) {
            TBase::AddHidingRule(MakeAtomicShared<THidingFieldRule>(value, field));
            return *this;
        }

    private:
        R_OPTIONAL(T, Default);
    };

    class TFSIgnore: public ISchemeElement {
        using TBase = ISchemeElement;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Ignore;
        }

        virtual void MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSIgnore> Registrator;
    };

    class TFSSeparator: public ISchemeElement {
        using TBase = ISchemeElement;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Separator;
        }

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSSeparator> Registrator;
    };

    class TFSString: public IDefaultSchemeElement<TString> {
        using TBase = IDefaultSchemeElement<TString>;

    public:
        enum class EVisualType {
            Color /* "color" */,
            ObjectId /* "id_select" */,
            UUID /* "UUID" */,
            GUID /* "GUID" */,
            File /* "File" */,
            StartrekAttachment /* "File_ST" */,
            CurrentSession /* "current_session_id" */,
        };

        class TValidator {
        public:
            bool DeserializeFromJson(const NJson::TJsonValue& json);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_OPTIONAL(TString, Regex);
            R_OPTIONAL(TString, ErrorString);
        };

        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::String;
        }

        virtual NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSString> Registrator;

        R_OPTIONAL(ui32, MaxLength);
        R_OPTIONAL(EVisualType, Visual);
        R_OPTIONAL(TValidator, Validator);
    };

    class TFSText: public IDefaultSchemeElement<TString> {
        using TBase = IDefaultSchemeElement<TString>;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Text;
        }

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSText> Registrator;
    };

    class TFSJson: public IDefaultSchemeElement<TString> {
        using TBase = IDefaultSchemeElement<TString>;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Json;
        }

        virtual NJson::TJsonValue GetDefaultValueView() const override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSJson> Registrator;
    };

    class TFSBoolean: public IDefaultSchemeElement<bool> {
        using TBase = IDefaultSchemeElement<bool>;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Boolean;
        }

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSBoolean> Registrator;
    };

    class TFSArray: public IBaseDefaultSchemeElement {
        using TBase = IBaseDefaultSchemeElement;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Array;
        }

        template <class T>
        T& SetElement() {
            ArrayTypeElement = MakeAtomicShared<T>();
            return *VerifyDynamicCast<T*>(ArrayTypeElement.Get());
        }

        template <class T>
        T& SetElement(const T& element) {
            ArrayTypeElement = MakeAtomicShared<T>(element);
            return *VerifyDynamicCast<T*>(ArrayTypeElement.Get());
        }

        template <class T>
        auto& AddDefault(const T& defaultValue) {
            Defaults.emplace_back(NJson::ToJson(defaultValue));
            return *this;
        }

        template <class T>
        auto& AddDefaults(const TVector<T>& defaultValues) {
            for (const T& defaultValue: defaultValues) {
                AddDefault(defaultValue);
            }
            return *this;
        }

        template <class T>
        auto& SetDefaults(const TVector<T>& defaultValues) {
            Defaults.clear();
            return AddDefaults(defaultValues);
        }

        virtual NJson::TJsonValue GetDefaultValueView() const override;
        virtual bool SetDefaultValueView(const NJson::TJsonValue& view) override;

        NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

        virtual void MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

    private:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;
        virtual void AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSArray> Registrator;

        R_FIELD(ISchemeElement::TPtr, ArrayTypeElement);

        R_READONLY(TVector<NJson::TJsonValue>, Defaults);
    };

    class TFSStructure: public ISchemeElement {
        using TBase = ISchemeElement;

    public:
        enum class EVisualType {
            Default,
            Table
        };

        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Structure;
        }

        template <class T>
        T& SetStructure() {
            StructureElement = MakeAtomicShared<T>();
            return *VerifyDynamicCast<T*>(StructureElement.Get());
        }

        template <class T>
        T& SetStructure(const T& value) {
            StructureElement = MakeAtomicShared<T>(value);
            return *VerifyDynamicCast<T*>(StructureElement.Get());
        }

        NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

        virtual void MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

    private:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& /*error*/) const override;
        virtual void AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSStructure> Registrator;

        R_FIELD(EVisualType, VisualType, EVisualType::Default);
        R_FIELD(ISchemeElement::TPtr, StructureElement);
    };

    class TFSVariants: public IBaseDefaultSchemeElement {
        using TBase = IBaseDefaultSchemeElement;

    public:
        class TCompoundVariant {
        public:
            explicit TCompoundVariant(const TString& value = {});
            TCompoundVariant(const TString& value, const TString& text, const TString& description = {});

            bool operator < (const TCompoundVariant& other) const;

            NJson::TJsonValue SerializeToJson() const;
            bool DeserializeFromJson(const NJson::TJsonValue& data);

        private:
            R_FIELD(TString, Value);
            R_FIELD(TString, Text);
            R_FIELD(TString, Description);
        };

        using TCompoundVariants = TSet<TCompoundVariant>;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Variants;
        }

        auto& Clear() {
            Variants.clear();
            CompoundVariants.clear();
            return *this;
        }

        template <class T>
        auto& AddVariant(const T& value) {
            return AddVariant(::ToString(value));
        }

        template <>
        auto& AddVariant(const TString& value) {
            Variants.emplace(value);
            if (CompoundVariants) {
                CompoundVariants.emplace(TCompoundVariant(value));
            }
            return *this;
        }

        template <class T>
        auto& AddVariants(std::initializer_list<T> container) {
            for (auto&& i : container) {
                AddVariant(i);
            }
            return *this;
        }

        template <class TContainer>
        auto& AddVariants(const TContainer& container) {
            for (auto&& i: container) {
                AddVariant(i);
            }
            return *this;
        }

        template <class T, class TKey = TString>
        auto& InitVariantsClass() {
            TSet<TKey> keys;
            T::TFactory::GetRegisteredKeys(keys);
            return SetVariants(keys);
        }

        template <class TEnum>
        auto& InitVariants() {
            Clear();
            for (auto&& [name, value]: GetEnumNames<TEnum>()) {
                AddVariant(value);
            }
            return *this;
        }

        template <class TContainer>
        auto& InitVariantsFromKeys(const TContainer& container) {
            Clear();
            for (auto&& i: container) {
                AddVariant(i.first);
            }
            return *this;
        }

        template <class T>
        auto& SetVariants(std::initializer_list<T> container) {
            Clear();
            return AddVariants(std::move(container));
        }

        template <class T>
        auto& SetVariants(const TMap<TString, T>& container) {
            Clear();
            return AddVariants(NContainer::Keys(container));
        }

        template <class TContainer>
        auto& SetVariants(const TContainer& container) {
            Clear();
            return AddVariants(container);
        }

        auto& AddCompoundVariant(const TCompoundVariant& value) {
            if (!CompoundVariants && Variants) {
                for (auto&& variant: Variants) {
                    CompoundVariants.emplace(TCompoundVariant(variant));
                }
            }
            Variants.emplace(value.GetValue());
            CompoundVariants.emplace(value);
            return *this;
        }

        auto& AddCompoundVariants(std::initializer_list<TCompoundVariant> container) {
            for (auto&& i : container) {
                AddCompoundVariant(i);
            }
            return *this;
        }

        template <class TContainer>
        auto& AddCompoundVariants(const TContainer& container) {
            for (auto&& i: container) {
                AddCompoundVariant(i);
            }
            return *this;
        }

        auto& SetCompoundVariants(std::initializer_list<TCompoundVariant> container) {
            Clear();
            return AddCompoundVariants(std::move(container));
        }

        template <class TContainer>
        auto& SetCompoundVariants(const TContainer& container) {
            Clear();
            return AddCompoundVariants(container);
        }

        template <class T>
        auto& MulVariantsLeft(std::initializer_list<T> container) {
            TSet<TString> variantsOld(std::move(Variants));
            Clear();

            for (auto&& v : variantsOld) {
                for (auto&& i : container) {
                    AddVariant(::ToString(i) + ::ToString(v));
                }
            }

            return *this;
        }

        virtual NJson::TJsonValue GetDefaultValueView() const override;
        virtual bool SetDefaultValueView(const NJson::TJsonValue& view) override;

        TFSVariants& SetDefault(const TString& value);

        NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;
        virtual TString AdditionalDescription() const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSVariants> Registrator;

        // variants are supposed to be either plain or compound
        // all compound variant names (Value fields) are stored as Variants as well
        // all plain variants are transformed to compound once compound is going to be added
        R_READONLY(TSet<TString>, Variants);
        R_READONLY(TCompoundVariants, CompoundVariants);

        R_FIELD(TVector<TString>, Defaults);  // last one used if not multi selectable

        R_FIELD(TString, Reference);
        R_FIELD(bool, Editable, false);
        R_FIELD(bool, MultiSelect, false);
        R_FIELD(bool, RefreshOnUpdate, false);
    };

    class TFSVariable: public ISchemeElement {
        using TBase = ISchemeElement;
        using TValueMapping = TMap<TString, ISchemeElement::TPtr>;

    public:
        using TCompoundVariant = TFSVariants::TCompoundVariant;

        TFSVariable(const TString& fieldName = "", const TString& description = "", const ui32 idx = Max<ui32>());

        virtual EElementType GetType() const override {
            return EElementType::Variable;
        }

        auto& AddVariant(const TString& variant) {
            Condition.AddVariant(variant);
            return *this;
        }

        auto& AddVariant(const TCompoundVariant& variant) {
            Condition.AddCompoundVariant(variant);
            return *this;
        }

        template <typename T, typename = std::enable_if_t<std::is_base_of<ISchemeElement, typename std::decay<T>::type>::value>>
        auto& AddVariant(const TString& variant, const T& value) {
            Condition.AddVariant(variant);
            ValueMapping.emplace(variant, MakeAtomicShared<T>(value));
            return *this;
        }

        template <typename T, typename = std::enable_if_t<std::is_base_of<ISchemeElement, typename std::decay<T>::type>::value>>
        auto& AddVariant(const TCompoundVariant& variant, const T& value) {
            Condition.AddCompoundVariant(variant);
            ValueMapping.emplace(variant.GetValue(), MakeAtomicShared<T>(value));
            return *this;
        }

        template <typename TMapping>
        auto& AddVariants(const TMapping& mapping) {
            for (auto&& [variant, value]: mapping) {
                AddVariant(variant, value);
            }
            return *this;
        }

        template <class T, typename = std::enable_if_t<std::is_base_of<ISchemeElement, typename std::decay<T>::type>::value>>
        auto& SetDefaultValue(const T& value) {
            DefaultValue = MakeAtomicShared<T>(value);
            return *this;
        }

        NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSVariable> Registrator;

        R_FIELD(TFSVariants, Condition);
        R_READONLY(ISchemeElement::TPtr, DefaultValue, nullptr);
        R_READONLY(TValueMapping, ValueMapping);
    };

    class TFSDuration: public IDefaultSchemeElement<TDuration> {
        using TBase = IDefaultSchemeElement<TDuration>;

    public:
        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::String;
        }

        EElementType GetRegisterType() const override {
            return EElementType::Duration;
        }

        virtual NJson::TJsonValue GetDefaultValueView() const override;

        NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;
        virtual TString AdditionalDescription() const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSDuration> Registrator;

        R_OPTIONAL(TDuration, Max);
        R_OPTIONAL(TDuration, Min);
    };

    class TFSNumeric: public IDefaultSchemeElement<double> {
        using TBase = IDefaultSchemeElement<double>;

    public:
        enum EVisualType {
            Raw /* "raw" */,
            Money /* "money" */,
            DateTime /* "timestamp" */
        };

        using TBase::TBase;

        virtual EElementType GetType() const override {
            return EElementType::Numeric;
        }

        NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

    protected:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;
        virtual TString AdditionalDescription() const override;

    private:
        static ISchemeElement::TFactory::TRegistrator<TFSNumeric> Registrator;

        R_OPTIONAL(double, Min);
        R_OPTIONAL(double, Max);
        R_FIELD(ui32, Precision, 0);
        R_FIELD(EVisualType, Visual, EVisualType::Raw);
    };

    class TScheme;

    class TSchemeTabGuard {
    public:
        using TPtr = TAtomicSharedPtr<TSchemeTabGuard>;

        TSchemeTabGuard(const TString& tabName, TScheme& ownedScheme);
        ~TSchemeTabGuard();

    private:
        TScheme& OwnedScheme;
    };

    class TScheme: public ISchemeElement {
        using TBase = ISchemeElement;

    protected:
        template <class T>
        struct TClassDetector {
            using SchemeClass = TNull;
        };

        template <>
        struct TClassDetector<ui32> {
            using SchemeClass = TFSNumeric;
        };

        template <>
        struct TClassDetector<double> {
            using SchemeClass = TFSNumeric;
        };

        template <>
        struct TClassDetector<TString> {
            using SchemeClass = TFSString;
        };

        template <>
        struct TClassDetector<TDuration> {
            using SchemeClass = TFSDuration;
        };

    public:
        using TSchemeTabGuard = TSchemeTabGuard;

        virtual EElementType GetType() const override {
            return EElementType::Structure;
        }

        bool IsEmpty() const;
        bool HasField(const TString& fieldName) const;
        const TSet<TString>& GetElementNames() const;
        const TVector<ISchemeElement::TPtr>& GetElements() const;

        ISchemeElement::TPtr Get(const TString& fName) const;
        ISchemeElement::TPtr AddFromJson(const TString& fieldName, const NJson::TJsonValue& json);
        bool Patch(const TString& fieldName, const ISchemeElement::TPtr& schemePatch);

        template <class T>
        typename TClassDetector<T>::SchemeClass& AddDetect(const TString& fieldName, const TString& description = "", const ui32 orderIdx = Max<ui32>()) {
            return Add<typename TClassDetector<T>::SchemeClass>(fieldName, description, orderIdx);
        }

        void AddToSchemeFromJson(const NJson::TJsonValue& json, const TSet<TString>& skipFields);
        void MergeScheme(const NDrive::TScheme& src);
        void Reorder(const TVector<TString>& order);
        bool Extend(const ISchemeElement::TPtr& element);

        template <class T>
        T& Add(const TString& fieldName, const TString& description = "", const ui32 orderIdx = Max<ui32>()) {
            Y_ASSERT(!!fieldName);
            Elements.emplace_back(MakeAtomicShared<T>(fieldName, !!description ? description : fieldName, orderIdx));
            bool inserted = ElementNames.emplace(fieldName).second;
            if (!inserted) {
                ALERT_LOG << "cannot insert field " << fieldName << " into scheme" << Endl;
            }
            Y_ASSERT(inserted);
            T* result = VerifyDynamicCast<T*>(Elements.back().Get());
            result->SetTabName(CurrentTab);
            return *result;
        }

        template <class T>
        T& AddPattern(const TRegExMatch& regEx, const TString& description = "", const ui32 orderIdx = Max<ui32>()) {
            Y_ASSERT(regEx.IsCompiled());
            TString fieldName = regEx.GetRegExpr();
            PatternElements.emplace_back(regEx, MakeAtomicShared<T>(fieldName, !!description ? description : fieldName, orderIdx));
            bool inserted = ElementNames.emplace(fieldName).second;
            if (!inserted) {
                ALERT_LOG << "cannot insert field " << fieldName << " into scheme" << Endl;
            }
            Y_ASSERT(inserted);
            T* result = VerifyDynamicCast<T*>(PatternElements.back().second.Get());
            result->SetTabName(CurrentTab);
            return *result;
        }

        TScheme& Extend(const TScheme& other);
        TScheme& ExtendSafe(const TScheme& other);

        TScheme& Remove(const TString& fName);

        TSchemeTabGuard::TPtr StartTabGuard(const TString& tabName) {
            return MakeAtomicShared<TSchemeTabGuard>(tabName, *this);
        }

        virtual NJson::TJsonValue SerializeToJson(TReportTraits traits = ReportAll) const override;
        void SerializeToJson(NJson::TJsonValue& result, TReportTraits traits = ReportAll) const;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;

        TString MakeJsonDescription(const NJson::TJsonValue& json) const;
        virtual void MakeJsonDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

    private:
        virtual bool DoValidateJson(const NJson::TJsonValue& json, const TString& path, TString& error) const override;
        virtual void AddValueToDescription(const NJson::TJsonValue& json, NJsonWriter::TBuf& buf, IOutputStream& os) const override;

        bool AddPattern(ISchemeElement::TPtr element);

    private:
        TSet<TString> ElementNames;
        TVector<ISchemeElement::TPtr> Elements;
        TVector<std::pair<TRegExMatch, ISchemeElement::TPtr>> PatternElements;

        R_FIELD(TString, CurrentTab, DefaultTabName);
        R_FIELD(bool, AdditionalProperties, false);
    };
}

using TFSIgnore = NDrive::TFSIgnore;
using TFSSeparator = NDrive::TFSSeparator;
using TFSString = NDrive::TFSString;
using TFSText = NDrive::TFSText;
using TFSJson = NDrive::TFSJson;
using TFSBoolean = NDrive::TFSBoolean;
using TFSArray = NDrive::TFSArray;
using TFSStructure = NDrive::TFSStructure;
using TFSVariants = NDrive::TFSVariants;
using TFSVariable = NDrive::TFSVariable;
using TFSNumeric = NDrive::TFSNumeric;
using TFSDuration = NDrive::TFSDuration;
