#pragma once

#include "replier.h"
#include "exception.h"

#include <rtline/util/types/exception.h>

#include <kernel/daemon/base_http_client.h>

#include <library/cpp/string_utils/quote/quote.h>
#include <util/string/subst.h>

template <class TRequestData>
class IHttpClient: public TCommonHttpClient<TRequestData> {
private:
    TInstant RequestStartTime = Now();
public:
    TInstant GetRequestStartTime() const {
        return RequestStartTime;
    }
    virtual const TBlob& GetBuf() const = 0;
};

template <class TContext>
class THttpClientImpl: public IHttpClient<typename TContext::TRequestData> {
private:
    using TBase = IHttpClient<typename TContext::TRequestData>;
    using TBase::ReleaseConnection;
    using TBase::ProcessHeaders;
    using TBase::RD;
    using TBase::Buf;

protected:
    using TBase::HttpServ;
    using TBase::ProcessSpecialRequest;
    using TBase::Output;

public:
    virtual ~THttpClientImpl() {
        try {
            ReleaseConnection();
        } catch (const std::exception& e) {
            DEBUG_LOG << "Connection releasing failed: " << FormatExc(e) << Endl;
            OnReleaseConnectionFailure();
        }
    }

    virtual bool Reply(void* /*ThreadSpecificResource*/) override {
        Output().Flush();
        if (!ProcessHeaders()) {
            return true;
        }

        const auto& serverOptions = HttpServ()->Options();
        RD.Scan();
        RD.SetHost(serverOptions.Host, serverOptions.Port);

        IReplyContext::TPtr context(new TContext(this));
        try {
            OnBeginProcess(context);

            if (ProcessSpecialRequest() || !ProcessAuth()) {
                return false;
            }

            IHttpReplier::TPtr searchReplier = DoSelectHandler(context);
            Y_ENSURE(searchReplier);
            searchReplier.Release()->Reply();
        } catch (const TCodedException& e) {
            MakeErrorPage(context, e.GetCode(), e.what());
        } catch (const std::exception& e) {
            MakeErrorPage(context, HTTP_INTERNAL_SERVER_ERROR, FormatExc(e));
        }
        return false;
    }

    virtual void OnReleaseConnectionFailure() {
    }

    virtual void OnBeginProcess(IReplyContext::TPtr /*context*/) {
    }

    virtual bool ProcessAuth() {
        return true;
    }

    const TBlob& GetBuf() const override {
        return Buf;
    }

    TServerRequestData& MutableRequestData() {
        return RD;
    }

protected:
    virtual void MakeErrorPage(IReplyContext::TPtr context, ui32 code, const TString& error) const {
        ::MakeErrorPage(context, code, error);
    }

private:
    virtual IHttpReplier::TPtr DoSelectHandler(IReplyContext::TPtr context) = 0;
};

template <class TRequestDataParam>
class TCommonHttpReplyContext: public IReplyContext {
public:
    using TRequestData = TRequestDataParam;
    using IHttpClient = ::IHttpClient<TRequestData>;

protected:
    THolder<IHttpClient> Client;

protected:
    virtual const TBlob& DoGetBuf() const override {
        return Client->GetBuf();
    }

private:
    TMap<TString, TString> HttpHeaders;
    bool ConnectionReleased = false;

public:
    TCommonHttpReplyContext(IHttpClient* client)
        : Client(client)
    {
        HttpHeaders.insert(std::make_pair("Content-Type", "text/plain"));
    }

    IOutputStream& Output() override {
        Y_ENSURE_BT(!ConnectionReleased);
        return Client->Output();
    }

    virtual bool IsHttp() const override {
        return true;
    }

    virtual bool IsLocal() const override {
        return Client->IsLocal();
    }

    virtual TInstant GetRequestStartTime() const override {
        return Client->GetRequestStartTime();
    }

    virtual void MakeSimpleReply(const TBuffer& buf, int code = HTTP_OK) override {
        MakeSimpleReplyImpl(buf, code);
    }

    template <class T>
    void MakeSimpleReplyImpl(const T& buf, int code = HTTP_OK) noexcept {
        MakeSimpleReplyImpl<T>(Output(), buf, code, HttpHeaders);
        ReleaseConnection();
    }

    void ReleaseConnection() noexcept try {
        ConnectionReleased = true;
        Client->ReleaseConnection();
    } catch (const std::exception& e) {
        ERROR_LOG << "Exception during ReleaseConnection: " << FormatExc(e) << Endl;
    }

    template <class T>
    static void MakeSimpleReplyImpl(IOutputStream& output, const T& buf, int code = HTTP_OK, const TMap<TString, TString>& headers = TMap<TString, TString>()) noexcept try {
        TStringBuf CrLf = "\r\n";
        output.Write(TStringBuf("HTTP/1.1 "));
        output.Write(HttpCodeStrEx(code));
        output.Write(CrLf);
        for (auto&& header : headers) {
            output.Write(header.first);
            output.Write(TStringBuf(": "));
            output.Write(header.second);
            output.Write(CrLf);
        }

        output.Write(TStringBuf("Content-Length: "));
        output.Write(ToString(buf.size()));
        output.Write(CrLf);
        output.Write(CrLf);
        output.Write(TStringBuf(buf.data(), buf.size()));
        output.Flush();
    } catch (const std::exception& e) {
        ERROR_LOG << "Exception during MakeSimpleReply: " << FormatExc(e) << Endl;
    }

    virtual void AddReplyInfo(const TString& key, const TString& value) override {
        if (value.find('\r') != TString::npos || value.find('\n') != TString::npos) {
            ERROR_LOG << "AddReplyInfo: value contains CR or LF, removing them. key: " << key << ", escaped value: " << CGIEscapeRet(value) << ", req_id: " << GetRequestId() << Endl;
            auto valueWithoutCrLf = value;
            SubstGlobal(valueWithoutCrLf, "\r", "");
            SubstGlobal(valueWithoutCrLf, "\n", "");
            HttpHeaders[key] = valueWithoutCrLf;
        } else {
            HttpHeaders[key] = value;
        }
    }
};
